# coding: utf-8
from __future__ import unicode_literals

from django.conf import settings
from rest_framework import status
from rest_framework.decorators import api_view
from rest_framework.response import Response

from common.models.transport import TransportType
from travel.rasp.morda_backend.morda_backend.tariffs.daemon.serialization import (
    DaemonQuerySchema, ResponseSchema, MinDynamicTariffsResponseSchema, MinDynamicTariffsQuerySchema,
    TariffsInitResponseSchema, TariffsPollQuerySchema
)
from travel.rasp.morda_backend.morda_backend.tariffs.daemon.service import (
    collect_daemon_segments, collect_segments, get_dynamic_min_tariffs,
    init_plane_tariff_queries, create_poll_tariff_query
)


@api_view(['GET'])
def tariffs(request, poll=False):
    """
    Поиск тарифов только воздушного транспорта через тикет-демон avia.
    Для остального транспорта возвращает пустой ответ.
    Пример запроса Питер - Москва:
    /ru/segments/tariffs/?pointFrom=c2&pointTo=c213&date=2016-05-10&national_version=ru
    Результат можно получить по адресу:
    /ru/segments/tariffs/poll/?pointFrom=c2&pointTo=c213&date=2016-05-10&national_version=ru
    Про настройки см. README.md
    """
    daemon_query, errors = DaemonQuerySchema().load(request.GET)
    if errors:
        return Response({'errors': errors})

    plane_types = [TransportType.get_plane_type()]
    if daemon_query.transport_types is None:
        daemon_query = daemon_query._replace(transport_types=plane_types)
    elif daemon_query.transport_types != plane_types:
        return Response(ResponseSchema().dump({'segments': [], 'querying': False}).data)

    if len(daemon_query.dates) > settings.DAEMON_MAX_DATES_TO_REQUEST:
        poll = True

    daemon_response = collect_segments(daemon_query, send_query=not poll)

    return Response(ResponseSchema().dump(daemon_response).data)


@api_view(['GET'])
def min_dynamic_tariffs(request):
    """
    Минимальные динамические цены.
    """
    query, errors = MinDynamicTariffsQuerySchema().load(request.GET)
    if errors:
        return Response({'errors': errors})

    min_tariffs = get_dynamic_min_tariffs(query)
    return Response(MinDynamicTariffsResponseSchema().dump({'tariffs': min_tariffs}).data)


@api_view(['GET'])
def tariffs_init(request):
    """
    Инициализация поиска тарифов воздушного транспорта через тикет-демон avia.
    https://st.yandex-team.ru/RASPFRONT-6375
    Пример запроса Питер - Москва:
    /ru/tariffs/plane/?pointFrom=c2&pointTo=c213&date=2018-12-30&date=2018-12-31&national_version=ru
    Возвращает qid - идетификатор запроса
    Результаты запроса (по мере возврата от партнеров) можно получить по адресу:
    /ru/tariffs/plane/poll/?qid={qid}&skip_partners=[]
    """
    query, errors = DaemonQuerySchema().load(request.GET)
    if errors:
        return Response({'errors': errors})

    if not query.dates:
        return Response({
            'errors': {'wrongRequest': ['should be at least one date']}
        }, status=status.HTTP_400_BAD_REQUEST)

    init_response = init_plane_tariff_queries(query)
    return Response(TariffsInitResponseSchema().dump(init_response).data)


@api_view(['GET'])
def tariffs_poll(request):
    """
    Возвращает результаты запроса по известному qid (идентификатору запроса)
    https://st.yandex-team.ru/RASPFRONT-6375
    Пример:
    /ru/tariffs/plane/poll/?qid={qid}&skip_partners=[]
    """
    query, errors = TariffsPollQuerySchema().load(request.GET)
    if errors:
        return Response({'errors': errors})

    daemon_queries = [create_poll_tariff_query(query)]
    daemon_response = collect_daemon_segments(daemon_queries)

    return Response(ResponseSchema().dump(daemon_response).data)
