# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from collections import namedtuple

import six
from django.conf import settings
from django.utils.encoding import force_text
from django.utils.functional import cached_property
from enum import Enum

from common.models.transport import TransportType
from common.models_utils.i18n import RouteLTitle
from common.utils.railway import get_railway_tz_by_express_code, get_railway_tz_by_point
from travel.rasp.morda_backend.morda_backend.tariffs.train.base.utils import get_point_express_code


class PlaceReservationType(Enum):
    USUAL = 'usual'  # выкуп мест по одному
    TWO_PLACES_AT_ONCE = 'two_places_at_once'  # выкуп двух мест сразу - "мягкие" вагоны, часть СВ в Стрижах
    FOUR_PLACES_AT_ONCE = 'four_places_at_once'  # выкуп четырех мест сразу - купе-переговорные в Сапсанах


@six.python_2_unicode_compatible
class TrainSegment(object):
    key = None
    original_number = None  # Номер поезда из N1
    number = None  # Номер для отображения пользователю N2
    title = None
    title_common = None
    thread = None
    thread_variant = None
    t_model = None
    station_from = None
    station_from_ufs_title = None
    station_from_express_code = None
    station_to = None
    station_to_ufs_title = None
    station_to_express_code = None
    possible_numbers = None
    can_supply_segments = None  # Можно ли использовать сегмент для пополнения
    start_express_title_or_code = None
    end_express_title_or_code = None
    railway_departure = None
    railway_arrival = None
    departure = None
    arrival = None
    is_deluxe = None
    coach_owners = ()
    raw_train_category = None
    raw_train_name = None
    has_dynamic_pricing = False
    two_storey = False

    def __init__(self):
        self.t_type = TransportType.objects.get(pk=TransportType.TRAIN_ID)
        self.tariffs = {}
        self.L_title = RouteLTitle(self)

    def __str__(self):
        return '<TrainSegment: \n  {}\n>'.format('\n  '.join(
            ('{}: {}'.format(k, force_text(v))) for k, v in self.__dict__.items()
        ))

    @property
    def company(self):
        return self.thread and self.thread.company

    @cached_property
    def duration(self):
        return self.arrival - self.departure

    def get_duration(self):
        return int(self.duration.total_seconds() // 60)


@six.python_2_unicode_compatible
class TrainTariffsQuery(namedtuple('TrainTariffsQuery', ('partner',
                                                         'departure_point', 'arrival_point', 'departure_date'))):
    @cached_property
    def departure_point_code(self):
        return get_point_express_code(self.departure_point)

    @cached_property
    def arrival_point_code(self):
        return get_point_express_code(self.arrival_point)

    @cached_property
    def departure_railway_tz(self):
        return (get_railway_tz_by_express_code(self.departure_point_code) or
                get_railway_tz_by_point(self.departure_point))

    @cached_property
    def arrival_railway_tz(self):
        return (get_railway_tz_by_express_code(self.arrival_point_code) or
                get_railway_tz_by_point(self.arrival_point))

    @property
    def cache_key(self):
        return settings.CACHEROOT + ('{self.partner.value}_tariffs/'
                                     '{self.departure_point_code}_{self.arrival_point_code}/'
                                     '{self.departure_date:%Y%m%d}').format(self=self)

    def __str__(self):
        return force_text(super(TrainTariffsQuery, self).__repr__())
