# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.conf import settings
from django.utils.lru_cache import lru_cache

from common.data_api.train_api.tariffs.utils import (
    TIME_RANGE_HOURS, NUMBER_FORMAT, TRAIN_KEY_FORMAT, _make_segment_datetime_keys, _make_datetime_keys,
    get_reverse_train_number, get_possible_numbers, make_datetime_key, make_precise_datetime_key, _StrangeTrainNumber,
    _get_number_and_first_letter
)
from common.models.geo import Station, Settlement, StationCode, StationMajority
from common.models.transport import TransportType
from common.utils.date import MSK_TZ
from common.utils.locations import set_lang_param

USSR_CODES = ['AB', 'AZ', 'AM', 'BY', 'GE', 'KZ', 'KG', 'LV', 'LT', 'MD', 'RU', 'TJ', 'TM', 'UZ', 'UA', 'EE']


def can_sale_on_rasp(segment, point_from, point_to):
    if getattr(segment, 'old_ufs_order', False):
        return False
    if segment.thread:
        if segment.thread.first_country_code and segment.thread.first_country_code not in USSR_CODES:
            return False
        if segment.thread.last_country_code and segment.thread.last_country_code not in USSR_CODES:
            return False
    if (point_from.country and point_from.country.code == 'UA') or (point_to.country and point_to.country.code == 'UA'):
        return False
    return True


def build_old_morda_order_url(train_segment, tariff_class, tariff, point_from, point_to, national_version):
    point_from = (train_segment.station_from if train_segment.station_from else point_from)
    point_to = (train_segment.station_to if train_segment.station_to else point_to)

    params = {
        'station_from': train_segment.station_from and train_segment.station_from.id,
        'station_to': train_segment.station_to and train_segment.station_to.id,
        'point_from': point_from.point_key,
        'point_to': point_to.point_key,
        'departure': train_segment.departure.astimezone(MSK_TZ).replace(tzinfo=None),
        'arrival': train_segment.arrival.astimezone(MSK_TZ).replace(tzinfo=None),
        'title': train_segment.L_title(),
        'date': train_segment.departure.date(),
        'number': train_segment.original_number,
        't_type': train_segment.t_type.code,
        'cls': tariff_class,
        'tariff': tariff.ticket_price.value,
    }

    if train_segment.tariffs['electronic_ticket']:
        params['et'] = 't'

    if train_segment.thread:
        params['thread'] = train_segment.thread.uid

    return '/buy/?' + set_lang_param(params, national_version=national_version)


def make_thread_train_keys(thread, departure):
    if not (thread and thread.t_type_id == TransportType.TRAIN_ID):
        return []

    numbers = get_possible_numbers(thread.number)
    return _make_datetime_keys(departure, numbers)


def make_segment_train_keys(segment):
    """
    Генерируем несколько ключей, чтобы привязываться к рейсам в пределах 4 часов.
    В том случае, если вдруг наше время отличается от времени UFS
    """

    if not (segment.thread and segment.thread.t_type_id == TransportType.TRAIN_ID):
        return []

    numbers = get_possible_numbers(segment.number)
    if getattr(segment, 'letters', None) and getattr(segment, 'train_number', None):
        digits = [segment.train_number.digits, get_reverse_train_number(segment.train_number.digits)]
        numbers = ['{}{}'.format(digit, letter) for letter in segment.letters for digit in digits]

    return _make_segment_datetime_keys(segment, numbers)


def make_suburban_express_keys(departure_dt, train_purchase_numbers):
    """
    Генерируем ключи из поездов-дублей для электричек-экспрессов
    """

    if not train_purchase_numbers:
        return []

    numbers = []
    for tpn in train_purchase_numbers:
        numbers += get_possible_numbers(tpn)

    return numbers + _make_datetime_keys(departure_dt, numbers)


def make_tariff_segment_key(segment):
    make_departure_key_func = make_datetime_key if settings.ALIGN_SEARCH_SEGMENT_KEYS else make_precise_datetime_key
    return TRAIN_KEY_FORMAT.format(segment.original_number, make_departure_key_func(segment.departure))


def find_city_train_station(city):
    codes_qs = StationCode.objects.filter(station__settlement=city,
                                          station__t_type__id=TransportType.TRAIN_ID,
                                          system__code='express').values_list('station__id', 'code')

    try:
        return codes_qs.filter(station__majority__id=StationMajority.EXPRESS_FAKE_ID)[0]
    except IndexError:
        pass

    try:
        return codes_qs.filter(station__hidden=False).order_by('station__majority__id')[0]
    except IndexError:
        pass

    return None, None


@lru_cache()
def get_point_express_code(point):
    if isinstance(point, Station):
        return point.get_code('express')

    if isinstance(point, Settlement):
        _station_id, express_code = find_city_train_station(point)
        return express_code

    return None


def make_time_key(dt):
    aligned_dt = dt.replace(hour=int(dt.hour // TIME_RANGE_HOURS) * TIME_RANGE_HOURS)
    return aligned_dt.strftime('%H')


def fix_train_number(train_number):
    """
    "*" встречается, например, в поиске Тюмень Сургут 2016-05-10, правда в теге N2.
    """
    train_number = train_number.replace('*', '')
    try:
        number, letter = _get_number_and_first_letter(train_number)
    except _StrangeTrainNumber:
        return train_number
    else:
        return NUMBER_FORMAT.format(number, letter)
