# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import calendar
import logging
from collections import defaultdict
from copy import copy
from datetime import timedelta, datetime
from itertools import chain

import pytz
from dateutil.relativedelta import relativedelta
from django.conf import settings
from django.db.models import Prefetch
from django.utils.translation import get_language

from common.apps.suburban_events.api import get_states_for_thread
from common.apps.suburban_events.utils import EventStateType
from common.data_api.platforms.helpers import PathPlatformsBatch
from common.data_api.platforms.instance import platforms as platforms_client
from common.models.geo import Country, Settlement
from common.models.schedule import RThread, RThreadType, RTStation, TrainPurchaseNumber, DeLuxeTrain
from common.models.transport import TransportType
from common.models_utils.i18n import RouteLTitle
from common.settings import DAYS_TO_PAST
from common.settings.utils import define_setting
from travel.rasp.library.python.common23.date import environment
from common.utils.date import RunMask
from common.utils.iterrecipes import unique_everseen
from common.xgettext.i18n import gettext
from mapping.views.paths import draw_path
from travel.rasp.morda_backend.morda_backend.tariffs.bus.service import make_ybus_thread_tariff_keys, KEY_TEMPLATES_ALL_DAYS
from travel.rasp.morda_backend.morda_backend.tariffs.static.service import make_min_static_thread_keys, make_thread_key
from travel.rasp.morda_backend.morda_backend.tariffs.train.base.utils import make_thread_train_keys
from travel.rasp.morda_backend.morda_backend.tariffs.suburban.service import make_suburban_thread_keys


# https://st.yandex-team.ru/RASPFRONT-6415#5c0e4af51e3183001d41d100
define_setting('SUBURBAN_BUS_SUPPLIER', default={44, 37, 175, 413})

log = logging.getLogger(__name__)


class FindThreadError(Exception):
    pass


def get_thread_map(context):
    """
    Получение данных для отображения карты нитки
    """
    segment, _ = _get_segments_and_select_current(context)

    # Не показываем какрту для Московского монорельса (компания "Московский метрополитен")
    # https://st.yandex-team.ru/RASPFRONT-3221
    if segment.thread.company_id == 59942:
        return {}

    path_data = draw_path(
        thread=segment.thread,
        thread_start_dt=segment.naive_start_dt,
        path=list(segment.thread.path),
        first=segment.station_from,
        last=segment.station_to,
        time_zone=context.time_zone
    )

    if not path_data:  # cannot draw map
        return {}

    map_data = path_data.__json__()

    # Swap all the coordinates ([longitude, latitude] -> [latitude, longitude])
    # Swap them with copying because there may be references to original coordinates
    if 'first' in map_data:
        map_data['first'] = [map_data['first'][1], map_data['first'][0]]
    if 'last' in map_data:
        map_data['last'] = [map_data['last'][1], map_data['last'][0]]
    if 'position' in map_data and map_data['position']:
        map_data['position'] = [map_data['position'][1], map_data['position'][0]]
    for st in map_data['stations']:
        st[0] = [st[0][1], st[0][0]]
    for seq in map_data['segments']:
        seq[0] = [[pair[1], pair[0]] for pair in seq[0]]

    return map_data


def get_thread(context):
    """
    Получение данных для ручки нитки
    """
    main_segment, segments = _get_segments_and_select_current(context)
    path = _fill_main_segment(main_segment, context)
    segments = _get_threads_selector(segments, main_segment, context.station_from is not None, context.time_zone)
    related_threads = _get_related_threads(main_segment.thread)
    main_segment.thread.run_days = _make_run_days(segments, context.time_zone)
    _find_other_today_threads(main_segment, segments, context.time_zone)

    return {
        'thread': main_segment.thread,
        'rtstations': path,
        'threads': segments,
        'related_threads': related_threads
    }


def _get_segments_and_select_current(context):
    """
    Получение списка сегментов и определение текущего сегмента.
    :return: сегмент текущей нитки
    """
    _validate_context_dates(context.departure_from, context.departure_from_date, context.departure)
    uid, canonical_uid = _define_thread_uids(context.uid, context.mixed_uid)
    segments = _load_segments(canonical_uid, context.station_from, context.station_to)
    for segment in segments:
        segment.process_departures(context.departure_from, context.departure_from_date,
                                   context.departure, context.time_zone)
    return _define_main_segment(segments, uid, canonical_uid), segments


def _validate_context_dates(departure_from, departure_from_date, departure):
    """
    Проверка дат из контекста запроса на допустимость
    """
    if departure_from:
        _validate_path_date(departure_from.date())
    if departure_from_date:
        _validate_path_date(departure_from_date)
    if departure:
        _validate_path_date(departure)


def _validate_path_date(path_date):
    today = environment.today()
    if not(today - timedelta(days=DAYS_TO_PAST) <= path_date < today + timedelta(365 - DAYS_TO_PAST)):
        raise FindThreadError('В БД нет информации об отправлениях на дату: {}'.format(path_date.isoformat()))


def _define_thread_uids(uid, mixed_uid):
    """
    Определение canonical_uid и uid нитки по заданным uid и mixed_uid.
    """
    canonical_uid = None

    if not uid and mixed_uid:
        if mixed_uid.startswith('T_') or mixed_uid.startswith('R_'):
            canonical_uid = mixed_uid
        else:
            uid = mixed_uid

    if not canonical_uid:
        try:
            thread = RThread.objects.only('canonical_uid', 'type_id').get(uid=uid)
            canonical_uid = thread.canonical_uid
        except RThread.DoesNotExist:
            raise FindThreadError('Нитки с uid {} нет в базе'.format(uid))

        if thread.type_id == RThreadType.CANCEL_ID:
            raise FindThreadError('Нитка с uid {} является ниткой отмены'.format(uid))

    return uid, canonical_uid


def _load_segments(canonical_uid, station_from, station_to):
    """
    Загрузка списка ниток и формирование списка сегментов.
    Для каждого сегмента определяются ограничения на rtstations отправления и прибытия.
    :return Список сегментов
    """
    threads = list(RThread.objects.filter(canonical_uid=canonical_uid).prefetch_related(
        Prefetch('rtstation_set', queryset=RTStation.objects.select_related('station').all().order_by('id'))
    ))
    if not threads:
        raise FindThreadError('Ниток с canonical_uid {} нет в базе'.format(canonical_uid))

    segments = []
    for thread in threads:
        segment = ThreadSegment(thread, station_from, station_to)
        if segment.is_valid:
            segments.append(segment)
    return segments


def _define_main_segment(segments, uid, canonical_uid):
    """
    Определение текущего сегмента.
    Выбирается сегмент с самым ранним отправлением из тех, для которых определилась thread_start_date.
    Если нет сегментов не раньше текущей даты, выбирается самый поздний из тех, что раньше.
    :return: главный сегмент (ThreadSegment)
    """
    candidates = {segment.thread.uid: segment
                  for segment in segments
                  if segment.thread_start_date and segment.thread.type_id != RThreadType.CANCEL_ID}
    if uid:
        if uid in candidates:
            main_segment = candidates[uid]
        else:
            raise FindThreadError('Нитка с uid {} не подходит под параметры запроса'.format(uid))
    else:
        if len(candidates) == 0:
            raise FindThreadError(
                'Подходящих под параметры запроса ниток с canonical_uid {} нет в базе'.format(canonical_uid)
            )

        today_and_future = [segment
                            for segment in candidates.values()
                            if segment.station_from_dt.date() >= environment.today()]
        if today_and_future:
            main_segment = min(today_and_future, key=lambda s: s.station_from_dt)
        else:
            main_segment = max(candidates.values(), key=lambda s: s.station_from_dt)

    main_segment.current = True
    return main_segment


class ThreadSegment(object):
    """
    Сегмент, содержащий нитку с указанием информации о начальной и конечной станции.
    В начале играет роль кандидата на роль главного сегмента и может оказаться главным.
    Потом становится элементом селектора ниток.
    """
    def __init__(self, thread, station_from, station_to):
        self.thread = thread
        self.path = None
        self.rtstation_from = None
        self.station_from = None
        self.rtstation_to = None
        self.station_to = None

        # Свойства для определния главного сегмента
        self.is_valid = True
        self.has_context_stations = True
        self.rtstations_from_indices = []
        self.rtstations_to_indices = []
        self.thread_start_date = None
        self.station_from_dt = None
        self.naive_start_dt = None

        # Свойства элемента селектора ниток
        self.current = False
        self.in_context = True
        self.start_departure_time = None  # Строка
        self.stop_arrival_time = None  # Строка
        self.first_departure_dt = None
        self.first_departure = None
        self.days_text = None

        self._load_path()
        self._validate()
        if self.is_valid:
            self._define_stations_from_to(station_from, station_to)

    # Свойства для сериализации
    @property
    def id(self):
        return self.thread.id

    @property
    def uid(self):
        return self.thread.uid

    @property
    def title(self):
        return self.thread.L_title(lang=get_language())

    @property
    def type(self):
        return self.thread.type.code

    def set_thread_deluxe_train(self):
        if self.thread.t_type.id == TransportType.TRAIN_ID:
            number = self.thread.number
            if number:
                deluxe_train = DeLuxeTrain.get_by_number(number)
                if deluxe_train:
                    self.thread.deluxe = deluxe_train

    def get_thread_full_title(self):
        def build_title(main, parts):
            if any(part for part in parts):
                return '{} {}'.format(main, ' '.join(part for part in parts if part))
            else:
                return main

        thread = self.thread
        number = thread.number
        title = thread.L_title()

        if thread.t_type.id == TransportType.TRAIN_ID:
            # если в rtstation есть номера поездов, берем их в порядке следования станций
            # https://st.yandex-team.ru/RASPFRONT-8346
            number = '/'.join(unique_everseen(s.train_number for s in self.path if s.train_number)) or number

            if thread.type.id == RThreadType.THROUGH_TRAIN_ID:
                return build_title(gettext('Беспересадочный вагон'), [number, title])

            if hasattr(thread, 'deluxe'):
                deluxe_title = thread.deluxe.L_title()
                if deluxe_title:
                    deluxe_title = '«{}»'.format(deluxe_title)
                if thread.deluxe.deluxe:
                    return build_title(gettext('Фирменный поезд'), [deluxe_title, number, title])
                else:
                    return build_title(gettext('Поезд'), [deluxe_title, number, title])

            return build_title(gettext('Поезд'), [number, title])

        if thread.t_type.id in [TransportType.SUBURBAN_ID, TransportType.BUS_ID, TransportType.WATER_ID]:
            if thread.t_type.id == TransportType.SUBURBAN_ID:
                default_type = gettext('Поезд')
            elif thread.t_type.id == TransportType.BUS_ID:
                default_type = gettext('Маршрут автобуса')
            elif thread.t_type.id == TransportType.WATER_ID:
                default_type = gettext('Теплоход')

            return build_title(default_type, [number, title])

        return title

    def _load_path(self):
        """
        Загрузка списка rtstation нитки сегмента c привязанными station
        """
        path = list(self.thread.rtstation_set.all())
        path.sort(key=lambda rts: rts.id)
        self.path = _filter_path(self.thread, path)

    def _validate(self):
        """
        Проверка корректности нитки
        """
        self.is_valid = len(self.path) > 1 and self.thread.first_run(environment.today()) is not None
        if not self.is_valid:
            self.has_context_stations = False

    def _define_stations_from_to(self, station_from, station_to):
        """
        Формирует свойства station_from и station_to сегмента, если station_to не задано, то и у сегмента оно не задано.
        Формирует списки возможных индексов станций from и to.
        """
        if station_from:
            self.station_from = station_from
        else:
            self.rtstations_from_indices.append(0)
            self.station_from = self.path[0].station

        if station_to:
            self.station_to = station_to
        else:
            self.rtstations_to_indices.append(len(self.path) - 1)

        for index in range(len(self.path)):
            if self._is_station_at_index(index, station_from) and self.path[index].tz_departure is not None:
                self.rtstations_from_indices.append(index)
            if self._is_station_at_index(index, station_to) and self.path[index].tz_arrival is not None:
                self.rtstations_to_indices.append(index)

    def _is_station_at_index(self, index, station):
        rts = self.path[index]
        return rts.station == station and not (rts.is_no_stop() or rts.is_technical_stop)

    def process_departures(self, departure_from, departure_from_date, departure, time_zone):
        """
        Обработка сегмента в соответствии с разными способами задания времени отправления.
        Для подходящих сегментов заполняются свойства сегмента, как кандидата на роль главного.
        """
        if departure_from:
            self._process_for_departure_from(departure_from)
        elif departure_from_date:
            self._process_for_departure_from_date(departure_from_date, time_zone)
        elif departure:
            self._process_for_departure(departure)
        else:
            self._process_without_context()

    def _process_for_departure_from(self, departure_from):
        """
        Проверка сегмента на наличие станции отправления с временем отправления departure_from в таймзоне станции.
        Если станция нашлась, то заполняются свойства сегмента, как кандидата.
        """
        departure_dt_loc = self.station_from.pytz.localize(departure_from)
        for index in self.rtstations_from_indices:
            rts = self.path[index]
            thread_start_dt = departure_dt_loc.astimezone(rts.pytz) - timedelta(minutes=rts.tz_departure)
            if thread_start_dt.time() == self.thread.tz_start_time and self.thread.runs_at(thread_start_dt.date()):
                thread_start_date = rts.calc_thread_start_date(event='departure', event_date=departure_from.date())
                if self._try_define_rtstations(index, thread_start_date):
                    return

    def _process_for_departure_from_date(self, departure_from_date, time_zone):
        """
        Проверка сегмента на наличие станции отправления с датой отправления departure_from_date
        в таймзоне time_zone или таймзоне станции, если time_zone не задана.
        Если станция нашлась, то заполняются свойства сегмента, как кандидата.
        """
        for index in self.rtstations_from_indices:
            rts = self.path[index]
            tz = _get_station_pytz(rts.station, time_zone)
            thread_start_date = rts.calc_thread_start_date(event='departure',
                                                           event_date=departure_from_date, event_tz=tz)
            if self.thread.runs_at(thread_start_date):
                if self._try_define_rtstations(index, thread_start_date):
                    break

    def _process_for_departure(self, departure):
        """
        Проверка сегмента на наличие рейса с датой отправления departure в таймзоне нитки от начальной станции нитки.
        Если сегмент подошел, то заполняются свойства сегмента, как кандидата.
        """
        thread_start_date = self.path[0].calc_thread_start_date('departure', departure, self.thread.pytz)
        if self.thread.runs_at(thread_start_date):
            self._try_define_rtstations(0, thread_start_date)

    def _process_without_context(self):
        """
        Обработка сегмента, если на заданы departure, departure_from и departure_from_date.
        Заполняются свойства сегмента, как кандидата.
        """
        thread_start_date = self.thread.first_run(environment.today())
        self._try_define_rtstations(0, thread_start_date)

    def _try_define_rtstations(self, from_index, thread_start_date):
        """
        Проверяет, может ли станция с индексом from_index быть станцией оправления (есть station_to после).
        Если может, то для сегмента вычисляются различные времена отправления.
        """
        for to_index in self.rtstations_to_indices:
            if to_index > from_index:
                self.rtstation_from = self.path[from_index]
                self.rtstation_to = self.path[to_index]

                self.thread_start_date = thread_start_date
                self.naive_start_dt = datetime.combine(thread_start_date, self.thread.tz_start_time)
                self.station_from_dt = self.rtstation_from.get_departure_loc_dt(self.naive_start_dt)
                return True
        return False

    def prepare_for_selector(self, main_segment, main_segment_in_context, time_zone):
        """
        Формирует время следования по первой и последней станции сегмента.
        Заполняет свойства для сериализации сегмента, как элемента селектора ниток.
        """
        self._define_rtstations_for_selector(main_segment)

        days_shift = self.rtstation_from.calc_days_shift(event='departure')
        thread_first_departure_date = self.thread.first_run(environment.today() - timedelta(days_shift))
        self.naive_first_departure_dt = datetime.combine(thread_first_departure_date, self.thread.tz_start_time)

        # Для поездов данные определяются в контексте границ сегмента, для остальных типов транспорта - вне контекста.
        is_train = self.thread.t_type_id == TransportType.TRAIN_ID
        self.in_context = self.has_context_stations and main_segment_in_context and is_train

        first_rts = self.rtstation_from if self.in_context else self.path[0]
        last_rts = self.rtstation_to if self.in_context else self.path[-1]

        # Для контекстных сегментов first_departure содержит время первого отправления от станции в таймзоне станции.
        # Для неконтекстных содержит дату первого отправления с начальной станции в таймзоне нитки.
        self.first_departure_dt = first_rts.get_departure_loc_dt(self.naive_first_departure_dt)
        first_dt = (datetime.combine(thread_first_departure_date, self.thread.tz_start_time)
                    + timedelta(minutes=self.path[0].tz_departure))  # tz_departure не всегда 0
        first_date = self.thread.pytz.localize(first_dt).date()
        self.first_departure = self.first_departure_dt.isoformat() if self.in_context else first_date.isoformat()

        # days_text, start_departure_time и stop_arrival_time вычисляются в time_zone, если она задана,
        # если не задана, то вычисляются в таймзоне станций отправления и прибытия.
        self.days_text = _make_run_days_text(self.thread, self.naive_first_departure_dt, first_rts, time_zone)

        self.start_departure_time = self.get_departure_time(first_rts, time_zone).strftime('%H:%M')
        self.stop_arrival_time = self.get_arrival_time(last_rts, time_zone).strftime('%H:%M')

    def _define_rtstations_for_selector(self, main_segment):
        """
        Определение станций отправления и прибытия для каждого элемента селектора ниток.
        """
        if self.current:
            return

        self.rtstation_from = self.rtstation_to = None
        for index in range(len(self.path)):
            rts = self.path[index]
            if self._is_station_at_index(index, main_segment.station_from) and rts.tz_departure is not None:
                self.rtstation_from = rts
            if (self.rtstation_from
                    and self._is_station_at_index(index, main_segment.station_to)
                    and rts.tz_arrival is not None):
                self.rtstation_to = rts
                return

        if self.rtstation_from and not main_segment.station_to:
            self.rtstation_to = self.path[-1]

        if not self.rtstation_from or not self.rtstation_to:
            self.has_context_stations = False
            self.rtstation_from = self.path[0]
            self.rtstation_to = self.path[-1]

    def get_departure_time(self, rtstation, time_zone):
        return rtstation.get_departure_dt(
            self.naive_first_departure_dt, _get_station_pytz(rtstation.station, time_zone)
        ).time()

    def get_arrival_time(self, rtstation, time_zone):
        return rtstation.get_arrival_dt(
            self.naive_first_departure_dt, _get_station_pytz(rtstation.station, time_zone)
        ).time()


def _fill_main_segment(segment, context):
    """
    Формирует наполнение главного сегмента страницы нитки.
    Получает данные, еще не полученные в segment, и кладет их в segment.thread или возвращает в path.
    :return набор rtstation нитки, готовый к сериализации.
    """
    RouteLTitle.fetch([segment.thread.L_title])
    segment.set_thread_deluxe_train()
    segment.thread.full_title = segment.get_thread_full_title()

    segment.thread.from_station_departure_local_dt = segment.station_from_dt.isoformat()
    segment.thread.is_no_change_wagon = segment.thread.type_id == RThreadType.THROUGH_TRAIN_ID
    segment.thread.run_days_text = _make_run_days_text(segment.thread, segment.naive_start_dt,
                                                       segment.rtstation_from, context.time_zone)
    has_context_departure = context.departure or context.departure_from or context.departure_from_date
    _add_tariffs(segment, has_context_departure, context.bus_settlement_keys)

    _add_domain_capital_info(segment, context.country)
    path = _make_path(segment, context.time_zone)
    _add_capitals_time_zones_info(segment, path, context.time_zone)
    _add_suburban_states(segment.thread, path)
    _update_platforms_by_dynamics(path, segment.naive_start_dt, segment.thread.number)

    return path


def _get_station_pytz(station, time_zone):
    """
    Возвращает pytz указанной таймзоны или станции
    """
    return pytz.timezone(time_zone) if time_zone else station.pytz


def _make_run_days_text(thread, naive_start_dt, rts_from, time_zone=None):
    """
    Формирование текста дней отправления от начальной станции сегмента
    """
    event_tz = pytz.timezone(time_zone) if time_zone else None
    shift = rts_from.calc_days_shift(event='departure', start_date=naive_start_dt, event_tz=event_tz)
    return thread.L_days_text(
        shift,
        except_separator=', ',
        html=False,
        template_only=False,
        show_days=True,
        show_all_days=False,
        thread_start_date=naive_start_dt.date()
    )


def _add_tariffs(segment, has_departure_context, bus_settlement_keys):
    """
    Добавляет в нитку информацию о тарифах
    """
    thread = segment.thread
    if thread.t_type_id in [TransportType.BUS_ID, TransportType.TRAIN_ID, TransportType.SUBURBAN_ID]:

        departure_dt = segment.station_from_dt.replace(tzinfo=None)
        station_from = segment.rtstation_from.station
        station_to = segment.rtstation_to.station

        train_purchase_numbers = []
        if thread.t_type_id == TransportType.SUBURBAN_ID:
            # добавялем ключи тарифов только для экспрессов ЦППК https://st.yandex-team.ru/RASPFRONT-8580
            if not thread.t_subtype or not thread.t_subtype.has_train_tariffs:
                return

            result = TrainPurchaseNumber.get_train_purchase_numbers([thread])
            train_purchase_numbers = frozenset(v for row in result.values() for v in row)

        if has_departure_context:
            tariffs = chain(
                make_ybus_thread_tariff_keys(thread, departure_dt, station_from, station_to, bus_settlement_keys=bus_settlement_keys),
                [make_thread_key(departure_dt, station_from.id, station_to.id, thread.uid)],
                make_thread_train_keys(thread, departure_dt),
                make_suburban_thread_keys(station_from.id, station_to.id, thread, departure_dt, train_purchase_numbers)
            )
        else:
            tariffs = chain(
                make_ybus_thread_tariff_keys(thread, departure_dt, station_from, station_to, KEY_TEMPLATES_ALL_DAYS, bus_settlement_keys=bus_settlement_keys),
                make_min_static_thread_keys(station_from.id, station_to.id, thread),
                make_thread_train_keys(thread, departure_dt),
                make_suburban_thread_keys(station_from.id, station_to.id, thread, departure_dt, train_purchase_numbers)
            )

        thread.tariffs_keys = tariffs

        if thread.t_type_id == TransportType.BUS_ID:
            thread.is_suburban_bus = thread.supplier.id in settings.SUBURBAN_BUS_SUPPLIER


def _add_capitals_time_zones_info(segment, path, time_zone):
    """
    Получение списка столиц стран по которым идет нитка, используемая для формирования списка таймзон.
    segment - сегмент главной нитки, path - rtstations сегмента, time_zone - таймзона из контекста нитки.
    :return список столиц, таймзоны по которым нужно показывать.
    """
    countries_dict = defaultdict(set)
    input_time_zone_not_in_thread = time_zone is not None
    for rts in path:
        country_id = rts.station.country_id
        zone = rts.station.time_zone
        countries_dict[country_id].add(zone)
        if zone == time_zone:
            input_time_zone_not_in_thread = False

    countries = Country.objects.filter(id__in=list(countries_dict.keys()))

    language = get_language()
    capitals = []
    for country in countries:
        capital_tz = country.get_capital_tz()
        if capital_tz == time_zone:
            input_time_zone_not_in_thread = False
        # Таймзона столицы добавляется всегда, кроме случая,
        # когда нитка проходит только через таймзону столицы одной страны, и эта таймзона не передается на вход ручки
        if (len(countries) > 1 or input_time_zone_not_in_thread or len(countries_dict[country.id]) > 1
                or capital_tz not in countries_dict[country.id]):
            capitals.append(_get_capital_data(country.get_capital(), capital_tz, language))

    if input_time_zone_not_in_thread:
        init_capitals = list(Settlement.objects.filter(time_zone=time_zone, majority__id=1))
        if init_capitals:
            capitals.append(_get_capital_data(init_capitals[0], time_zone, language))

    segment.thread.capitals = capitals


def _get_capital_data(capital, capital_tz, language):
    return {
        'slug': capital.slug,
        'time_zone': capital_tz,
        'title': capital.L_title(lang=language),
        'title_genitive': capital.L_title(lang=language, case='genitive', fallback=False),
        'abbr_title': capital.L_abbr_title(lang=language),
    }


def _add_domain_capital_info(segment, country_code):
    """
    Добавление в segment.thread информации о столице страны, в домене которой зашел пользователь
    """
    country = Country.objects.get(code=country_code)
    segment.thread.capital_slug = country.get_capital().slug
    segment.thread.capital_tz = country.get_capital_tz()


def _make_path(segment, time_zone):
    """
    Формирование списка rtstations, составляющих маршрут нитки
    """
    path = list(segment.thread.path.select_related(
        'station', 'station__settlement')
    )
    path = _filter_path(segment.thread, path)
    for rts in path:
        rts.is_station_from = rts == segment.rtstation_from
        rts.is_station_to = rts == segment.rtstation_to

        tz = _get_station_pytz(rts.station, time_zone)
        if rts.tz_departure is not None:
            rts.departure_dt = rts.get_departure_dt(segment.naive_start_dt, tz)
        if rts.tz_arrival is not None:
            rts.arrival_dt = rts.get_arrival_dt(segment.naive_start_dt, tz)

        _calc_time_zone_difference(rts, segment)

    return path


def _filter_path(thread, path):
    """
    Дополнительная фильтрация списка rtstation
    """
    # Для экспрессов и аэроэкспрессов не показываем остановки без остановки
    if thread.t_type_id == TransportType.SUBURBAN_ID and (
        thread.is_aeroexpress or thread.is_express
    ):
        return [rts for rts in path if not rts.is_no_stop()]
    return path


def _calc_time_zone_difference(rts, segment):
    """
    Вычисление сдвига таймзоны станции относительно столицы
    """
    capital_tz = pytz.timezone(segment.thread.capital_tz)
    if rts.tz_departure is not None:
        rts_dt = rts.get_departure_dt(segment.naive_start_dt)
    else:
        rts_dt = rts.get_arrival_dt(segment.naive_start_dt)

    capital_dt = rts_dt.astimezone(capital_tz).replace(tzinfo=None)
    rts_dt = rts_dt.astimezone(rts.station.pytz).replace(tzinfo=None)
    delta = relativedelta(rts_dt, capital_dt)
    rts.capital_time_offset = '{}{:02}:{:02}'.format('+' if delta.hours >= 0 else '-', abs(delta.hours), delta.minutes)


def _add_suburban_states(thread, path):
    """
    Для электричек добавляется информация об опозданиях
    """
    if thread.t_type_id == TransportType.SUBURBAN_ID:
        start_date = path[0].departure_dt.astimezone(thread.pytz).date()
        try:
            thread_state, station_states = get_states_for_thread(
                thread, start_date, path, all_keys=True, cancels_as_possible_delay=False
            )
        except Exception as ex:
            log.exception(repr(ex))
            station_states = {}

        _attach_states_to_stations(path, station_states)


def _update_platforms_by_dynamics(path, naive_start_dt, train_number):
    if not path:
        return

    dynamic_platforms = PathPlatformsBatch(naive_start_dt, train_number)
    dynamic_platforms.try_load(platforms_client, path)

    for rts in path:
        rts.platform = dynamic_platforms.get_platform(rts, rts.platform)


def _attach_states_to_stations(path, station_states):
    for rtstation in path:
        rts_state = station_states.get(rtstation)
        if rts_state:
            state = {}
            for event in ['arrival', 'departure']:
                event_state = getattr(rts_state, event)
                if event_state:
                    state[event] = _get_event_state_dict(event_state)
            if state:
                state['key'] = rts_state.key
                rtstation.state = state


def _get_event_state_dict(state):
    event_state_dict = {}
    if state.type:
        event_state_dict['type'] = state.type

    if state.type == EventStateType.FACT:
        dt = pytz.timezone(state.tz).localize(state.dt)
        event_state_dict['fact_time'] = dt.isoformat()

    if state.type in [
        EventStateType.POSSIBLE_DELAY, EventStateType.FACT_INTERPOLATED, EventStateType.FACT, EventStateType.CANCELLED
    ]:
        event_state_dict['minutes_from'] = state.minutes_from
        event_state_dict['minutes_to'] = state.minutes_to

    return event_state_dict


def _get_threads_selector(segments, main_segment, main_segment_is_in_context, time_zone):
    """
    Формирование селектора ниток на основе списка сегментов.
    :return: Список сегментов как элементов селектора ниток
    """
    for segment in segments:
        segment.prepare_for_selector(main_segment, main_segment_is_in_context, time_zone)

    if segments and segments[0].thread.t_type_id == TransportType.TRAIN_ID:
        segments = _get_united_segments(segments, time_zone)

    return segments


def _get_united_segments(segments, time_zone):
    """
    Формирование элементов селектора поездов. Обычно элемент имеет тип UnitedThreadSegment.
    В один элемент группируются сегменты с одинаковыми заголовками, временами отправления-прибытия, контекстом, типом.
    Элементы не группируются, если текст дней хождения прописан явно (не пустое значение plain_days_text)
    """
    uniteds_dict = {}
    manuals = []
    for segment in segments:
        if segment.thread.plain_days_text(segment.rtstation_from.calc_days_shift(event='departure')):
            manuals.append(segment)
        else:
            key = (
                segment.title,
                segment.start_departure_time,
                segment.stop_arrival_time,
                segment.in_context,
                segment.has_context_stations,
                segment.type
            )
            if key not in uniteds_dict:
                uniteds_dict[key] = UnitedThreadSegment(key)
            uniteds_dict[key].segments.append(segment)

    for united in uniteds_dict.values():
        united.calc_attributes(time_zone)

    uniteds = list(uniteds_dict.values()) + manuals
    uniteds.sort(key=lambda u: (u.in_context, u.type, u.first_departure_dt))
    return uniteds


class UnitedThreadSegment(object):
    """
    Объединенный набор сегментов для селектора поездов.
    Один набор объединяет сегменты с одинаковымы заголовком ниток, временами отправления и прибытия,
    наличием контекста, наличием станций контекста и типом нитки
    """
    def __init__(self, key):
        (
            self.title, self.start_departure_time, self.stop_arrival_time,
            self.in_context, self.has_context_stations, self.type
        ) = key
        self.days_text = None
        self.current = False
        self.id = None
        self.uid = None
        self.first_departure_dt = None
        self.first_departure = None
        self.station_from = None
        self.rtstation_from = None

        self.segments = []

    def calc_attributes(self, time_zone):
        """
        Вычисляет различные атрибуты набора сегментов.
        Объединяет маски ниток и формирует текст дней хождения.
        """

        # Сортируем сегменты. Если у сегмента все дни хождения только в прошлом, то его ставим в конец.
        def _sort_key(seg):
            if seg.first_departure_dt.date() >= environment.today():
                return seg.first_departure_dt
            return seg.first_departure_dt + timedelta(days=366)

        self.segments.sort(key=lambda s: _sort_key(s))
        self.thread = copy(self.segments[0].thread)
        self.thread.id = None
        self.id = self.segments[0].thread.id
        self.uid = self.segments[0].thread.uid
        self.first_departure_dt = self.segments[0].first_departure_dt
        self.first_departure = self.segments[0].first_departure
        self.naive_first_departure_dt = self.segments[0].naive_first_departure_dt
        self.station_from = self.segments[0].station_from
        self.rtstation_from = self.segments[0].rtstation_from

        mask = RunMask(self.thread.year_days)
        for segment in self.segments:
            if segment.current:
                self.current = True
            mask = mask | RunMask(segment.thread.year_days)

        self.thread.year_days = str(mask)
        self.days_text = _make_run_days_text(self.thread, self.segments[0].naive_first_departure_dt,
                                             self.rtstation_from, time_zone)


def _get_related_threads(main_thread):
    """
    Формирование списка ниток, связанных с данной.
    Сейчас формируются только связи поездов с безпересадочными вагонами, связь по route_id нитки.
    :return: Список связанных ниток
    """
    if main_thread.t_type_id != TransportType.TRAIN_ID:
        return []

    threads = RThread.objects.filter(route_id=main_thread.route_id)
    language = get_language()
    related_dict = {}
    for thread in threads:
        if thread.canonical_uid != main_thread.canonical_uid and thread.canonical_uid not in related_dict:
            relation_type = (
                RelatedThread.NO_CHANGE_WAGON_RELATION
                if thread.type_id == RThreadType.THROUGH_TRAIN_ID
                else RelatedThread.BASIC_TRAIN_RELATION
            )
            related_dict[thread.canonical_uid] = RelatedThread(thread.canonical_uid,
                                                               thread.L_title(lang=language), relation_type)

    return list(related_dict.values())


class RelatedThread:
    BASIC_TRAIN_RELATION = 'basicTrain'
    NO_CHANGE_WAGON_RELATION = 'noChangeWagon'

    def __init__(self, canonical_uid, title, relation_type):
        self.canonical_uid = canonical_uid
        self.title = title
        self.relation_type = relation_type


def _make_run_days(segments, time_zone):
    """
    Формирование общей маски хождения для всех указанных сегментов ниток.
    :return: Маска дней хождения, сгруппированная по годам и месяцам.
    Для каждого дня указан id нитки сегмента или объединенного сегмента. Если нитка не ходит, то указан 0.
    """
    run_days = defaultdict(dict)
    # Порядок сортировки ниток по типу (Основная и др. (0), Изменение (1), Отмена (2))
    order = {RThreadType.CHANGE_ID: 1, RThreadType.CANCEL_ID: 2}
    for segment in sorted(segments, key=lambda s: order.get(s.thread.type_id, 0)):
        if not segment.has_context_stations:
            continue

        thread_mask, from_date = _calc_thread_mask(segment, time_zone)
        for run_date in thread_mask.iter_dates(past=True):
            year = from_date.year if from_date.month <= run_date.month else from_date.year + 1
            year_key = str(year)
            if year_key not in run_days:
                run_days[year_key] = dict()

            month_key = str(run_date.month)
            if month_key not in run_days[year_key]:
                last_day_idx = calendar.monthrange(year, run_date.month)[1]
                month_days = [0] * last_day_idx
                run_days[year_key][month_key] = month_days

            run_days[year_key][month_key][run_date.day - 1] = segment.id

    return run_days


def _calc_thread_mask(segment, time_zone):
    """
    Вычисление маски дней хождения нитки.
    Маска вычисляется начиная с 30-го дня перед текущей датой
    """
    from_date = environment.now_aware().astimezone(segment.station_from.pytz).date() - timedelta(DAYS_TO_PAST)
    tz = _get_station_pytz(segment.station_from, time_zone)
    shift = segment.rtstation_from.calc_days_shift(
        event='departure', start_date=segment.naive_first_departure_dt, event_tz=tz
    )
    return RunMask(segment.thread.year_days, from_date).shifted(shift), from_date


def _find_other_today_threads(main_segment, segments, time_zone):
    """
    Если есть нитки, идущие со станции отправления в тот же день, что и главная,
    то они добавляются в main_segment.thread.other_today_threads
    """
    others = []
    from_date = main_segment.station_from_dt.date()
    for united in segments:
        if isinstance(united, ThreadSegment):
            _add_other_today_thread(others, united, main_segment, from_date, time_zone)
        else:
            for segment in united.segments:
                _add_other_today_thread(others, segment, main_segment, from_date, time_zone)

    if others:
        main_segment.thread.other_today_threads = others


def _add_other_today_thread(others, segment, main_segment, from_date, time_zone):
    """
    Проверяет, стартует ли segment в тот же день, что и main_segment.
    Если стартует, то формирует ее для сериализации и добавляет в others.
    """
    if segment == main_segment or not segment.has_context_stations:
        return

    mask, _ = _calc_thread_mask(segment, time_zone)
    for run_date in mask.iter_dates(past=True):
        if run_date == from_date:
            from_time = segment.get_departure_time(segment.rtstation_from, time_zone)
            tz = _get_station_pytz(segment.rtstation_from.station, time_zone)
            departure_dt = (
                tz.localize(datetime.combine(from_date, from_time)).astimezone(segment.rtstation_from.station.pytz)
            )

            others.append({
                'uid': segment.uid,
                'title': segment.title,
                'start_departure_time': from_time.strftime('%H:%M'),
                'stop_arrival_time': segment.get_arrival_time(segment.rtstation_to, time_zone).strftime('%H:%M'),
                'departure_dt': departure_dt.isoformat()
            })
        if run_date >= from_date:
            break
