# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from django.http import QueryDict
from marshmallow import Schema, fields, pre_load, post_dump, post_load, validates_schema, ValidationError, missing

from common.models.transport import TransportType
from common.utils.namedtuple import namedtuple_with_defaults
from travel.rasp.morda_backend.morda_backend.serialization.fields import StationField
from travel.rasp.morda_backend.morda_backend.tariffs.bus.serialization import BusSettlementKeysSchemaMixin


ThreadContext = namedtuple_with_defaults('ThreadContext', [
    'uid', 'mixed_uid',
    'station_from', 'station_to',
    'departure', 'departure_from', 'departure_from_date',
    'country', 'time_zone', 'bus_settlement_keys'
], defaults={'country': 'RU'})


class ThreadContextQuerySchema(Schema, BusSettlementKeysSchemaMixin):
    uid = fields.String()
    mixed_uid = fields.String()
    station_from = StationField()
    station_to = StationField()
    departure = fields.Date()
    departure_from_date = fields.Date()  # calculated, use only if the date of departure is in request
    departure_from = fields.DateTime()  # use if full datetime of departure in request
    country = fields.String()
    time_zone = fields.String()

    @pre_load
    def separate_departure_from(self, data):
        if 'departure_from' in data:
            if isinstance(data, QueryDict):
                data = data.dict()
            data['departure_from_date'] = data['departure_from']
            if len(data['departure_from']) <= 10:  # is date by ISO8601 date format length
                del data['departure_from']
        return data

    @validates_schema(skip_on_field_errors=True)
    def validate_params(self, data):
        if 'uid' not in data and 'mixed_uid' not in data:
            raise ValidationError(
                'uid or mixed_uid should be in the request params',
                ['uid', 'mixed_uid']
            )
        if ('departure_from' in data or 'departure_from_date' in data) and 'station_from' not in data:
            raise ValidationError(
                'station_from should be set with departure_from together',
                ['station_from', 'departure_from']
            )

    @post_load
    def make_context(self, data):
        return ThreadContext(**data)


class ResponseThreadCompanySchema(Schema):
    title = fields.String()
    address = fields.String()
    description = fields.String()
    url = fields.String()
    phone = fields.String()
    email = fields.String()
    hidden = fields.Bool()
    strange = fields.Bool()


class ResponseCapitalSchema(Schema):
    slug = fields.String()
    time_zone = fields.String(dump_to='timeZone')
    title = fields.String()
    title_genitive = fields.String(dump_to='titleGenitive')
    abbr_title = fields.String(dump_to='abbrTitle')
    departure_dt = fields.String(dump_to='departureDt')


class ResponseOtherTodayThreadSchema(Schema):
    uid = fields.String()
    title = fields.String()
    start_departure_time = fields.String(dump_to='startDepartureTime')
    stop_arrival_time = fields.String(dump_to='stopArrivalTime')
    departure_dt = fields.String(dump_to='departureDt')


class ResponseDeluxeTrainSchema(Schema):
    title = fields.Function(lambda obj: obj.L_title())
    short_title = fields.Function(lambda obj: obj.L_title_short(), dump_to='shortTitle')
    is_deluxe = fields.Boolean(attribute='deluxe', dump_to='isDeluxe', default=False)


class ResponseThreadSchema(Schema):
    uid = fields.String()
    canonical_uid = fields.String(dump_to='canonicalUid')
    title = fields.Function(lambda obj: obj.L_title())
    full_title = fields.String(dump_to='fullTitle')
    number = fields.String()
    company = fields.Nested(ResponseThreadCompanySchema, default=None)
    days_text = fields.String(attribute='run_days_text', dump_to='daysText')
    run_days = fields.Dict(dump_to='runDays')
    is_suburban_bus = fields.Bool(dump_to='isSuburbanBus')
    transport_type = fields.Function(lambda obj: obj.t_type.code, dump_to='transportType')
    tariffs_keys = fields.List(fields.String, dump_to='tariffsKeys')
    comment = fields.String()
    from_station_departure_local_dt = fields.String(dump_to='fromStationDepartureLocalDt')

    is_interval = fields.Bool(dump_to='isInterval')
    # For interval rthreads:
    begin_time = fields.Date(dump_to='beginTime')
    end_time = fields.Date(dump_to='endTime')
    density = fields.String()

    capital_slug = fields.String(dump_to='capitalSlug')
    capital_tz = fields.String(dump_to='capitalTimezone')
    capitals = fields.Nested(ResponseCapitalSchema, many=True)
    other_today_threads = fields.Nested(ResponseOtherTodayThreadSchema, dump_to='otherTodayThreads', many=True)

    is_no_change_wagon = fields.Boolean(dump_to='isNoChangeWagon')
    deluxe = fields.Nested(ResponseDeluxeTrainSchema, dump_to='deluxeTrain')
    is_express = fields.Boolean(dump_to='isExpress', default=False)
    is_aeroexpress = fields.Boolean(dump_to='isAeroExpress', default=False)

    @post_dump
    def remove_skip_values(self, data):
        if not data['isInterval']:
            del data['beginTime']
            del data['endTime']
            del data['density']
        return data


class ResponseSettlementSchema(Schema):
    id = fields.Int()
    title = fields.Function(lambda obj: obj.L_title() if obj else missing)


class ResponseCountrySchema(Schema):
    id = fields.Int()
    code = fields.String()
    title = fields.Function(lambda obj: obj.L_title() if obj else missing)


class ResponseStateSchema(Schema):
    type = fields.String()
    fact_time = fields.String(dump_to='factTime')
    minutes_from = fields.Int(dump_to='minutesFrom')
    minutes_to = fields.Int(dump_to='minutesTo')


class ResponseThreadStationStateSchema(Schema):
    key = fields.String()
    arrival = fields.Nested(ResponseStateSchema)
    departure = fields.Nested(ResponseStateSchema)


class ResponseThreadStationSchema(Schema):
    id = fields.Function(lambda obj: obj.station.id)
    slug = fields.Function(lambda obj: obj.station.slug)
    settlement = fields.Function(lambda obj: ResponseSettlementSchema().dump(obj.station.settlement).data)
    country = fields.Function(lambda obj: ResponseCountrySchema().dump(obj.station.country).data)
    title = fields.Method('get_title')
    page_type = fields.Function(lambda obj: obj.station.page_type, dump_to='pageType')
    main_subtype = fields.Function(lambda obj: obj.station.main_subtype, dump_to='mainSubtype')
    is_station_from = fields.Boolean(dump_to='isStationFrom')
    is_station_to = fields.Boolean(dump_to='isStationTo')
    platform = fields.String()
    timezone = fields.Function(lambda obj: obj.station.time_zone)
    url = fields.Function(lambda obj: _get_station_url(obj.station))
    arrival_local_dt = fields.LocalDateTime(attribute='arrival_dt', dump_to='arrivalLocalDt')
    departure_local_dt = fields.LocalDateTime(attribute='departure_dt', dump_to='departureLocalDt')
    capital_time_offset = fields.String(dump_to='capitalTimeOffset')
    is_fuzzy = fields.Bool(dump_to='isFuzzy', default=False)
    is_technical_stop = fields.Bool(dump_to='isTechnicalStop')
    hidden = fields.Function(lambda obj: obj.station.hidden)
    is_no_stop = fields.Function(lambda obj: obj.is_no_stop(), dump_to='isNoStop')
    state = fields.Nested(ResponseThreadStationStateSchema)
    is_combined = fields.Boolean(dump_to='isCombined')

    @classmethod
    def get_title(cls, obj):
        if obj.thread.t_type.id in [TransportType.SUBURBAN_ID]:
            return obj.station.L_title_with_railway_prefix()
        else:
            return obj.station.L_title()


def _get_station_url(station):
    return '/station/{}'.format(station.id)


class ResponseThreadItemSchema(Schema):
    uid = fields.String()
    title = fields.String()
    type = fields.String()
    in_context = fields.Boolean(dump_to='inContext')
    current = fields.Boolean()
    start_departure_time = fields.String(dump_to='startDepartureTime')
    stop_arrival_time = fields.String(dump_to='stopArrivalTime')
    first_departure = fields.String(dump_to='firstDeparture')
    days_text = fields.String(dump_to='daysText')


class ResponseRelatedThreadSchema(Schema):
    canonical_uid = fields.String(dump_to='canonicalUid')
    title = fields.String()
    relation_type = fields.String(dump_to='relationType')


class ResponseSchema(Schema):
    thread = fields.Nested(ResponseThreadSchema)
    threads = fields.Method('make_threads')
    rtstations = fields.Nested(ResponseThreadStationSchema, many=True)
    related_threads = fields.Nested(ResponseRelatedThreadSchema, many=True, dump_to='relatedThreads')

    @classmethod
    def make_threads(cls, obj):
        res = {}
        if 'threads' in obj:
            for t in obj['threads']:
                data, errors = ResponseThreadItemSchema().dump(t)
                res[t.id] = data
        return res
