# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from common.models.geo import Country, Settlement
from common.models.transport import TransportType
from common.utils.settlement import get_main_stations


log = logging.getLogger(__name__)


def make_transport_result(full_data, t_type_code, country_code, language):
    t_type = TransportType.objects.get(code=t_type_code)
    init_data = full_data[t_type_code][country_code]

    result = {'main_city': _make_main_city_result(init_data['main_city'], t_type, language)}
    if 'secondary_city' in init_data:
        result['secondary_city'] = _make_main_city_result(init_data['secondary_city'], t_type, language)

    if 'cities' in init_data:
        cities = []
        for city_slug in init_data['cities']:
            city_result = _make_city_result(city_slug, t_type, language)
            if city_result is not None:
                cities.append(city_result)
        if cities:
            result['cities'] = cities

    result['countries'] = _make_countries(full_data[t_type_code], language)

    return result


def _make_main_city_result(slug, t_type, language):
    city = Settlement.objects.get(slug=slug)
    stations = get_main_stations(city, t_type.id)
    result_stations = [
        {
            'title': station.L_popular_title(language),
            'id': station.id,
            'page_type': station.page_type,
            'main_subtype': station.main_subtype
        }
        for station in sorted(stations, key=lambda s: (s.majority.id, s.L_popular_title(language)))
    ]

    return {
        'slug': city.slug,
        'title': city.L_title(language),
        'title_genitive': city.L_title(lang=language, case='genitive', fallback=False),
        'stations': result_stations
    }


def _make_city_result(slug, t_type, language):
    city = Settlement.objects.get(slug=slug)
    stations = get_main_stations(city, t_type.id)

    if len(stations) == 0:
        return None

    if len(stations) == 1:
        station = stations[0]
        return {
            'title': city.L_title(language),
            'station_id': station.id,
            'page_type': station.page_type,
            'main_subtype': station.main_subtype
        }

    return {
        'title': city.L_title(language),
        'slug': slug
    }


def _make_countries(t_type_data, language):
    countries_data = []
    for key in sorted(t_type_data.keys()):
        country = Country.objects.get(code=key)
        countries_data.append({
            'code': key,
            'title': country.L_title(language),
            'title_genitive': country.L_title(lang=language, case='genitive', fallback=False),
            'title_prepositional': '{} {}'.format(
                country.L_title(lang=language, case='preposition_v_vo_na', fallback=False),
                country.L_title(lang=language, case='locative', fallback=False)
            )
        })
    return countries_data
