# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import os

from common.settings import *  # noqa
from common.settings import COMMON_CONTRIB_APPS, COMMON_INSTALLED_APPS, PROJECT_PATH, XGETTEXT_KEYSETS, TEMPLATES
from common.settings.configuration import BaseConfigurator, Configuration
from common.settings.utils import configure_raven, get_app_version
from travel.rasp.library.python.common23.logging.configure import configure_default_logging

from mapping.settings import *  # noqa
from url_mapper.settings import *  # noqa

from travel.rasp.library.python.common23.logging.qloud import context_extractor


INSTALLED_APPS = COMMON_CONTRIB_APPS + [
    'django_opentracing',
    'django.contrib.staticfiles',
    'raven.contrib.django.raven_compat',
    'rest_framework',
    'rest_framework_swagger',
] + COMMON_INSTALLED_APPS + [
    'route_search',
    'travel.rasp.morda_backend.morda_backend',
    'mapping',
]


MIDDLEWARE_CLASSES = [
    'django_opentracing.OpenTracingMiddleware',
    'travel.library.python.tracing.django.TracingMiddleware',
    'common.middleware.extract_context.ExtractRequestMiddleware',
    'common.middleware.ping.PingResponder',
    'django.middleware.locale.LocaleMiddleware',
    'travel.rasp.morda_backend.morda_backend.middleware.set_locale.SetLocale',
    'travel.rasp.morda_backend.morda_backend.middleware.set_exps_flags.SetExpsFlags'
]

STATICFILES_FINDERS = [
    'django.contrib.staticfiles.finders.FileSystemFinder',
    'django.contrib.staticfiles.finders.AppDirectoriesFinder',
    'library.python.django.contrib.staticfiles.finders.ArcadiaAppFinder'
]

TEMPLATES.append(
    {
        'BACKEND': 'library.python.django.template.backends.arcadia.ArcadiaTemplates',
        'OPTIONS': {
            'debug': False,
            'loaders': [
                'library.python.django.template.loaders.resource.Loader',
                'library.python.django.template.loaders.app_resource.Loader',
            ],
            'context_processors': [
                'django.template.context_processors.debug',
                'django.template.context_processors.request',
                'django.contrib.auth.context_processors.auth',
                'django.contrib.messages.context_processors.messages',
            ],
        },
    }
)

FORM_RENDERER = 'library.python.django.template.backends.forms_renderer.ArcadiaRenderer'


ROOT_URLCONF = 'travel.rasp.morda_backend.morda_backend.urls'
STATIC_ROOT = os.path.join(PROJECT_PATH, 'static')


def with_suffix(suffix, prefixes):
    return [prefix + suffix for prefix in prefixes]


PRECACHE = with_suffix('.objects.precache', [
    'common.models.geo:StationType',
    'common.models.tariffs:TariffType',
    'common.models.transport:TransportType',
    'common.models.transport:TransportSubtype',
    'common.models.schedule:PlatformTranslation',
    'common.models.geo:CodeSystem',
    'common.models.geo:StationTerminal',
    'common.models.geo:WayToAirport',
    # Эти объекты используются в station_schedule
    'common.models.geo:Direction',
    'common.models.geo:DirectionFromTranslate',
    'common.models.geo:ExternalDirection',
    # Эти объекты используются в route_search
    'common.models.schedule:Company',
    'common.models.schedule:Supplier',
    'common.models.schedule:RThreadType',  # для Schedule.L_days_text_dict
    'common.models.geo:StationMajority',
    'common.models.geo:Country',  # нужно для получения timezone ч/з Express2Country
    'common.models.currency:Currency',
    'geosearch.models:DefaultPoint',
]) + [
    'common.importinfo.models:Express2Country.precache',
    'common.models.geo:ReplaceException.precache',
    'common.models.geo:Station.code_manager.precache',
    'mapping.models:RouteMapBlacklist.precache_numbers',
    'mapping.drawers:TrainPathDrawer.storage.preload',
    'mapping.drawers:LimePathDrawer.storage.preload',
]

REST_FRAMEWORK = {
    'EXCEPTION_HANDLER': 'travel.rasp.morda_backend.morda_backend.data_layer.error.rest_framework_exception_handler'
}

OPENTRACING_TRACE_ALL = True
OPENTRACING_TRACED_ATTRIBUTES = ['path', 'method']

ALLOWED_HOSTS = ['*']

# RASPFRONT-1958 используется для запросов к ticketdaemon
PROJECT_CODE = 'rasp_morda_backend'

# Время ожидания ответа поставщика
TARIFF_SOCKET_TIMEOUT = 60 * 3
# Время ожидания заполнения кеша данными от поставщика
TARIFF_SUPPLIERWAIT_TIMEOUT = 60 * 5
UFS_TRAIN_DETAILS_ERROR_TIMEOUT = 60 * 15

DAEMON_API_HTTP_TIMEOUT = 1
DAEMON_MAX_DATES_TO_REQUEST = 2

UFS_MAX_DATES_TO_REQUEST = 2

UKRMINTRANS_TRAIN_DEFAULT_DEPTH_OF_SALES = 45

UFS_RESPONSE_LOG = '/dev/null'
IM_CAR_PRISING_LOG = '/dev/null'
DINAMIC_TARIFFS_LOG = '/dev/null'

XGETTEXT_KEYSETS.update({
    'morda_backend': {
        'filename': 'morda_backend/xgettext/keyset.json',
        'dirs': [
            'morda_backend',
        ]
    }
})

YASMAGENT_ITYPE = 'raspback'
YASMAGENT_PROJECT = 'raspback'

PYLOGCTX_REQUEST_EXTRACTOR = context_extractor

PKG_VERSION = get_app_version()


class RaspBackendConfigurator(BaseConfigurator):
    def apply_base(self, settings):
        settings['RAVEN_CONFIG'] = {
            'ignore_exceptions': [
                'pymongo.errors.NetworkTimeout',
                'MySQLdb.OperationalError',
                'common.data_api.yandex_bus.api.BusConnectionError',
                'common.data_api.yandex_bus.api.BusApiError',
            ]
        }
        settings['LOG_PATH'] = settings.get('LOG_PATH', '/var/log/app')

        settings['CACHEROOT'] = '/yandex/rasp-morda-backend/'  # CACHEROOT переписывается в switch_db
        settings['SAAS_INDEX_HOST'] = 'http://saas-indexerproxy-maps-prestable.yandex.net:80'
        settings['SAAS_SERVICE_KEY'] = 'c478bc8a64643879fdc4c78e676e288d'
        settings['SAAS_SEARCH_HOST'] = 'http://saas-searchproxy-maps-prestable.yandex.net:17000'
        settings['SAAS_SERVICE_NAME'] = 'rasp_wizard'
        settings['MONGO_LOG_COMMANDS'] = True
        settings['MYSQL_LOG_QUERIES'] = True
        settings['TVM_DESTINATIONS'] = ['TVM_TRAVEL_API']
        settings['ENABLE_HOTEL_BANNER'] = True

        for alias in ['default']:
            settings['MONGO_DATABASES'].setdefault(alias, {}).setdefault('options', {}).update({
                'heartbeatFrequencyMS': 3000,
                'connectTimeoutMS': 1000
            })

        self.setup_logging(settings)
        self.setup_yt_logs(settings)

    def apply_production(self, settings):
        configure_raven(settings, 'morda_backend')

        settings['SAAS_INDEX_HOST'] = 'http://saas-indexerproxy-maps-kv.yandex.net:80'
        settings['SAAS_SEARCH_HOST'] = 'http://saas-searchproxy-maps-kv.yandex.net:17000'
        settings['TICKETS_QUERY_URL'] = 'http://daemon.avia.yandex.net/api/1.0/'
        settings['TVM_SERVICE_ID'] = 2033985

    def apply_testing(self, settings):
        configure_raven(settings, 'morda_backend')

        settings['TVM_SERVICE_ID'] = 2033981

    def apply_development(self, settings):
        settings['DEBUG'] = True
        settings['RASP_PROJECT'] = 'morda_backend'
        settings['YANDEX_INTERNAL_ROOT_CA'] = False
        settings['STATIC_URL'] = '/static/'
        settings['TVM_SERVICE_ID'] = 2033981

        settings['LOG_PATH'] = os.path.join(os.path.dirname(__file__), 'log')
        self.setup_logging(settings)
        self.setup_yt_logs(settings)

        self._setup_django_toolbar(settings)

    @classmethod
    def setup_logging(cls, settings, base_log_level=None):
        line_format = '%(asctime)s %(process)d %(levelname)s %(name)s %(message)s %(context)s'
        settings['LOGGING'] = configure_default_logging(settings, base_log_level, line_format, jaeger_tracing=True)

        if settings.get('ENABLE_DURATIONS_LOGGING'):
            cls.setup_durations_logging(settings)

    @classmethod
    def setup_durations_logging(cls, settings):
        log_path = settings['LOG_PATH']

        settings['LOGGING']['loggers'].update({
            'django.db.backends': {
                'handlers': ['mysql_durations'],
                'level': 'DEBUG',
            },
            'common.db.mongo.command_logger': {
                'handlers': ['mongo_durations'],
                'level': 'DEBUG',
            },
        })

        settings['LOGGING']['handlers'].update({
            'mongo_durations': {
                'class': 'travel.rasp.library.python.common23.logging.WatchedFileHandler',
                'filters': ['context_as_dict_filter'],
                'formatter': 'mongo_durations_json',
                'filename': os.path.join(log_path, 'durations.log'),
            },
            'mysql_durations': {
                'class': 'travel.rasp.library.python.common23.logging.WatchedFileHandler',
                'filters': ['context_as_dict_filter'],
                'formatter': 'mysql_durations_json',
                'filename': os.path.join(log_path, 'durations.log'),
            },
            'response_durations': {
                'class': 'travel.rasp.library.python.common23.logging.WatchedFileHandler',
                'filters': ['context_as_dict_filter'],
                'formatter': 'response_durations_json',
                'filename': os.path.join(log_path, 'durations.log'),
            },
        })

        settings['LOGGING']['formatters'].update({
            'mongo_durations_json': {
                '()': 'travel.rasp.library.python.common23.logging.JsonFormatter',
                'keys_to_log': ['name', 'context', 'duration', 'command_name'],
            },
            'mysql_durations_json': {
                '()': 'travel.rasp.library.python.common23.logging.JsonFormatter',
                'keys_to_log': ['name', 'context', 'duration'],
            },
            'response_durations_json': {
                '()': 'travel.rasp.library.python.common23.logging.JsonFormatter',
                'keys_to_log': ['name', 'context', 'duration'],
            },
        })

    @classmethod
    def setup_yt_logs(cls, settings):
        log_path = settings['LOG_PATH']

        settings['DINAMIC_TARIFFS_LOG'] = os.path.join(log_path, 'yt/dynamic_tariffs.log')
        settings['UFS_RESPONSE_LOG'] = os.path.join(log_path, 'yt/ufs_carlistex.log')
        settings['IM_CAR_PRISING_LOG'] = os.path.join(log_path, 'yt/im_carpricing.log')

    @classmethod
    def _setup_django_toolbar(cls, settings):
        if 'debug_toolbar' not in settings['INSTALLED_APPS']:
            settings['INSTALLED_APPS'].extend([
                'debug_toolbar'
            ])
        settings['MIDDLEWARE_CLASSES'] = ['debug_toolbar.middleware.DebugToolbarMiddleware'] + settings['MIDDLEWARE_CLASSES']
        settings['DEBUG_TOOLBAR_PATCH_SETTINGS'] = False
        settings['DEBUG_TOOLBAR_CONFIG'] = {
            'SHOW_TOOLBAR_CALLBACK': lambda _: True
        }

        settings['DEBUG_TOOLBAR_PANELS'] = [
            'debug_toolbar.panels.versions.VersionsPanel',
            'debug_toolbar.panels.timer.TimerPanel',
            'debug_toolbar.panels.settings.SettingsPanel',
            'debug_toolbar.panels.headers.HeadersPanel',
            'debug_toolbar.panels.request.RequestPanel',
            'debug_toolbar.panels.sql.SQLPanel',
            'debug_toolbar.panels.staticfiles.StaticFilesPanel',
            'debug_toolbar.panels.templates.TemplatesPanel',
            'debug_toolbar.panels.cache.CachePanel',
            'debug_toolbar.panels.signals.SignalsPanel',
            'debug_toolbar.panels.logging.LoggingPanel',
            'debug_toolbar.panels.redirects.RedirectsPanel',
            'debug_toolbar.panels.profiling.ProfilingPanel',
        ]


Configuration.configurator_classes.append(RaspBackendConfigurator)

TRUST_COLLECT_METRICS = False

MYSQL_READ_TIMEOUT = 19
