import logging
import os
from contextlib import closing

import MySQLdb

from travel.rasp.mysql_dumper.lib.dumpers import (
    CurrencyDumper,
    RThreadDumper,
    RTStationDumper,
    SettlementDumper,
    Station2SettlementDumper,
    StationDumper,
    SupplierDumper,
    TariffDumper,
)
from travel.rasp.mysql_dumper.lib.loaders import (
    CurrencyLoader,
    MySQLConnector,
    RThreadLoader,
    RTStationLoader,
    SettlementLoader,
    Station2SettlementLoader,
    StationLoader,
    SupplierLoader,
    TariffLoader,
)
from travel.rasp.mysql_dumper.lib.normalizer import initialize


def partial_dump(loader, dumper, filename, left, right):
    logging.info('Getting objects with {} <= id and id < {} from {}...'.format(left, right, loader.TABLE_NAME))
    data = loader.get_range(left, right)
    logging.info('Loading completed. Writing data to file {}'.format(filename))
    with open(filename, 'wb') as output:
        output.write(dumper.dumps(data))


def dump(loader, dumper, directory, objects_in_file):
    objects_in_file = max(objects_in_file, 20000)
    request_parts = (loader.rows + objects_in_file - 1) // objects_in_file
    intervals = loader.split_interval(request_parts)
    for i, interval in enumerate(intervals):
        name = os.path.join(directory, str(i).rjust(5, '0'))
        partial_dump(loader, dumper, name, *interval)


def run_in_one_process(config, file_by_type):
    with closing(MySQLdb.connect(**config)) as connection:
        for name in TYPES_TO_DUMP:
            filename = file_by_type.get(name)
            amount = 10 ** 6
            if not filename:
                print(filename)
                continue
            loader = get_loader(name, connection)
            dumper = get_dumper(name)
            assert len(loader.COLUMNS) == len(dumper.FIELDS)
            dump(loader, dumper, filename, amount)


def run_dump(host, user, password, db, precache_path, file_by_type):
    config = {
        'host': host,
        'user': user,
        'passwd': password,
        'db': db,
        'use_unicode': True,
        'charset': 'utf8'
    }

    initialize(precache_path)
    run_in_one_process(config, file_by_type)


def get_dumper(name):
    return DUMPERS[name]()


def get_loader(name, connection):
    loader = LOADERS[name]
    connector = MySQLConnector(connection, loader.TABLE_NAME, loader.COLUMNS)
    return LOADERS[name](connector)


DUMPERS = {
    'currency': CurrencyDumper,
    'settlement': SettlementDumper,
    'rtstation': RTStationDumper,
    'rthread': RThreadDumper,
    'station': StationDumper,
    'station2settlement': Station2SettlementDumper,
    'threadtariff': TariffDumper,
    'supplier':  SupplierDumper
}

LOADERS = {
    'currency': CurrencyLoader,
    'settlement': SettlementLoader,
    'rtstation': RTStationLoader,
    'rthread': RThreadLoader,
    'station': StationLoader,
    'station2settlement': Station2SettlementLoader,
    'threadtariff':  TariffLoader,
    'supplier': SupplierLoader
}

TYPES_TO_DUMP = [key for key in DUMPERS.keys() if key in LOADERS]
