# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import json

import six
from six.moves import zip

from travel.rasp.mysql_dumper.lib.normalizer import CACHE
from travel.rasp.mysql_dumper.lib.protos.currency_pb2 import TCurrency, TCurrencyPack
from travel.rasp.mysql_dumper.lib.protos.rthread_pb2 import TRThread, TRThreadPack
from travel.rasp.mysql_dumper.lib.protos.rtstation_pb2 import TThreadStation, TThreadStationPack
from travel.rasp.mysql_dumper.lib.protos.settlement_pb2 import TSettlement, TSettlementPack
from travel.rasp.mysql_dumper.lib.protos.station_pb2 import TStation, TStationPack
from travel.rasp.mysql_dumper.lib.protos.station_to_settlement_pb2 import TStation2Settlement, TStation2SettlementPack
from travel.rasp.mysql_dumper.lib.protos.supplier_pb2 import TSupplier, TSupplierPack
from travel.rasp.mysql_dumper.lib.protos.tariff_pb2 import TThreadTariff, TThreadTariffPack


MAX_DAY_PER_MONTH = 31
MONTH_PER_YEAR = 12


class BaseDumper(object):
    FIELDS = []
    _rules = dict()

    def __init__(self, proto_class, pack_class):
        self.proto_class = proto_class
        self.pack_class = pack_class

    def _init_object(self, proto_object, object_info):
        for name, value in zip(self.FIELDS, object_info):
            if value is not None:
                rule = self._rules.get(name)
                if rule is None:
                    setattr(proto_object, name, value)
                else:
                    rule(proto_object, name, value)

    def get_objects_from_data(self, data):
        result = []
        for object_info in data:
            current_object = self.proto_class()
            self._init_object(current_object, object_info)
            result.append(current_object)
        return result

    def dumps(self, data):
        pack_object = self.pack_class()
        pack_object.items.extend(self.get_objects_from_data(data))
        return pack_object.SerializeToString()

    def dumps_proto_objects(self, objects):
        pack_object = self.pack_class()
        pack_object.items.extend(objects)
        return pack_object.SerializeToString()


class SettlementDumper(BaseDumper):
    FIELDS = ['id', 'title', 'title_in', 'title_from', 'title_to', 'title_dat', 'title_ua', 'title_en',
              'title_abbr', 'title_ru_preposition_v_vo_na', 'title_ru_genitive', 'title_ru_accusative',
              'title_ru_locative', 'title_ru', 'title_uk', 'title_tr', 'majority_id', 'country_id',
              'region_id', 'district_id', 'time_zone', 'geo_id', 'disputed_territory', 'slug',
              'longitude', 'latitude', 'is_hidden']

    _rules = {
        'time_zone': lambda proto_object, name, value: setattr(proto_object, name, CACHE.timezone_ids[value]),
        'majority_id': lambda proto_object, name, value:
            setattr(proto_object, name, SettlementDumper._majority_id[value]),
    }

    _majority_id = {
        1: TSettlement.CAPITAL_ID,
        2: TSettlement.REGION_CAPITAL_ID,
        3: TSettlement.POPULATION_MILLION_ID,
        4: TSettlement.COMMON_CITY_ID,
        5: TSettlement.VILLAGE_ID,
        99: TSettlement.MAX_ID,
    }

    def __init__(self):
        super(SettlementDumper, self).__init__(TSettlement, TSettlementPack)


class RTStationDumper(BaseDumper):
    FIELDS = ['id', 'station_id', 'tz_departure', 'tz_arrival', 'time_zone', 'is_searchable_from',
              'is_searchable_to', 'departure_code_sharing', 'arrival_code_sharing', 'thread_id',
              'is_technical_stop', 'terminal_id', 'is_fuzzy', 'in_station_schedule']

    _rules = {
        'time_zone': lambda proto_object, name, value: setattr(proto_object, name, CACHE.timezone_ids[value]),
        'tz_arrival': lambda proto_object, name, value:
            RTStationDumper._parse_event_offset(proto_object, 'arrival', value),
        'tz_departure': lambda proto_object, name, value:
            RTStationDumper._parse_event_offset(proto_object, 'departure', value),
    }

    def __init__(self):
        super(RTStationDumper, self).__init__(TThreadStation, TThreadStationPack)

    @staticmethod
    def _parse_event_offset(proto_object, event_name, value):
        if value is not None:
            setattr(proto_object, 'has_{}'.format(event_name), True)
            setattr(proto_object, 'tz_{}'.format(event_name), value)


class StationDumper(BaseDumper):
    FIELDS = ['id', 'settlement_id', 'majority_id', 'is_airport', 'popular_title', 'popular_title_en',
              'popular_title_gen', 'popular_title_ru', 'popular_title_ru_genitive', 'popular_title_tr',
              'popular_title_uk', 'title', 'title_en', 'title_from', 'title_in', 'title_ru',
              'title_ru_accusative', 'title_ru_genitive', 'title_ru_locative', 'title_ru_preposition_v_vo_na',
              'title_short', 'title_to', 'title_tr', 'title_ua', 'title_uk', 'time_zone', 'is_hidden']

    _majority_id = {
        1: TStation.MAIN_IN_CITY_ID,
        2: TStation.IN_TABLO_ID,
        3: TStation.NOT_IN_TABLO_ID,
        4: TStation.STATION_ID,
        5: TStation.NOT_IN_SEARCH_ID,
        100: TStation.EXPRESS_FAKE_ID,
        1000: TStation.MAX_ID,
    }

    PLANE_ID = 2

    _rules = {
        'majority_id': lambda proto_object, name, value:
            setattr(proto_object, name, StationDumper._majority_id[value]),
        'is_airport': lambda proto_object, name, value:
            setattr(proto_object, name, value == StationDumper.PLANE_ID),
        'time_zone': lambda proto_object, name, value:
            setattr(proto_object, name, CACHE.timezone_ids[value]),
    }

    def __init__(self):
        super(StationDumper, self).__init__(TStation, TStationPack)


class Station2SettlementDumper(BaseDumper):
    FIELDS = ['station_id', 'settlement_id']

    def __init__(self):
        super(Station2SettlementDumper, self).__init__(TStation2Settlement, TStation2SettlementPack)


class TitleDumperError(Exception):
    pass


def _convert_year_days(string):
    result = []
    for i in range(0, MAX_DAY_PER_MONTH * MONTH_PER_YEAR, MAX_DAY_PER_MONTH):
        result.append(int(string[i:i + MAX_DAY_PER_MONTH], 2))
    return result


_transport_type_from_id = {
    1: TRThread.TRAIN,
    2: TRThread.PLANE,
    3: TRThread.BUS,
    6: TRThread.SUBURBAN,
    7: TRThread.HELICOPTER,
    11: TRThread.WATER
}

_currency = {
        'RUR': TCurrency.RUR,
        'USD': TCurrency.USD,
        'EUR': TCurrency.EUR,
        'UAH': TCurrency.UAH,
        'KZT': TCurrency.KZT,
        'BYN': TCurrency.BYN,
        'MDL': TCurrency.MDL,
        'TRY': TCurrency.TRY
    }


class RThreadDumper(BaseDumper):
    FIELDS = ['id', 'year_days', 'tz_start_time', 'type_id', 'title_common', 'number', 't_type_id', 'uid',
              'company_id', 'route_id', 'is_hidden']
    T_TYPE_ID_INDEX = FIELDS.index('t_type_id')

    _title_type = {
        'default': TRThread.TRThreadTitle.DEFAULT,
        'suburban': TRThread.TRThreadTitle.SUBURBAN,
        'mta': TRThread.TRThreadTitle.MTA
    }

    _type_id = {
        1: TRThread.BASIC_ID,
        3: TRThread.THROUGH_TRAIN_ID,
        9: TRThread.CHANGE_ID,
        10: TRThread.CANCEL_ID,
        11: TRThread.ASSIGNMENT_ID,
        12: TRThread.INTERVAL_ID,
        13: TRThread.PSEUDO_BACK_ID,
    }

    _transport_type = {
        'suburban': TRThread.SUBURBAN,
    }

    _is_ring = ['add_ring', 'add_circular', 'add_circular_mta']

    _rules = {
        'year_days': lambda proto_object, name, value:
            getattr(proto_object, name).extend(_convert_year_days(value)),
        'tz_start_time': lambda proto_object, name, value:
            setattr(proto_object, name, value.seconds),
        'type_id': lambda proto_object, name, value:
            setattr(proto_object, name, RThreadDumper._type_id[value]),
        't_type_id': lambda proto_object, name, value:
            setattr(proto_object, name, _transport_type_from_id[value]),
        'title_common': lambda proto_object, name, value:
            RThreadDumper._parse_title_common(proto_object, value)
    }

    def __init__(self):
        super(RThreadDumper, self).__init__(TRThread, TRThreadPack)

    @staticmethod
    def _parse_title_common(proto_object, title_common):
        title_dict = json.loads(title_common)
        title_common = proto_object.title_common
        title_common.type = RThreadDumper._title_type[title_dict['type']]
        if 't_type' in title_dict:
            title_common.t_type = RThreadDumper._transport_type[title_dict['t_type']]
        title_common.is_combined = 'is_combined' in title_dict
        title_common.is_ring = any(x in title_dict for x in RThreadDumper._is_ring)

        for part in title_dict['title_parts']:
            current_part = title_common.title_parts.add()
            if isinstance(part, six.text_type):
                if part[0] == 'c':
                    current_part.settlement = int(part[1:])
                elif part[0] == 's':
                    current_part.station = int(part[1:])
                else:
                    raise TitleDumperError("Unknown title part type: " + part)
            elif isinstance(part, dict):
                current_part.settlement = int(part['settlement'][1:])
                current_part.station = int(part['station'][1:])
            else:
                raise TitleDumperError('Unknown part type' + str(part))

    def get_objects_from_data(self, data):
        filtered_data = filter(lambda object_data: object_data[self.T_TYPE_ID_INDEX] in _transport_type_from_id, data)
        return super(RThreadDumper, self).get_objects_from_data(filtered_data)


class TariffDumper(BaseDumper):
    FIELDS = ['id', 'thread_id', 'station_from_id', 'station_to_id', 'tariff', 'year_days_from',
              'settlement_from_id', 'settlement_to_id', 'currency', 'time_zone_from', 't_type',
              'supplier_id', 'time_from']

    _rules = {
        'thread_id': lambda proto_object, name, value:
            setattr(proto_object, name, CACHE.uid_to_id[value]),
        'year_days_from': lambda proto_object, name, value:
            getattr(proto_object, name).extend(_convert_year_days(value)),
        'time_zone_from': lambda proto_object, name, value:
            setattr(proto_object, name, CACHE.timezone_ids[value]),
        't_type_id': lambda proto_object, name, value:
            setattr(proto_object, name, _transport_type_from_id[value]),
        'time_from': lambda proto_object, name, value:
            setattr(proto_object, name, value.seconds),
        'currency': lambda proto_object, name, value:
            setattr(proto_object, name, _currency[value])
    }

    def __init__(self):
        super(TariffDumper, self).__init__(TThreadTariff, TThreadTariffPack)


class SupplierDumper(BaseDumper):
    FIELDS = ['id', 'title', 'code', 'can_buy_ru', 'can_buy_ua', 'can_buy_tr', 'sale_url_template',
              'sale_start_days', 'sale_stop_hours']

    def __init__(self):
        super(SupplierDumper, self).__init__(TSupplier, TSupplierPack)


class CurrencyDumper(BaseDumper):
    FIELDS = ['id', 'code']

    _rules = {
        'code': lambda proto_object, name, value: setattr(proto_object, name, _currency[value])
    }

    def __init__(self):
        super(CurrencyDumper, self).__init__(TCurrency, TCurrencyPack)
