# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from contextlib import closing

from six.moves import range


class MySQLConnector(object):
    def __init__(self, connection, table_name, columns):
        self.connection = connection
        self.table_name = table_name
        self.columns = columns
        self.cursor = connection.cursor()

    def get_min_id(self):
        self.cursor.execute('select id from {} order by id asc limit 1'.format(self.table_name))
        return int(self.cursor.fetchone()[0])

    def get_max_id(self):
        self.cursor.execute('select id from {} order by id desc limit 1'.format(self.table_name))
        return int(self.cursor.fetchone()[0])

    def get_count(self, min_id, max_id):
        self.cursor.execute('select count(1) from {} where id >= %s and id < %s'.format(self.table_name),
                            (min_id, max_id))
        return self.cursor.fetchone()[0]

    def get_objects_in_range(self, min_id, max_id):
        query = 'select {} from {} where id >= %s and id < %s'.format(', '.join(self.columns), self.table_name)
        with closing(self.connection.cursor()) as cursor:
            cursor.execute(query, (min_id, max_id))
            data = cursor.fetchall()
        return data


class BaseDBLoader(object):
    COLUMNS = []
    TABLE_NAME = ''

    def __init__(self, connector):
        self.connector = connector
        self._init_range()

    def _init_range(self):
        self.min_id = self.connector.get_min_id()
        self.max_id = self.connector.get_max_id()
        self.rows = self.connector.get_count(self.min_id, self.max_id + 1)

    def get_all(self):
        return self.get_range(self.min_id, self.max_id + 1)

    def find_right_border(self, start, count):
        left = start
        right = self.max_id + 1
        while left + 1 < right:
            mid = (left + right) // 2
            cur_count = self.connector.get_count(start, mid)
            if cur_count >= count:
                right = mid
            else:
                left = mid
        return right

    def split_interval(self, parts_amount):
        ranges = []
        start = self.min_id
        one_part = self.rows // parts_amount
        remainder = self.rows % parts_amount
        for i in range(parts_amount):
            this_part_rows = one_part
            if remainder:
                remainder -= 1
                this_part_rows += 1
            end = self.find_right_border(start, this_part_rows)
            ranges.append((start, end))
            start = end
        return ranges

    def get_range(self, min_id, max_id):
        return self.connector.get_objects_in_range(min_id, max_id)


class SettlementLoader(BaseDBLoader):
    COLUMNS = ['id', 'title', 'title_in', 'title_from', 'title_to', 'title_dat', 'title_ua', 'title_en',
               'title_abbr', 'title_ru_preposition_v_vo_na', 'title_ru_genitive', 'title_ru_accusative',
               'title_ru_locative', 'title_ru', 'title_uk', 'title_tr', 'majority_id', 'country_id',
               'region_id', 'district_id', 'time_zone', '_geo_id', 'disputed_territory', 'slug',
               'longitude', 'latitude', 'hidden']

    TABLE_NAME = 'www_settlement'

    def __init__(self, connector):
        super(SettlementLoader, self).__init__(connector)


class Station2SettlementLoader(BaseDBLoader):
    COLUMNS = ['station_id', 'settlement_id']

    TABLE_NAME = 'www_station2settlement'

    def __init__(self, connector):
        super(Station2SettlementLoader, self).__init__(connector)


class RTStationLoader(BaseDBLoader):
    COLUMNS = ['id', 'station_id', 'tz_departure', 'tz_arrival', 'time_zone', 'is_searchable_from',
               'is_searchable_to', 'departure_code_sharing', 'arrival_code_sharing', 'thread_id',
               'is_technical_stop', 'terminal_id', 'is_fuzzy', 'in_station_schedule']

    TABLE_NAME = 'www_rtstation'

    def __init__(self, connector):
        super(RTStationLoader, self).__init__(connector)


class StationLoader(BaseDBLoader):
    COLUMNS = ['id', 'settlement_id', 'majority_id', 't_type_id', 'popular_title', 'popular_title_en',
               'popular_title_gen', 'popular_title_ru', 'popular_title_ru_genitive', 'popular_title_tr',
               'popular_title_uk', 'title', 'title_en', 'title_from', 'title_in', 'title_ru', 'title_ru_accusative',
               'title_ru_genitive', 'title_ru_locative', 'title_ru_preposition_v_vo_na', 'title_short',
               'title_to', 'title_tr', 'title_ua', 'title_uk', 'time_zone', 'hidden']

    TABLE_NAME = 'www_station'

    def __init__(self, connector):
        super(StationLoader, self).__init__(connector)


class RThreadLoader(BaseDBLoader):
    COLUMNS = ['id', 'tz_year_days', 'tz_start_time', 'type_id', 'title_common', 'route_number', 't_type_id', 'uid',
               'company_id', 'route_id', 'hidden']

    TABLE_NAME = 'www_rthread'

    def __init__(self, connector):
        super(RThreadLoader, self).__init__(connector)


class TariffLoader(BaseDBLoader):
    COLUMNS = ['id', 'thread_uid', 'station_from_id', 'station_to_id', 'tariff', 'year_days_from',
               'settlement_from_id', 'settlement_to_id', 'currency', 'time_zone_from', 't_type_id',
               'supplier_id', 'time_from']

    TABLE_NAME = 'www_threadtariff'

    def __init__(self, connector):
        super(TariffLoader, self).__init__(connector)


class SupplierLoader(BaseDBLoader):
    COLUMNS = ['id', 'title', 'code', 'can_buy_ru', 'can_buy_ua', 'can_buy_tr', 'sale_url_template',
               'sale_start_days', 'sale_stop_hours']

    TABLE_NAME = 'www_supplier'

    def __init__(self, connector):
        super(SupplierLoader, self).__init__(connector)


class CurrencyLoader(BaseDBLoader):
    COLUMNS = ['id', 'code']

    TABLE_NAME = 'currency_currency'

    def __init__(self, connector):
        super(CurrencyLoader, self).__init__(connector)
