#!/usr/bin/python
# coding: utf8

"""
Скрипт для генерации файла конфигурации 3proxy

Пример получаемого конфига при дефолтных параметрах:

# Generated config
auth strong
users rasp:CL:raspp

log
proxy -46 -p80 -i2a02:6b8:c15:2a93:0:4111:557e:0 -e192.168.1.2
ftppr -46 -p8021 -i2a02:6b8:c15:2a93:0:4111:557e:0 -e192.168.1.2

"""

import os
import socket


def _generate_services(service, num, port, internal_ip, external_ip):
    services = []
    for i in range(num):
        services.append('{service} -46 -p{port} {internal_ip} {external_ip}'.format(
            service=service,
            port=port,
            internal_ip=internal_ip,
            external_ip=external_ip,
        ))
    return '\n'.join(services)


def get_external_ipv4():
    ipv4 = ''
    try:
        with socket.socket(socket.AF_INET, socket.SOCK_DGRAM) as s:
            s.connect(('77.88.8.8', 80))
            ipv4 = s.getsockname()[0]
    except:
        pass

    return ipv4


def main():
    internal_ipv6 = os.getenv('DEPLOY_POD_IP_0_ADDRESS', '')
    external_ipv4 = get_external_ipv4()

    with open(os.getenv('RASP_PROXY_CONFIG_FILE'), 'w') as f:
        f.write('# Generated config\n')
        f.write('auth strong\n')
        f.write('users {login}:CL:{password}\n'.format(
            login=os.getenv('RASP_PROXY_LOGIN', 'rasp'),
            password=os.getenv('RASP_PROXY_PASSWORD', 'raspp'),
        ))
        f.write('\nlog\n')
        f.write(_generate_services(
            service='proxy',
            num=int(os.getenv('RASP_PROXY_SERVICE_THREADS', 1)),
            port='80',
            internal_ip='-i{}'.format(internal_ipv6),
            external_ip='-e{}'.format(external_ipv4) if external_ipv4 else '',
        ))
        f.write('\n')
        f.write(_generate_services(
            service='ftppr',
            num=int(os.getenv('RASP_FTP_PROXY_SERVICE_THREADS', 1)),
            port=os.getenv('RASP_FTP_PROXY_PORT', '8021'),
            internal_ip='-i{}'.format(internal_ipv6),
            external_ip='-e{}'.format(external_ipv4) if external_ipv4 else '',
        ))
        f.write('\n')


if __name__ == '__main__':
    main()
