#if !defined(CONDITION_H)
#define CONDITION_H

#include "path.h"

namespace Pathfinder
{
    // хранит начальные условия для поиска
    class Condition
    {
    public:
        // начальная станция (идентификатор)
        StationInfoPtrs _from;
        const StationInfo* _fromStation;
        // конечная  станция (идентификатор)
        StationInfoPtrs _to;
        const StationInfo* _toStation;
        bool _toStationIsTown;
        // списки городов отправления и назначения
        StationInfoPtrs _endingTowns;

        Geom::Point2d getLocation(const StationInfoPtrs& list) const
        {
                StationInfoPtrs::const_iterator it = list.begin();
                for (; it != list.end() && ((Geom::Point2d)(*it)->location).IsZero(); ++it);
                if (it != list.end())
                    return (*it)->location;
                return Geom::Point2d();
        }

        double getDist()const
        {
            if (_fromLoc.IsZero() || _toLoc.IsZero())
                return -1.0;
            return ((Geom::Point2d)(_fromLoc-_toLoc)).Len();
        }
        double getSphereDist()const
        {
            if (_fromLoc.IsZero() || _toLoc.IsZero())
                return -1.0;
            return Geom::SphereLen(_fromLoc, _toLoc);
        }


        std::vector<int>	_minDelayOut;
        std::vector<int>	_minDelayIn;
        std::vector<int>	_minDelayInAfterTeleportation;

        //для гортарнса
        bool _canChangeInStartTown;
        bool _canChangeInFinishTown;
        bool _canChangeInAnyTown;
        bool _canTeleport;
        bool  _usePrice;
    private:
        // время старта (в минутах)
        int    _time;
        // время ожидания (в минутах; время на "нулевую пересадку", на практике это время равно суткам)
        int    _boarding;
        // минимальное время на пересадку (в минутах) 
        int    _minDelay;
        // максимальное время на пересадку (в минутах)
        int    _maxDelay;
        // допустимые типы транспорта (идентификатор)
        Transport::Type  _transports;
        int _changes;

    public:
        // конструктор по умолчанию
        Condition():
            _canChangeInStartTown(false), _canChangeInFinishTown(false), _canChangeInAnyTown(false), _canTeleport(true)
            , _usePrice(false), _time(-1), _boarding(-1), _minDelay(-1), _maxDelay(-1), _changes(MAX_CHANGES)
        { _transports = Transport::NONE; }
        // конструктор
        Condition(const int & time, const int & boarding, const int & minDelay, const int & maxDelay
                , const Transport::Type& transports, const bool iLoveSport
                , const bool canChangeInStartTown, const bool canChangeInFinishTown, const bool canChangeInAnyTown
                , const bool ct, const bool usePrice, const int changes)
            : _canChangeInStartTown(canChangeInStartTown)
                , _canChangeInFinishTown(canChangeInFinishTown), _canChangeInAnyTown(canChangeInAnyTown)
                , _canTeleport(ct), _changes(changes)
            { init(time, boarding, minDelay, maxDelay, transports, iLoveSport, usePrice); }
        // деструктор
        ~Condition(){}

    public:
        // инициализация
        void init(const int & time, const int & boarding, const int & minDelay, const int & maxDelay, const Transport::Type& transports, const bool iLoveSport, const bool usePrice)
        {
            _time = time;
            _boarding = boarding;
            _minDelay = minDelay;
            _maxDelay = maxDelay;
            _transports = transports;
            _usePrice = usePrice;

            if (iLoveSport)
            {
                Transport::setDefaultMinDelay_i_love_sport(_minDelayOut);
                Transport::setDefaultMinDelay_i_love_sport(_minDelayIn);
                Transport::setDefaultMinDelay_i_love_sport(_minDelayInAfterTeleportation);
            }
            else
            {
                Transport::setDefaultMinDelay(_minDelayOut, 0);
                Transport::setDefaultMinDelay(_minDelayIn, 1);
                Transport::setDefaultMinDelay(_minDelayInAfterTeleportation, 2);
            }
        }

    private:
        mutable Geom::Point2d _fromLoc, _toLoc;
    public:
        void initLocs() const
        {
            _fromLoc = getLocation(_from);
            _toLoc = getLocation(_to);
        }
        //inline int maxDaysForStation(const StationInfo* si) const
        //{
        //	Geom::Point2d loc = si->location;
        //	if (_fromLoc.IsZero() || loc.IsZero())
        //		return 10;
        //	double dist = ((Geom::Point2d)(_fromLoc-loc)).Len();
        //	int tripLong = (int)(dist/(40*24)+1.5);
        //	return tripLong;
        //}

    public:
        // методы доступа:
        inline const StationInfoPtrs& getFrom() const { return _from; }
        inline const StationInfoPtrs& getTo() const { return _to; }
        inline int getTime() const { return _time; }
        inline int getBoarding() const { return _boarding; }
        int getMinDelay() const { return _minDelay; }
        int getMaxDelay() const { return _maxDelay; }
        Transport::Type getTransports() const { return _transports; }
        inline int getEndTime() const
        {
#ifdef FOR_WIZARD
            return getTime() + getBoarding();
#else
            return getTime() + getBoarding() + MINUTES_IN_DAY/2;
#endif
        }
        inline int begFirstDay() const
        {
            return getTime() / MINUTES_IN_DAY;
        }
        inline int begLastDay() const
        {
            return (getTime() + getBoarding()) / MINUTES_IN_DAY;
        }
        inline int endTripDay(int tripLong = 10) const
        {
            return begLastDay() + tripLong;
        }

        inline int getChanges() const { return _changes;}
        inline void setChanges(const int changes) { _changes = changes;}

    public:
        // в строку
        std::string toString() const
        {
            std::ostringstream ost;
            ost << "from=";
            for (StationInfoPtrs::const_iterator it = _from.begin(); it != _from.end(); ++it)
                    ost << (*it)->id << ',';
            ost  << " to=";
            for (StationInfoPtrs::const_iterator it = _to.begin(); it != _to.end(); ++it)
                    ost << (*it)->id << ',';
            ost << " _time=" << _time;
            ost << " boarding=" << _boarding << " minDelay=" << _minDelay << " maxDelay=" << _maxDelay;
            ost << " transports=" << _transports;
            return ost.str();
        }
    };
}
#endif

