#ifndef CurveH
#define CurveH

#include <iostream>
#include <fstream>
#include <iomanip>
#include <sstream>
#include <queue>
#include <stdio.h>

#include "geom.h"
#include "priority_queue.h"
#include "transport.h"

#define mmEps 1e-12
#define mmEps2 1e-24
#define mEps 1e-12
#define mEps3 1e-36

namespace Pathfinder
{
    struct StationInfo;
    class StationTown;
    typedef std::vector<StationInfo> StationInfos;
    namespace Transport
    {
        enum Type;
    }
}

namespace Geom
{
    struct Curve;
    struct CurveMapInfo;

    struct CurvePoint
    {
        Point2d point;
        Curve* curve;
        int n;
        CurvePoint(Point2d p = Point2d(0), Curve* c = 0, int i = 0)
                : point(p), curve(c), n(i)
        {}
        inline bool operator<(const CurvePoint& cp) const
        {
            return (point < cp.point);
        }
    };
    typedef std::vector<CurvePoint> CurvePoints;
    struct Point2CurvePoint
    {
        Point2d from;
        CurvePoint to;

        Point2CurvePoint()
        {}
        Point2CurvePoint(Point2d p, const CurvePoint& cp)
                : from(p), to(cp)
        {}

        inline bool operator<(const Point2CurvePoint& cp) const
        {
            return (from < cp.from);
        }
    };
    typedef std::vector<Point2CurvePoint> Point2CurvePoints;

    struct Curve
    {
        int id;
        Points2d points;
        std::vector<int> bestPoint;
        Point2d lb;
        Point2d rt;
        double coef;
        Point2d maplb;
        Point2d maprt;
        Point2CurvePoints neigCurves;
        mutable bool used;
        Curve()
                : id(0)
                , lb(1e+30)
                , rt(-1e+30)
                , coef(-1.0)
                , maplb(1e+30)
                , maprt(-1e+30)
                , used(false)
        {}

        int simplifi();

        void setCoef(const Point2d mlb, const Point2d mrt)
        {
            maplb = mlb;
            maprt = mrt;
            coef = (mlb-mrt).LenAbs();
        }

        bool inThisMap(const Point2d a, const double radius) const
        {
            if (a.x >= maplb.x-radius && a.x <= maprt.x+radius)
            {
                if (a.y >= maplb.y-radius && a.y <= maprt.y+radius)
                    return true;
            }
            return false;
        }

        int setBestWithMap(const CurveMapInfo& map);

        int getNearestPoint(const Point2d point)
        {
            size_t n = points.size();
            int res = 0;
            double resLen = 1e+30;
            for (int i = 0; i < n; i++)
            {
                double len = (points[i] - point).Len();
                if (len < resLen)
                {
                    res = i;
                    resLen = len;
                }
            }
            return res;
        }

        double getLen()
        {
            size_t n = points.size();
            double res = 0;
            for (int i = 1; i < n; i++)
            {
                double len = (points[i-1] - points[i]).Len();
                res += len;
            }
            return res;
        }

        Point2d getNearestSeg(const Point2d point, bool& resIsNewPoint, bool& resIsEndingPoint)
        {
            size_t n = points.size();
            Point2d res;
            int resi = 0;
            double resLen = 1e+30;
            resIsNewPoint = false;
            resIsEndingPoint = false;
            for (int i = 0; i < n; i++)
            {
                double len = (points[i] - point).Len();
                if (len < resLen)
                {
                    res = points[i];
                    resLen = len;
                    resi = i;
                    resIsNewPoint = false;
                    if (i == 0 || i == n-1)
                        resIsEndingPoint = true;
                    else
                        resIsEndingPoint = false;
                }
                if (i>0) //проверяем отрезок
                {
                    Point2d a = points[i-1];
                    Point2d b = points[i];
                    double ablen = (a-b).Len();
                    if (ablen < 0.00001)
                        continue;
                    double vecMult = (point - a) ^ (b - a);
                    double hlen = (1.0 / ablen) * vecMult;
                    Point2d h(b.y-a.y, a.x-b.x);
                    h = (hlen/ablen) * h;
                    Point2d c = point - h;
                    hlen = std::abs(hlen);

                    if (hlen < resLen && std::abs((a-c).Len()+(b-c).Len()-ablen) < 0.0001 && std::abs(hlen-(point-c).Len()) < 0.0001)
                    {
                        res = c;
                        resLen = hlen;
                        resi = i;
                        resIsNewPoint = true;
                        resIsEndingPoint = false;
                    }
                    c = point - h;
                    if (hlen < resLen && std::abs((a-c).Len()+(b-c).Len()-ablen) < 0.0001 && std::abs(hlen-(point-c).Len()) < 0.0001)
                    {
                        res = c;
                        resLen = hlen;
                        resi = i;
                        resIsNewPoint = true;
                        resIsEndingPoint = false;
                    }
                }
            }
            if (resIsNewPoint)
                points.insert(points.begin() + resi, res);
            return res;
        }

        int	findPoint(const Point2d point) const
        {
            size_t n = points.size();
            int i;
            for (i = 0; i < n && points[i] != point; i++);
            if (i < n)
                return i;
            return -1;
        }
        int	findPointEps(const Point2d point, double eps) const
        {
            size_t n = points.size();
            int i;
            for (i = 0; i < n && (points[i]-point).Len() > eps; i++);
            if (i < n)
                return i;
            return -1;
        }
        void updateBox(const Point2d a)
        {
            lb = MinCoord(lb, a);
            rt = MaxCoord(rt, a);
        }
        void updateBox()
        {
            size_t n = points.size();
            for (int i = 0; i < n; i++)
                updateBox(points[i]);
        }
        bool inBox(const Point2d a, const double radius)
        {
            if (a.x >= lb.x-radius && a.x <= rt.x+radius)
            {
                if (a.y >= lb.y-radius && a.y <= rt.y+radius)
                    return true;
            }
            return false;
        }

        void savePolish(std::ostream& out, const char* color, const char* label = 0) const
        {
            out << "[POLYLINE]\nType=" << color << "\nData0=";
            savePolishPointsList(out);
            if (label)
                out << "\nLabel=" << label;
            out << "\n[END]\n" << std::endl;
        }

        size_t savePolishPointsList(std::ostream& out) const
        {
            size_t res = 0;
            for (Points2d::const_iterator it = points.begin(); it != points.end(); ++it)
            {
                if (res)
                    out << ",";
                out << "("
                << it->x << "," << it->y
                << ")";
                res++;
            }
            return res;
        }

        size_t savePointsList(std::ostream& out) const
        {
            size_t res = 0;
            for (Points2d::const_iterator it = points.begin(); it != points.end(); ++it)
            {
                if (res)
                    out << " ";
                out << it->x << " " << it->y;
                res++;
            }
            return res;
        }

        bool glueWith(Curve& curve, double eps)
        {
            if ((points.back()-curve.points[0]).Len() < eps)
            {
                points.insert(points.end(), curve.points.begin(), curve.points.end());
                return true;
            }
            if ((points[0]-curve.points.back()).Len() < eps)
            {
                points.insert(points.begin(), curve.points.begin(), curve.points.end());
                return true;
            }
            std::reverse(points.begin(), points.end());
            if ((points.back()-curve.points[0]).Len() < eps)
            {
                points.insert(points.end(), curve.points.begin(), curve.points.end());
                return true;
            }
            if ((points[0]-curve.points.back()).Len() < eps)
            {
                points.insert(points.begin(), curve.points.begin(), curve.points.end());
                return true;
            }
            return false;
        }
    };

    struct NearCurve
    {
        const Curve* curve;
        std::vector<const Curve*> neigCurves;
        Point2d point;
        double dist;
        bool newP;
        bool endingP;
        bool parentLine;

        inline bool operator<(const NearCurve& nc) const
        {
            //double d1 = dist;
            //double d2 = nc.dist;
            if ((dist < 1e-3 && nc.dist>dist*25) || (nc.dist <1e-3 && dist>nc.dist*25))
                return dist < nc.dist;
            const bool map1 = curve->inThisMap(point, 0);
            const bool map2 = nc.curve->inThisMap(nc.point, 0);
            if (map1 != map2)
                return map1 > map2;
            if (map1 && map2)
            if (curve->coef != nc.curve->coef)
                return curve->coef < nc.curve->coef;
            return dist < nc.dist;
        }
    };
    inline bool lessNearCurve(const NearCurve& nc1, const NearCurve& nc2)
    {
        return nc1.dist < nc2.dist;
    }

    typedef std::vector<NearCurve> NearCurves;

    typedef std::vector<Curve> Curves;
    typedef std::map<Point2d, const ::Pathfinder::StationInfo*> Point2StationMap;
    struct CurveMapInfo
    {
        std::string name;
        Point2d lb;
        Point2d rt;
        CurveMapInfo()
                : lb(1e+30)
                , rt(-1e+30)
        {}
        CurveMapInfo(const std::string& name0, const Point2d& lb0, const Point2d& rt0)
                : name(name0), lb(lb0), rt(rt0)
        {}
        double coef() const
        {
            return (lb-rt).LenAbs();
        }
        inline bool operator<(const CurveMapInfo& a) const
        {
            return coef() < a.coef();
        }

        void updateBox(const Point2d a)
        {
            lb = MinCoord(lb, a);
            rt = MaxCoord(rt, a);
        }
        bool inThisMap(const Point2d a, const double radius=0) const
        {
            if (a.x >= lb.x-radius && a.x <= rt.x+radius)
            {
                if (a.y >= lb.y-radius && a.y <= rt.y+radius)
                    return true;
            }
            return false;
        }

        bool intersectWith(const CurveMapInfo& map) const
        {
            if (rt.x >= map.lb.x && lb.x <= map.rt.x)
            if (rt.y >= map.lb.y && lb.y <= map.rt.y)
                return true;
            return false;
        }
    };

    struct CurveMap
    {
        Curves curves;
        Point2d lb;
        Point2d rt;
        Point2StationMap point2StationMap;
        std::vector<CurveMapInfo> allCoefs;

        CurveMap()
                : lb(1e+30)
                , rt(-1e+30)
        {}

        void clear()
        {
            curves.clear();
            lb = Point2d(1e+30);
            rt = Point2d(-1e+30);
        }

        void updateBox(const Point2d a)
        {
            lb = MinCoord(lb, a);
            rt = MaxCoord(rt, a);
        }
        void updateBox()
        {
            size_t n = curves.size();
            for (int i = 0; i < n; i++)
            {
                updateBox(curves[i].lb);
                updateBox(curves[i].rt);
            }
        }

        bool sameDir(Point2d a, Point2d b, Point2d c, Point2d d, double eps)
        {
            if ((b-c).Len()<eps || (a-d).Len()<eps)
                return ((b-a) & (d-c)) >= 0;
            if ((a-c).Len()<eps || (b-d).Len()<eps)
                return ((b-a) & (d-c)) <= 0;
            return false;
        }

        size_t loadMIF(std::istream& in)
        {
            size_t res = 0;
            std::string s;
            while (std::getline(in, s))
            {
                std::istringstream iss(s);
                std::string key;
                iss >> key;
                if (key == "Pline")
                {
                    Curve curve;
                    int nlines;
                    iss >> nlines;
                    Point2d a;
                    for (int i = 0; i < nlines; i++)
                    {
                        in >> a.y >> a.x;
                        curve.points.push_back(a);
                        curve.updateBox(a);
                        res++;
                    }
                    curves.push_back(curve);
                }
                else if (key == "Line")
                {
                    Curve curve;
                    Point2d a;
                    for (int i = 0; i < 2; i++)
                    {
                        iss >> a.y >> a.x;
                        curve.points.push_back(a);
                        curve.updateBox(a);
                        res++;
                    }
                    curves.push_back(curve);
                }
            }
            updateBox();
            return res;
        }

        size_t loadMIDMIF(std::istream& mid, std::istream& mif, std::istream& for_delete_in)
        {
            std::string s;
            std::set< std::string > rwtype_fordelete;
            while (std::getline(for_delete_in, s))
                rwtype_fordelete.insert(s);

            size_t res = 0;
            while (std::getline(mif, s))
            {
                std::istringstream iss(s);
                std::string key;
                iss >> key;
                if (key == "Pline" || key == "Line" || key == "PLINE")
                {
                    std::string mids;
                    char mapname[100], rwtype[100];
                    std::getline(mid, mids);
                    sscanf(mids.c_str(), "%[^\t] %[^\n]", mapname, rwtype);
                    bool ifdelete = (rwtype_fordelete.find(rwtype) != rwtype_fordelete.end());

                    std::string temps;
                    int ncurves = 1;
                    int nlines = 2;
                    bool isLine = false;
                    if (key == "Pline")
                        iss >> nlines;
                    else if (key == "Line")
                        isLine = true;
                    else
                        iss >> temps >> ncurves;
                    for (int icurve = 0; icurve < ncurves; icurve++)
                    {
                        Curve curve;
                        std::set<Point2d> used;
                        if (!temps.empty())
                        {
                            mif >> nlines;
                        }
                        for (int i = 0; i < nlines; i++)
                        {
                            Point2d a;
                            if (isLine)
                                iss >> a.y >> a.x;
                            else
                                mif >> a.y >> a.x;
                            if (used.find(a) == used.end())
                            {
                                curve.points.push_back(a);
                                curve.updateBox(a);
                                used.insert(a);
                            }
                        }
                        if (!ifdelete && !curve.points.empty())
                        {
                            if (allCoefs.empty())
                                allCoefs.push_back(CurveMapInfo(mapname, curve.lb, curve.rt));
                            else if (allCoefs.back().name != mapname)
                            {
                                int tailBeg;
                                for (tailBeg = (int)curves.size()-1; tailBeg>=0 && curves[tailBeg].coef<0; tailBeg--);
                                tailBeg++;
                                glueCurves(tailBeg);
                                //curves.tailCoef(allCoefs.back().lb, allCoefs.back().rt);
                                for (int i = (int)curves.size()-1; i>=0 && curves[i].coef<0; i--)
                                    curves[i].setCoef(allCoefs.back().lb, allCoefs.back().rt);
                                allCoefs.push_back(CurveMapInfo(mapname, curve.lb, curve.rt));
                            }
                            else
                            {
                                allCoefs.back().updateBox(curve.lb);
                                allCoefs.back().updateBox(curve.rt);
                            }
                            curves.push_back(curve);
                            res++;
                        }
                    }
                }
            }
            for (int i = (int)curves.size()-1; i>=0 && curves[i].coef<0; i--)
                curves[i].setCoef(allCoefs.back().lb, allCoefs.back().rt);
            updateBox();
            return res;
        }


        size_t load(std::istream& in)
        {
            size_t res = 0;
            std::string s;
            while (std::getline(in, s))
            {
                Curve curve;
                int n;
                in >> n;
                std::set<Point2d> used;
                for (int i = 0; i < n; i++)
                {
                    Point2d a;
                    in >> a.x >> a.y;
                    if (used.find(a) == used.end())
                    {
                        curve.points.push_back(a);
                        used.insert(a);
                    }
                    curve.updateBox(a);
                    res++;
                }
                std::getline(in, s);
                curves.push_back(curve);
            }
            updateBox();
            return res;
        }

        void setCurveCoefs()
        {
            for (Curves::iterator it = curves.begin(); it != curves.end(); ++it)
                it->setCoef(lb, rt);
        }

        int markBestPoints()
        {
            int res = 0;
            for (size_t i = 0; i < curves.size(); i++)
                for (size_t j = 0; j < allCoefs.size(); j++)
                {
                    res = std::max(res, curves[i].setBestWithMap(allCoefs[j]));
                }
            return res;
        }

        void merge(CurveMap& map)
        {
            map.setCurveCoefs();
            for (size_t i = 0; i < map.curves.size(); i++)
            {
                curves.push_back(map.curves[i]);
            }
            updateBox();
        }

        void addMap(const char* fileName, const double eps = 0.002)
        {
            std::ifstream fin1(fileName);
            Geom::CurveMap curveMapTemp;
            curveMapTemp.load(fin1);
            curveMapTemp.glueCurves(0, eps);

            merge(curveMapTemp);

            if (!curveMapTemp.curves.empty())
                allCoefs.push_back(CurveMapInfo(fileName, curveMapTemp.lb, curveMapTemp.rt));
            std::sort(allCoefs.begin(), allCoefs.end());
        }

        void getNearCurves(const std::vector<const Curve*>& neigCurves, const Point2d a, const double radius, const double coefNeig, NearCurves& nearCurves)
        {
            nearCurves.clear();
            int i = 0;
            for (Curves::iterator it = curves.begin(); it != curves.end(); ++it, i++)
            {
                //if (i == 1085)
                //	int oh = 1;
                if (it->inBox(a, radius))
                {
                    NearCurve nearCurve;
                    nearCurve.curve = &(*it);
                    nearCurve.point = it->getNearestSeg(a, nearCurve.newP, nearCurve.endingP);
                    nearCurve.dist = (nearCurve.point - a).Len();

                    size_t i;
                    for (i = 0; i < neigCurves.size() && neigCurves[i] != nearCurve.curve; i++);
                    if (i < neigCurves.size())
                    {
                        nearCurve.dist /= coefNeig;
                        nearCurve.parentLine = true;
                    }
                    else
                        nearCurve.parentLine = false;
                    nearCurve.neigCurves = neigCurves;

                    if (nearCurve.dist <= radius)
                        nearCurves.push_back(nearCurve);
                }
            }
        }

        void savePolish(std::ostream& out, const char* color)
        {
            char buf[1000];
            int i = 0;
            for (Curves::iterator it = curves.begin(); it != curves.end(); ++it, i++)
            {
                sprintf(buf, "%d %lf", it->id, it->coef);
                it->savePolish(out, color, buf);
            }
        }

        void updateBoxes()
        {
            for (Curves::iterator it = curves.begin(); it != curves.end(); ++it)
                it->updateBox();
        }

        int glueCurves(int beg = 0, double eps = 0.001)
        {
            size_t n = curves.size();
            for (int i = beg; i < n; i++)
            {
                size_t m = curves[i].points.size();
                if (m >= 2)
                {
                    bool glued = false;
                    for (int j = i+1; j < n && !glued; j++)
                    {
                        size_t k = curves[j].points.size();
                        if (k >= 2)
                        {
                            if (sameDir(curves[i].points[0], curves[i].points.back(), curves[j].points[0], curves[j].points.back(), eps))
                            {
                                glued = curves[j].glueWith(curves[i], eps);
                                if (glued)
                                    curves[i].points.clear();
                            }
                        }
                    }
                }
            }
            int j = beg;
            for (int i = beg; i < n; i++)
                if (curves[i].points.size() > 1)
                {
                    curves[j] = curves[i];
                    curves[j].updateBox();
                    j++;
                }
            curves.resize(j);
            return (int)n - j;
        }

        size_t setStationsOnCurves(std::ostream& out, std::ostream& log, ::Pathfinder::StationInfos& si, const ::Pathfinder::Transport::Type& trFilter, const double eps = 0.1, const double coefNeig = 1.0, const bool paint = true, const bool do_log = true);
        int setStationsOnCurves_correction(std::istream& in, std::ostream& out, std::ostream& log, ::Pathfinder::StationTown& st);
        size_t createPoint2StationMap(::Pathfinder::StationInfos& si, const ::Pathfinder::Transport::Type& trFilter);

//------------------------------------------------------------------------------------------------------------

        /**
        Создаем списки соседних точек на других кривых для каждой точки каждой кривой.
        Для этого все точки всех кривых сортируем в лексикографическом порядке. Сравниваем соседние.
        **/
        void createCurveNeigsMap(double eps = 0.001)
        {
            int i = 0;
            for (Curves::iterator itA = curves.begin(); itA != curves.end(); ++itA, i++)
            {
                Point2d a = itA->points.front();
                for (Curves::iterator itB = curves.begin(); itB != curves.end(); ++itB)
                {
                    if (itA->coef <= itB->coef && itB->inBox(a, eps))
                    {
                        NearCurve nearCurve;
                        nearCurve.point = itB->getNearestSeg(a, nearCurve.newP, nearCurve.endingP);
                    }
                }
                a = itA->points.back();
                for (Curves::iterator itB = curves.begin(); itB != curves.end(); ++itB)
                {
                    if (itA->coef <= itB->coef && itB->inBox(a, eps))
                    {
                        NearCurve nearCurve;
                        nearCurve.point = itB->getNearestSeg(a, nearCurve.newP, nearCurve.endingP);
                    }
                }
                //itA->id = i;
            }

            CurvePoints curvePoints;
            for (Curves::iterator it = curves.begin(); it != curves.end(); ++it)
            {
                int k = 0;
                for (Points2d::iterator itP = it->points.begin(); itP != it->points.end(); ++itP, k++)
                {
                    curvePoints.push_back(CurvePoint(*itP, &(*it), k));
                }
            }
            std::sort(curvePoints.begin(), curvePoints.end());

            Point2d center;
            for (CurvePoints::iterator itA = curvePoints.begin(); itA != curvePoints.end(); ++itA)
            {
                for (CurvePoints::iterator itB = itA+1; itB != curvePoints.end() && itB->point.x < itA->point.x+eps; ++itB)
                {
                    double dist = (itA->point - itB->point).Len();
                    if (dist < eps && itA->curve != itB->curve)
                    {
                        size_t i;
                        Point2CurvePoint seg(itA->point, *itB);
                        //for (i = n; i < itA->curve->neigCurves.size() && itA->curve->neigCurves[i].to.curve != itB->curve; i++);
                        for (i = 0; i < itA->curve->neigCurves.size() && !(itA->point == itA->curve->neigCurves[i].from && itA->curve->neigCurves[i].to.curve == itB->curve); i++);
                        if (i < itA->curve->neigCurves.size())
                        {
                            if ((itA->curve->neigCurves[i].to.point - itA->point).Len() > dist)
                            {
                                itA->curve->neigCurves[i] = seg;
                            }
                        }
                        else
                            itA->curve->neigCurves.push_back(seg);

                        Point2CurvePoint seg2(itB->point, *itA);
                        for (i = 0; i < itB->curve->neigCurves.size() && !(itB->point == itB->curve->neigCurves[i].from  && itB->curve->neigCurves[i].to.curve == itA->curve); i++);
                        if (i < itB->curve->neigCurves.size())
                        {
                            if ((itB->curve->neigCurves[i].to.point - itB->point).Len() > dist)
                            {
                                itB->curve->neigCurves[i] = seg2;
                            }
                        }
                        else
                            itB->curve->neigCurves.push_back(seg2);
                    }
                }
            }

            for (Curves::iterator it = curves.begin(); it != curves.end(); ++it)
                std::sort(it->neigCurves.begin(), it->neigCurves.end());
        }

//------------------------------------------------------------------------------------------------------------
#include <math.h>

        struct Node
        {
            Node(const Curve* c=0, int n0=0, const Point2d& p=Point2d(0))
                    : len(0.0), wlen(0), len2(0), lenFromStation(0)
                    , jump(0), jumpslen(0.0)
                    , stationsInPath(0)
                    , point(p), curve(c), n(n0)
                    , coefd(1.0)
                    , predNode(0)
                    , c1(0), c2(0)
                    , si(0)
            {}
            double len;
            double wlen;
            double len2;
            double lenFromStation;
            int jump;
            double jumpslen;
            int stationsInPath;
            Point2d point;
            const Curve* curve;
            int n;
            std::vector<double> coef;
            double coefd;
            double getCoef()
            {
                int L = std::max(curve->bestPoint[n], curve->bestPoint[predNode->n]);
                if (L == 0)
                    return 1;
                if (L == 1)
                    return 1.3;
                if (L == 2)
                    return 1.5;
                return 1.7;
            }

            const Node* predNode;
            Point2d p1;
            const Curve* c1;
            const Curve* c2;

            const ::Pathfinder::StationInfo* si;
            //if si!=0 then curve mb 0

            double Rast() const
            {
                double res = wlen+(point-p1).Len();
                return res;
            }

            inline bool operator<(const Node& sn) const
            {
                return Rast() > sn.Rast();
            }

        };
        struct SetNode
        {
            SetNode(Node* n = 0)
                    : node(n)
            {}
            Node* node;
            inline bool operator<(const SetNode& t) const
            {
                if (node->curve != t.node->curve)
                    return node->curve < t.node->curve;
                return node->n < t.node->n;
            }
        };
        struct SetKey
        {
            SetKey(const Curve* c, const Point2d p, const int n0)
                    : curve(c), point(p), n(n0)
            {}
            SetKey()
                    : curve(0), n(0)
            {}
            const Curve* curve;
            const Point2d point;
            const int n;
            inline bool operator<(const SetKey& t) const
            {
                if (curve != t.curve)
                    return curve < t.curve;
                if (n != t.n)
                    return n < t.n;
                return point < t.point;
            }
        };
        struct HeapNode
        {
            HeapNode(Node* n = 0)
                    : node(n)
            {}
            Node* node;
            inline bool operator<(const HeapNode& t) const
            {
                return node->Rast() > t.node->Rast();
            }
        };
        typedef std::vector<Node> Nodes;
        typedef std::map<SetKey, SetNode> SetNodes;
        typedef std::priority_queue<HeapNode> Queue;

//------------------------------------------------------------------------------------------------------------

        bool addNode(const Point2d& p1, const Point2d& p2, const double d
                , const Node* parent, const Curve* curve, int n, Point2d p
                , Nodes& nodes, SetNodes& setNodes, Queue& queue
                , const double maxrast, const double eps = 0);

        int calcP2PCurve(std::ostream& log, Nodes& nodes, const Curve* c1, const Point2d p1, const Curve* c2, const Point2d p2, Points2d& resCurve, const double eps);

//------------------------------------------------------------------------------------------------------------

        /**
        Прокладываем кривую с точки p1, стоящей на c1, до p2 стоящей на c2.
        Результат кладем в points
        Максимальный прыжок между кривыми eps
        **/
        double addCurve(const Curve& c1, const Point2d p1, const Curve& c2, const Point2d p2, Points2d& points, double eps)
        {
            int n1 = c1.findPointEps(p1, eps);
            int n2 = c2.findPointEps(p2, eps);
            double resLen = 0;
            if (&c1 == &c2)
            {
                if (n1 >= 0 && n2 >= 0)
                {
                    if (n1 < n2)
                        for (int p = n1; p <= n2; p++)
                        {
                            if (!points.empty())
                                resLen += (points.back() - c1.points[p]).Len();
                            points.push_back(c1.points[p]);
                        }
                    else if (n2 < n1)
                        for (int p = n1; p >= n2; p--)
                        {
                            if (!points.empty())
                                resLen += (points.back() - c1.points[p]).Len();
                            points.push_back(c1.points[p]);
                        }
                    return resLen;
                }
                return 0.0;
            }

            Point2CurvePoints::const_iterator cp;
            double bestLen = -1;
            double dist = (p1-p2).Len();
            for (Point2CurvePoints::const_iterator it = c1.neigCurves.begin(); it != c1.neigCurves.end(); ++it)
            {
                if (it->to.curve == &c2)
                {
                    double len = (p1 - it->from).Len() + (p2 - it->to.point).Len() + 10*(it->from - it->to.point).Len();
                    if ((bestLen < 0 || len < bestLen) && len < 3*dist)
                    {
                        cp = it;
                        bestLen = len;
                    }
                }
            }
            if (bestLen >= 0)
            {
                resLen += addCurve(c1, p1, c1, cp->from, points, eps);
                resLen += addCurve(c2, cp->to.point, c2, p2, points, eps);
                return true;
            }

            Point2CurvePoints::const_iterator cp1, cp2;
            const Curve* c3 = 0;
            bestLen = -1;
            int count = 0;
            for (Point2CurvePoints::const_iterator it = c1.neigCurves.begin(); it != c1.neigCurves.end() && count < 10000; ++it)
                for (Point2CurvePoints::const_iterator it1 = it->to.curve->neigCurves.begin(); it1 != it->to.curve->neigCurves.end(); ++it1)
                {
                    count++;
                    if (it1->to.curve == &c2)
                    {
                        double len = (p1 - it->from).Len() + (it->to.point - it1->from).Len() + (p2 - it1->to.point).Len();
                        if ((bestLen < 0 || len < bestLen) && len < 3*dist)
                        {
                            c3 = it->to.curve;
                            cp1 = it;
                            cp2 = it1;
                            bestLen = len;
                        }
                    }
                }
            if (bestLen >= 0)
            {
                resLen += addCurve(c1, p1, c1, cp1->from, points, eps);
                resLen += addCurve(*c3, cp1->to.point, *c3, cp2->from, points, eps);
                resLen += addCurve(c2, cp2->to.point, c2, p2, points, eps);
                return true;
            }


            return false;
        }


    };


}

#endif
