#pragma once
#ifndef DATES_H
#define DATES_H

#include "pf.h"
#include <chrono>

#ifndef INT_LEN
#define INT_LEN 32
#endif

namespace Pathfinder
{
    // вычисляем дату один раз при старте приложения
    // дата связана с генерацией новых данных для пересадочника в админке
    extern const std::tm* start_time;

    // для работы с датами
    namespace Dates
    {
        // является ли год високосным?
        static bool _isLeap(const int year) { return (year % 4 == 0); }
        // сколько прошло дней начиная с _originYear до year
        static int _toDay(const int year)
        {
            int days(0);
            for(int y = ORIGIN_YEAR; y < year; ++y)
                days += (_isLeap(y)? DAYS_IN_YEAR_LEAP : DAYS_IN_YEAR_UNLEAP);
            return days;
        }

        // сколько дней в месяце
        // ВАЖНО: считается, что все года високосные и в феврале 29 дней
        // (так как длина маски совпадает с количеством дней в високосном гооду)
        static int _daysInMonth(const int year, const int month)
        {
            // количнество дней в месяце
            switch(month)
            {
                case 0: case 2: case 4: case 6: case 7: case 9: case 11: return 31;
                case 3: case 5: case 8: case 10: return 30;
                case 1: return (_isLeap(year) ? 29 : 28);
            }
            assert(false);
            return 0;
        }

        // сколько прошло дней начиная с начала года до months
        static int _toDay(const int year, const int month)
        {
            int days(0);
            for(int m = 0; m < month; ++m)
                days += _daysInMonth(year, m);
            return days;
        }

        // преобразовать дату в день (т.е. кол-во дней с ORIGIN_YEAR:01:01)
        static int toDay(const int year, const int month, const int dayOfMonth)
        { return _toDay(year) + _toDay(year, month) + dayOfMonth; }
        // преобразовать день day (т.е. кол-во дней с ORIGIN_YEAR:01:01) в дату year:month:dayOfMonth
        static void toDate(const int day, int & year, int & month, int & dayOfMonth)
        {
            // определяем год
            int yearDays = 0;
            int y = ORIGIN_YEAR;
            for (; ; ++y)
            {
                const int ydays = (_isLeap(y)? DAYS_IN_YEAR_LEAP : DAYS_IN_YEAR_UNLEAP);
                if (yearDays + ydays > day)
                    break;
                else
                    yearDays += ydays;
            }
            year = y;

            // определяем месяц
            int monthYearDays = yearDays;
            int m = 0;
            for (; ; ++m)
            {
                const int mdays = _daysInMonth(year, m);
                if (monthYearDays + mdays > day)
                    break;
                else
                    monthYearDays += mdays;
            }
            month = m;

            // определяем день месяца
            dayOfMonth = day - monthYearDays;
        }

        // в строку
        static inline std::string toString(const int & time, bool dateOut = true, bool sec=false)
        {
            const int day(time / MINUTES_IN_DAY);
            int year(0);
            int month(0);
            int dayOfMonth(0);

            toDate(day, year, month, dayOfMonth);

            std::ostringstream ost;
            if (dateOut)
                ost << year << '-' << (month + 1 < 10 ? "0" : "") << month + 1 << '-' << (dayOfMonth + 1 < 10 ? "0" : "") << dayOfMonth + 1 << ' ';

            const int hr = (time % MINUTES_IN_DAY) / MINUTES_IN_HOUR;
            const int mn = (time % MINUTES_IN_DAY) % MINUTES_IN_HOUR;

            ost << (hr < 10 ? "0" : "") << hr << ':' << (mn < 10 ? "0" : "") << mn;
            if (sec)
                ost << ":00";

            return ost.str();
        }

        static inline std::string toStringDate(const int & time)
        {
            const int day(time / MINUTES_IN_DAY);
            int year(0);
            int month(0);
            int dayOfMonth(0);

            toDate(day, year, month, dayOfMonth);

            std::ostringstream ost;
            ost << year << '-' << (month + 1 < 10 ? "0" : "") << month + 1 << '-' << (dayOfMonth + 1 < 10 ? "0" : "") << dayOfMonth + 1 << ' ';
            return ost.str();
        }

        static inline int fromString(const char* str)
        {
            int year(0), month(0), dayOfMonth(0), hour(0), minute(0), second(0);
            std::istringstream ist(str);
            char sep('\0');
            if (ist >> year >> sep >> month >> sep >> dayOfMonth >> sep >> hour >> sep >> minute >> sep >> second)
            {
                // переводим в минуты начиная с 1-го января ORIGIN_YEAR года
                int res = minute + hour * MINUTES_IN_HOUR + Dates::toDay(year, month - 1, dayOfMonth - 1) * MINUTES_IN_DAY;
                return res;
            }
            else
                return -1;
        }

        static inline int fromStringTime(const char* str)
        {
            int hour(0), minute(0), second(0);
            std::istringstream ist(str);
            char sep('\0');
            ist >> hour >> sep >> minute >> sep >> second;
            // переводим в минуты начиная с 1-го января ORIGIN_YEAR года
            int res = minute + hour * MINUTES_IN_HOUR;
            return res;
        }

        static inline int fromStringDate(const char* str)
        {
            int year(0), month(0), dayOfMonth(0);
            std::istringstream ist(str);
            char sep('\0');
            ist >> year >> sep >> month >> sep >> dayOfMonth;
            // переводим в минуты начиная с 1-го января ORIGIN_YEAR года
            int res = Dates::toDay(year, month - 1, dayOfMonth - 1) * MINUTES_IN_DAY;
            return res;
        }


    }//namespace Dates

    struct SearchHelper
    {
        int i, j;
        size_t mask1, mask2;
        static size_t daysInThisYear;
        SearchHelper():i(-1), j(-1), mask1(0), mask2(0)
        {}
        SearchHelper(size_t a)
        {
            i = (int)(a>>5);//a/INT_LEN;
            j = -1;
            mask1 = (size_t)(1<<(a&31));//(a%INT_LEN);
            mask2 = 0;
        }
        SearchHelper(size_t a, size_t b) : i(-1), j(-1), mask1(0), mask2(0)
        {
            if (a > b)
                b += daysInThisYear;
            for (size_t d = a; d < b; d++)
            {
                size_t c = d%daysInThisYear;
                size_t cd32 = c>>5;///INT_LEN;
                size_t co32 = c&31;//%INT_LEN;
                if (i == -1)
                    i = (int)cd32;
                if (i != (int)cd32)
                {
                    j = (int)cd32;
                    mask2 |= (int)(1<<co32);
                }
                else
                    mask1 |= (int)(1<<co32);
            }
        }
    };

    // информация об отправлениях, привязана к Москве
    class Departures
    {
    private:
        // маска хождения: true - транспорт в этот день отправляется, иначе - false
        // ВАЖНО: размер маски независимо от того год является високосным или нет равен 366 = DAYS_IN_YEAR_LEAP
        // если год не високосный, то 29 февраля (isGoodDayMask[59]), считается, что транспорт не ходит
        // таким образом, в году всегда есть 29 февраля; только в не високосный год
        // за 28 февраля следует 1 марта (за 58 днем следует 60 день)
        size_t BYTES;
        size_t dayMask[DAYS_IN_YEAR_LEAP/32+1];
        // размер маски (в начале равен 0, затем 366 = DAYS_IN_YEAR_LEAP)
        size_t maskSize;

    public:
        // время отправления: в минутах начиная с 00:00
        // т.е. маска + время дают всю временную информацию
        int time;

    public:
        // конструктор
        Departures(): maskSize(0), time(-1)
        {
            BYTES = DAYS_IN_YEAR_LEAP/32 + 1;
            for (size_t i = 0; i < BYTES; i++)
                dayMask[i] = 0;
        }

    public:
        // все хорошо
        //TODO: разрешить maskSize == 0 или нет?
        inline bool isValid() const { return maskSize == BYTES && time >= 0; }
        // установить время равным hour:minute:second
        void setTime(const int hour, const int minute, const int second)
        {
            // секунды не используются
            time = hour * MINUTES_IN_HOUR + minute + second * 0;
        }
        void setTime(const int t) {time = t;}
        // возвращаем время в минутах если есть отправление (начиная с ORIGIN_YEAR),
        // в противном случае возвращаем -1
        inline int getTime(const int day) const
        { return (isGoodDay(day) ? (day * MINUTES_IN_DAY + time) : -1); }
        inline int getTime() const
        { return time; }
        inline bool operator==(const Departures& d) const
        {
            for (size_t i = 0; i < BYTES; i++)
                if (dayMask[i] != d.dayMask[i])
                    return false;
            return true;
        }


    public:
        static inline int toOurDay(const int day)
        {
            // считаем какой это день начиная с 1-го января ТЕКУЩЕГО года
            const int period = DAYS_IN_YEAR_LEAP + DAYS_IN_YEAR_UNLEAP * 3;
            const int rest = day % period;
            int d = (rest < DAYS_IN_YEAR_LEAP) ? rest : (rest - DAYS_IN_YEAR_LEAP) % DAYS_IN_YEAR_UNLEAP;

            // если следующий год високосный и до 29 февраля (59 день в маске) осталось меньше DAYS_IN_YEAR_UNLEAP - DAYS_TO_PAST дней,
            // то маска будет сдвинута вперёд на 1 день
            const int current_days = Dates::toDay(start_time->tm_year + 1900, start_time->tm_mon, start_time->tm_mday - 1);
            const int days_current_period = current_days % period;
            const int last_year_days = DAYS_IN_YEAR_LEAP + DAYS_IN_YEAR_UNLEAP * 2;
            const int before_leap = days_current_period - last_year_days;
            if (before_leap > 0 && ((before_leap - DAYS_TO_PAST) >= 59)
            && rest >= last_year_days && d >= 59)
            {
                d++;
            }
            // если текущий год високосный и до 1 марта (59 день в маске) осталось меньше DAYS_IN_YEAR_UNLEAP - DAYS_TO_PAST дней,
            // то маска будет сдвинута назад на 1 день
            else if(Dates::_isLeap(start_time->tm_year + 1900) && ((days_current_period - 1 - DAYS_TO_PAST) >= 59)
            && rest < DAYS_IN_YEAR_LEAP && d >= 59)
            {
                d--;
            }

            return d;
        }

    public:
        inline bool isGoodDay(const SearchHelper& sh) const
        {
            return (dayMask[sh.i]&sh.mask1) || (sh.j >= 0 && (dayMask[sh.j]&sh.mask2));
        }

        // отправляется в этот день транспорт или нет
        // день - это количество дней начиная с ORIGIN_YEAR
        inline bool isGoodDay(const int day) const
        {
            const size_t d = toOurDay(day);
            return (bool)(dayMask[d>>5]&(1<<(d&31)));
            //return isGoodDay(SearchHelper(d));
        }

        // отправляется в этот промежуток дней транспорт или нет
        // день - это количество дней начиная с ORIGIN_YEAR
        inline bool isGoodDay(const size_t day1, const size_t day2) const
        {
            const size_t d1 = toOurDay(day1);
            const size_t d2 = toOurDay(day2);
            return isGoodDay(SearchHelper(d1, d2));
        }

        // по строке из нулей и единиц формируем маску отправлений
        void initGoodDayMask(const std::string & yearMask)
        {
            maskSize = yearMask.size();
            if (SearchHelper::daysInThisYear == 0)
                SearchHelper::daysInThisYear = maskSize;
            for (size_t day = 0, i = 0, j = 0; day < maskSize; day++, j++)
            {
                if (j >= INT_LEN)
                {
                    j = 0;
                    i++;
                }
                //size_t i = day/INT_LEN;
                //size_t j = day%INT_LEN;
                dayMask[i] |= ((size_t)(yearMask[day] == '1') << j);
            }
        }

        // вернуть время отправления после времени time (время прибытия на предыдущую станцию) или -1
        inline int getDepartureTimeAfter(const int time) const
        {
            int day = time / MINUTES_IN_DAY;
            int dt = getTime(day);
            if (dt == -1 || time > dt)
            {
                day++;
                dt = getTime(day);
            }
            return dt;
        }
    };


    struct TimeZone
    {
        std::string name;
        int gmt[366];
        TimeZone() {}
        TimeZone(const std::string& n) {name = n;}
        inline bool operator<(const TimeZone& t) const
        {
            return name < t.name;
        }
        int getNightTime(int from, int to) const
        {
            int day = Departures::toOurDay(from);
            int g = gmt[day];
            int d = from / MINUTES_IN_DAY;
            int f = d*MINUTES_IN_DAY;
            int timef = (from - f) + g;
            int timet = (to - f) + g;
            if (timef < 0)
            {
                timef += MINUTES_IN_DAY;
                timet += MINUTES_IN_DAY;
            }
            int nd = timet / MINUTES_IN_DAY;
            int res = nd * MINUTES_IN_HOUR * 6;
            timet -= nd * MINUTES_IN_DAY;
            res += std::max(0, 6*MINUTES_IN_HOUR-timef);
            res -= std::max(0, 6*MINUTES_IN_HOUR-timet);
            return res;
        }
    };
    typedef std::vector<TimeZone> TimeZones;
    struct TimeZoneStorage
    {
        TimeZones timeZones;
        int Load(const char* tzfile)
        {
            std::ifstream fin(tzfile);
            if (fin.fail())
                return 0;
            int res = 0;
            std::string tzname;
            for (; fin >> tzname; res++)
            {
                TimeZone tz(tzname);
                int gmt = 0;
                for (int i = 0; i < 366 && fin >> gmt; i++)
                    tz.gmt[i] = gmt;
                timeZones.push_back(tz);
            }
            std::sort(timeZones.begin(), timeZones.end());
            return res;
        }
        const TimeZone* getTimeZone(const std::string& tzname)
        {
            TimeZone tz(tzname);
            TimeZones::iterator it = std::lower_bound(timeZones.begin(), timeZones.end(), tz);
            if (it == timeZones.end() || it->name != tzname)
                return 0;
            return &(*it);
        }
    };
}

#endif

