#ifndef ESTIMATION_H
#define ESTIMATION_H

#include <stdint.h>
#include "pf.h"
#include "stationgraph.h"
#include "dates.h"

namespace Pathfinder
{
    // храним оценки снизу на расстояния от всех станций до станции назначения
    struct Estimation
    {
        int changes;
        int discomfort;
        int weight;
        int price;
        int trPoint;
        int begTime;
        int timeForBoarding;
        int hash;
#ifdef DISCOMFORT_STR_ON
        std::string discomfortString;
#endif
        Estimation() : changes(MAX_CH)
                , discomfort(0)
                , weight(0), price(0), trPoint(0), begTime(0),  timeForBoarding(0), hash(0)
        {}
        explicit Estimation(const int ch, const int wei=0, const int p=0, const int tp=0, const int d = 0, const int t=0)
                : changes(ch), discomfort(d), weight(wei), price(p)
                , trPoint(tp)
                , begTime(t),  timeForBoarding(0), hash(0) {}

        // ничем разумным не инициализировано
        bool empty() const
        {
            return changes >= MAX_CH;
        }
        int endTime() const
        {
            return begTime+weight;
        }

        // оператор сравнения
        inline bool operator<(const Estimation& ws) const
        {
            uint_fast64_t me    = (uint_fast64_t)changes & 0xF;
            uint_fast64_t other = (uint_fast64_t)ws.changes & 0xF;
            me    <<= 28;
            me    |=  price & 0xFFFFFFF;
            me    <<= 16;
            me    |=  weight & 0xFFFF;
            me    <<= 16;
            me    |=  discomfort & 0xFFFF;

            other <<= 28;
            other |=  price & 0xFFFFFFF;
            other <<= 16;
            other |=  ws.weight & 0xFFFF;
            other <<= 16;
            other |=  ws.discomfort & 0xFFFF;
            return me < other;
        }
        inline bool operator<=(const Estimation& ws) const
        {
            uint_fast64_t me    = (uint_fast64_t)changes & 0xF;
            uint_fast64_t other = (uint_fast64_t)ws.changes & 0xF;
            me    <<= 28;
            me    |=  price & 0xFFFFFFF;
            me    <<= 16;
            me    |=  weight & 0xFFFF;
            me    <<= 16;
            me    |=  discomfort & 0xFFFF;

            other <<= 28;
            other |=  price & 0xFFFFFFF;
            other <<= 16;
            other |=  ws.weight & 0xFFFF;
            other <<= 16;
            other |=  ws.discomfort & 0xFFFF;
            return me <= other;
        }

        inline bool operator==(const Estimation& ws) const
        {
            return (changes == ws.changes && weight == ws.weight && price == ws.price
                    && discomfort == ws.discomfort
                    && begTime == ws.begTime
                    && timeForBoarding == ws.timeForBoarding && hash == ws.hash);
        }
        inline bool operator!=(const Estimation& ws) const
        {
            return !(*this == ws);
        }

        double getPriceForPrint() const
        {
            if (price >= MAX_PRICE)
                return -1;
            return price*0.01;
        }
        void plusPrice(const int p)
        {
            price += p;
            price = std::min(price, MAX_PRICE);
        }
#ifdef DISCOMFORT_STR_ON
        void plusDiscomfort(const int d
			, const char* dStr = 0 )
		{
			discomfort += d;
			discomfort = std::min(discomfort, MAX_DISCOMFORT);
			if (dStr && dStr[0])
			{
				discomfortString += " + ";
				char buf[100];
				sprintf(buf, "%d:", d);
				discomfortString += buf;
				discomfortString += dStr;
			}
		}
		void setDiscomfort(const int d
			, const char* dStr = 0 )
		{
			if (dStr && dStr[0])
			{
				discomfortString += " + ";
				char buf[100];
				sprintf(buf, "%d:", d-discomfort);
				discomfortString += buf;
				discomfortString += dStr;
			}
			discomfort = d;
			discomfort = std::min(discomfort, MAX_DISCOMFORT);
		}
#else
        void plusDiscomfort(const int d
                , const char* dStr = 0 )
        {
            discomfort += d;
            discomfort = std::min(discomfort, MAX_DISCOMFORT);
        }
        void setDiscomfort(const int d
                , const char* dStr = 0 )
        {
            discomfort = d;
            discomfort = std::min(discomfort, MAX_DISCOMFORT);
        }
#endif

        void save(std::ostream& fout) const
        {
            fout << "ch=" << changes << "\ttrPoint=" << trPoint << "\twei=" << weight;
        }

        inline const Estimation& min(const Estimation& e) const
        {
            if (*this < e)
                return *this;
            return e;
        }
        inline const Estimation& plusplus(const Estimation& e)
        {
            changes += e.changes; changes = std::min(changes, MAX_CH);
            price += e.price; price = std::min(price, MAX_PRICE);
            weight += e.weight;
            trPoint += e.trPoint;
            discomfort += e.discomfort;
            return *this;
        }

        enum BetterItem
        {
            NONE = 0,
            BETTER = 1,
            WORSE = 2,
            BOTH = 4,
            ANY = 8,
            EPS = 16,
            BETTER_EPS = 17,
            WORSE_EPS = 18
        };

        inline BetterItem better4(const Estimation& e) const
        {
            if (weight == e.weight && begTime == e.begTime && changes == e.changes && price == e.price && discomfort == e.discomfort)
            {
                if (timeForBoarding > e.timeForBoarding)
                    return BETTER;
                if (timeForBoarding < e.timeForBoarding)
                    return WORSE;
                return ANY;
            }
#ifdef FOR_WIZARD
            if (changes == e.changes)
#else
            if (changes == e.changes || changes==1 || std::abs(e.begTime-begTime)>23*60)
#endif
            if ((begTime > e.begTime && endTime() > e.endTime()) || (begTime < e.begTime && endTime() < e.endTime())) //не вкладываются
                return BOTH;

            int ch = e.changes - changes;
            int w = e.weight - weight;
            int p = e.price - price;
            int d = e.discomfort - discomfort;

            if (ch >= 0 && w >=0 && p >= 0 && d >= 0)
                return BETTER;
            if (ch <= 0 && w <=0 && p <= 0 && d <= 0)
                return WORSE;

            bool chEps = e.changes == changes;
            bool wEps = e.weight == weight;
            bool pEps = std::abs(p) <= ((e.price + price)>>(EPS_SHIFT+1));
            bool dEps = std::abs(d) <= ((e.discomfort + discomfort)>>EPS_SHIFT);

            if (chEps & wEps & pEps & dEps)
            {
                if (changes + weight + price + discomfort <= e.changes + e.weight + e.price + e.discomfort)
                    return BETTER;
                return WORSE;
            }

            if (((ch>=0) | chEps) && ((w>=0) | wEps) && ((p>=0) | pEps) && ((d>=0) | dEps))
                return BETTER;
            if (((ch<=0) | chEps) && ((w<=0) | wEps) && ((p<=0) | pEps) && ((d<=0) | dEps))
                return WORSE;
            return BOTH;
        }

        inline BetterItem better4result(const Estimation& e) const
        {
            if (weight == e.weight && begTime == e.begTime && changes == e.changes)
            if (price == e.price && discomfort == e.discomfort)
            {
                if (timeForBoarding > e.timeForBoarding)
                    return BETTER;
                if (timeForBoarding < e.timeForBoarding)
                    return WORSE;
                return ANY;
            }
            int wE = (e.weight + weight)>>(EPS_SHIFT+1);
#ifdef FOR_WIZARD
            if (changes == e.changes)
#else
            if (changes == e.changes || changes==1 || std::abs(e.begTime-begTime)>23*60)
#endif
            {
                int d1 = begTime - e.begTime;
                int d2 = endTime() - e.endTime();
                if ((d1>0 && d2>0 && d1+d2>wE) || (d1<0 && d2<0 && -d1-d2>wE)) //не вкладываются
                    return BOTH;
            }

            int ch = e.changes - changes;
            int w = e.weight - weight;
            int p = e.price - price;
            int d = e.discomfort - discomfort;

            if (ch >= 0 && w >=0 && p >= 0 && d >= 0)
                return BETTER;
            if (ch <= 0 && w <=0 && p <= 0 && d <= 0)
                return WORSE;

            bool chEps = e.changes == changes;
            bool wEps = std::abs(w) <= wE;
            bool pEps = std::abs(p) <= ((e.price + price)>>(EPS_SHIFT+1));
            bool dEps = std::abs(d) <= 10 || std::abs(d) <= ((e.discomfort + discomfort)>>(EPS_SHIFT-1));

            if (chEps & wEps & pEps & dEps)
            {
                if (changes + weight + price + discomfort <= e.changes + e.weight + e.price + e.discomfort)
                    return BETTER;
                return WORSE;
            }

            if (((ch>=0) | chEps) && ((w>=0) | wEps) && ((p>=0) | pEps) && ((d>=0) | dEps))
                return BETTER;
            if (((ch<=0) | chEps) && ((w<=0) | wEps) && ((p<=0) | pEps) && ((d<=0) | dEps))
                return WORSE;
            return BOTH;
        }

        //задает только частичный порядок
        //отношение "точно, без сомнения лучше"
        //но транзитивность должна быть обязательно!!!
        inline bool better(const Estimation& e) const
        {
            if (changes > e.changes)
                return false;
            if (price > e.price)
                return false;
            if (weight > e.weight)
                return false;
#ifdef FOR_WIZARD
            if (changes == e.changes)
#else
            if (changes == e.changes || changes==1)
#endif
            {
                if (begTime < e.begTime)
                    return false;
                if (endTime() > e.endTime())
                    return false;
            }
            if (changes == e.changes && price == e.price && weight == e.weight && discomfort == e.discomfort && begTime == e.begTime)
            {
                if (timeForBoarding > e.timeForBoarding)
                    return true;
                return false;
            }

            return discomfort <= e.discomfort;
        }

        //используется для пропускания через станцию, где есть вертекс с
        inline bool worseOrEqual(const Estimation& e) const
        {
            if (changes < e.changes)
                return false;
            if (price < e.price)
                return false;
            if (weight < e.weight)
                return false;
#ifdef FOR_WIZARD
            if (changes == e.changes)
#else
            if (changes == e.changes || changes==1)
#endif
            {
                if (begTime > e.begTime)
                    return false;
                if (endTime() < e.endTime())
                    return false;
            }
            if (changes == e.changes && price == e.price && weight == e.weight && discomfort == e.discomfort && begTime == e.begTime)
            {
                if (timeForBoarding < e.timeForBoarding)
                    return true;
                return hash == e.hash;
            }
            return discomfort >= e.discomfort;
        }

        std::string toString() const
        {
            std::ostringstream ost;
            toStream(ost);
            return ost.str();
        }
        void toStream(std::ostream& ost) const
        {
            std::string begTimeStr = Dates::toString(begTime);

            ost << "est=("
            << begTimeStr << " "
            << changes << ".ch "
            << price << ".pr "
            << weight << ".wei "
            << discomfort << ".dis "
            << hash << ".hash"
            << ")";
        }
    };
    typedef std::vector<Estimation> Estimations;

    inline std::ostream& operator << (std::ostream& stream, const Estimation& e)
    {
        std::string begTimeStr = Dates::toString(e.begTime);
        if (e.begTime==0)
            begTimeStr="";
        return stream << "("
               << begTimeStr << " "
               << e.changes << ".ch "
               << e.price << ".pr "
               << e.weight << ".wei "
               << e.discomfort << ".dc "
               << e.hash << ".hash"
               << ")";
    }

}

#endif

