#ifndef GeomH
#define GeomH

#include <math.h>
#include <iostream>
#include <fstream>
#include <iomanip>
#include <vector>

#define mmEps 1e-12
#define mmEps2 1e-24
#define mEps 1e-12
#define mEps3 1e-36

namespace Geom
{
/****************************************************************************
							Point3
****************************************************************************/
//---------------------------------------------------------------------------
    template<class N> class Point3
    {
    public:

        N x; N y; N z;

        Point3() :  x( 0 ), y( 0 ), z( 0 ) {}
        explicit Point3( const N a_t ) :  x( a_t ), y( a_t ), z( a_t ) {}
        explicit Point3( const N a_x, const N a_y, const N a_z ) :  x( a_x ), y( a_y ), z( a_z ) {}
        template <class T> explicit Point3<N>(Point3<T> a) :  x( a.x ), y( a.y ), z( a.z ) {}

        Point3 operator - () const {return Point3<N>(-x, -y, -z);};
        Point3& operator += ( const Point3& a) {x += a.x; y += a.y; z += a.z; return *this;}
        Point3& operator -= ( const Point3& a) {x -= a.x; y -= a.y; z -= a.z; return *this;}
        Point3& operator *= ( const N a) {x *= a; y *= a; z *= a; return *this;}
        Point3& operator /= ( const N a) {x /= a; y /= a; z /= a; return *this;}

        Point3& Normalize()
        { N d = x*x+y*y+z*z;
            if (d > mmEps2)
            { d = 1./sqrt(d);
                x *= d; y *= d; z *= d;
            }
            return *this;
        }
        Point3& NormalizeAbs()
        { N d = std::abs(x)+std::abs(y)+std::abs(z);
            if (d > mmEps)
            { d = 1./d;
                x *= d; y *= d; z *= d;
            }
            return *this;
        }
        Point3& NormalizeSqrt()
        { N d = std::abs(x)+std::abs(y)+std::abs(z);
            if (d > mmEps)
            { d = 1./sqrt(d);
                x *= d; y *= d; z *= d;
            }
            return *this;
        }

        N Len2() const {return x*x + y*y + z*z;}
        N Len() const {return sqrt(x*x + y*y + z*z);}
        N operator!() const {return Len();}
        N LenAbs() const {return std::abs(x) + std::abs(y) + std::abs(z);}

        N Pow(const int a, const int b, const int c) const
        {
            int i;
            N res = 1;
            for (i = 0; i < a; i++)
                res *= x;
            for (i = 0; i < b; i++)
                res *= y;
            for (i = 0; i < c; i++)
                res *= z;
            return res;
        }

        void SetColor(int color) {
            x = (color & 0x00FF0000) >> 16;
            y = (color & 0x0000FF00) >> 8;
            z = (color & 0x000000FF);
        }

        const N * ptr() const { return & x; }  //  v x,y,z
        N * ptr() { return & x; }  //  v x,y,z
    };//Point3

    template <typename N>
    inline Point3<N> operator + (const Point3<N>& a, const Point3<N>& b){return Point3<N>(a.x+b.x, a.y+b.y, a.z+b.z);}
    template <typename N>
    inline Point3<N> operator - (const Point3<N>& a, const Point3<N>& b){return Point3<N>(a.x-b.x, a.y-b.y, a.z-b.z);}
    template <typename N, typename T>
    inline Point3<N> operator * ( const T a, const Point3<N>& b ){return Point3<N>(a*b.x, a*b.y, a*b.z);}
    template <typename N>
    inline N  operator & (const Point3<N>& a, const Point3<N>& b){return a.x*b.x + a.y*b.y + a.z*b.z;}
    template <typename N>
    inline Point3<N> operator ^ (const Point3<N>& a, const Point3<N>& b )
    {return Point3<N>(a.y*b.z - a.z*b.y, a.z*b.x - a.x*b.z, a.x*b.y - a.y*b.x);}

    template <typename N>
    inline N ScalMult(const Point3<N>& a, const Point3<N>& b) {return a.x*b.x + a.y*b.y + a.z*b.z;}
    template <typename N>
    inline Point3<N> VecMult(const Point3<N>& a, const Point3<N>& b)
    {return Point3<N>(a.y*b.z - a.z*b.y, a.z*b.x - a.x*b.z, a.x*b.y - a.y*b.x);}
    template <typename N>
    inline N MixMult(const Point3<N>& a, const Point3<N>& b, const Point3<N>& c) {return ScalMult(VecMult(a, b), c);}

    template <typename N>
    inline int operator == (const Point3<N>& a, const Point3<N>& b)
    {return std::abs(a.x - b.x)
            + std::abs(a.y - b.y)
            + std::abs(a.z - b.z) < mmEps;}
    template <typename N>
    inline int operator != (const Point3<N>& a, const Point3<N>& b)
    {return std::abs(a.x - b.x)
            + std::abs(a.y - b.y)
            + std::abs(a.z - b.z) >= mmEps;}
    template <typename N>
    inline int operator > (const Point3<N>& a, const Point3<N>& b)
    {
        if (a.x > b.x) return 1;
        if (b.x > a.x) return 0;
        if (a.y > b.y) return 1;
        if (b.y > a.y) return 0;
        if (a.z > b.z) return 1;
        return 0;
    }

    template <typename N>
    inline Point3<N> Normalize(const Point3<N>& a)
    {
        Point3<N> b(a);
        b.Normalize();
        return b;
    }

    template <typename N>
    inline Point3<N> MinCoord (Point3<N> a, const Point3<N>& b)
    {
        if (a.x > b.x)
            a.x = b.x;
        if (a.y > b.y)
            a.y = b.y;
        if (a.z > b.z)
            a.z = b.z;
        return a;
    }
    template <typename N>
    inline Point3<N> MaxCoord (Point3<N> a, const Point3<N>& b)
    {
        if (a.x < b.x)
            a.x = b.x;
        if (a.y < b.y)
            a.y = b.y;
        if (a.z < b.z)
            a.z = b.z;
        return a;
    }
    template <typename N>
    inline void ReBox_(const Point3<N>& lbn, const Point3<N>& rtf, const Point3<N>& lbnNew, const Point3<N>& rtfNew
            , Point3<N>& center, Point3<N>& shift, N& scale)
    {
        Point3<N> l = rtf - lbn;
        Point3<N> lNew = rtfNew - lbnNew;

        N max = std::max(
                std::max(
                        (l.x != 0 && lNew.x != 0)?(l.x/lNew.x):0,
                        (l.y != 0 && lNew.y != 0)?(l.y/lNew.y):0
                ),
                (l.z != 0 && lNew.z != 0)?(l.z/lNew.z):0
        );

        center = 0.5*(rtf + lbn);
        shift = 0.5*(rtfNew + lbnNew);
        scale = 1.0;
        if (max != 0.0) scale = 1/max;
    }
    template <typename N>
    inline N Min(Point3<N> a)
    {
        if (a.x > a.y)
            a.x = a.y;
        if (a.x > a.z)
            a.x = a.z;
        return a.x;
    }
    template <typename N>
    inline N Max(Point3<N> a)
    {
        if (a.x < a.y)
            a.x = a.y;
        if (a.x < a.z)
            a.x = a.z;
        return a.x;
    }

    template <typename N>
    inline N Volume(const Point3<N>& a) {    return a.x * a.y * a.z;	}

    template <typename N>
    inline int GetRectIntersect(const Point3<N>& lbn1, const Point3<N>& rtf1
            , const Point3<N>& lbn2, const Point3<N>& rtf2
            , Point3<N>& lbnRes, Point3<N>& rtfRes)
    {
        lbnRes = MaxCoord(lbn1, lbn2);
        rtfRes = MinCoord(rtf1, rtf2);
        if (lbnRes == MinCoord(lbnRes, rtfRes))
            return 1;
        return 0;
    }

    template <typename N>
    inline N GetMinSin(const Point3<N>& a, const Point3<N>& b, const Point3<N>& c)
    {
        Point3<N> la, lb, lc;
        N n1, n2, n3;
        la = (c - b).Normalize();
        lb = (a - c).Normalize();
        lc = (b - a).Normalize();
        /*v  */
        n1 = VecMult(lc, lb).Len();
        n2 = VecMult(la, lc).Len();
        n3 = VecMult(la, lb).Len();
        return std::min(std::min(n1, n2), n3);
    }/*GetMinSin*/

    template <typename N>
    inline std::istream& operator >> (std::istream& stream, Point3<N>& a)
    {   return stream >> a.x >> a.y >> a.z;   }
    template <typename N>
    inline std::ostream& operator << (std::ostream& stream, const Point3<N>& a)
    {   return stream << a.x << ' ' << a.y << ' ' << a.z;   }


/****************************************************************************
							Point2
****************************************************************************/
    template<class N> class Point2
    {
    public:
        N x; N y;
        Point2() :  x( 0 ), y( 0 ) {  }
        explicit Point2( const N a_t ) :  x( a_t ), y( a_t ) {  }
        explicit Point2( const N a_x, const N a_y ) :  x( a_x ), y( a_y ) {  }
        template <class T> explicit Point2(Point2<T> a) :  x( a.x ), y( a.y ) {}

        Point2 operator - () const {return Point2(-x, -y);};
        Point2 Reverse(const Point2& a) { return Point2(a.y, a.x); }
        Point2& operator += ( const Point2& a) {x += a.x; y += a.y; return *this;}
        Point2& operator -= ( const Point2& a) {x -= a.x; y -= a.y; return *this;}
        Point2& operator *= ( const N a) {x *= a; y *= a; return *this;}

        Point2& Normalize()
        { N d = x*x+y*y;
            if (d > mmEps2)
            { d = 1./sqrt(d);
                x *= d; y *= d;
            }
            return *this;
        }
        bool IsZero() const
        { N d = x*x+y*y;
            return (d <= mmEps2);
        }
        N Tang() const
        { if (std::abs(x) < mmEps)
                return 1e6;
            return y/x;   }

        N Len2() const {return x*x + y*y;}
        N Len() const {return sqrt(x*x + y*y);}
        N LenAbs() const {return std::abs(x) + std::abs(y);}

        N Pow(const int a, const int b) const
        {
            int i;
            N res = 1;
            for (i = 0; i < a; i++)
                res *= x;
            for (i = 0; i < b; i++)
                res *= y;
            return res;
        }

        const N * ptr() const { return & x; }
        N * ptr() { return & x; }
    };//Point2

    template <typename N>
    inline Point2<N> operator + (const Point2<N>& a,   const Point2<N>& b){return Point2<N>(a.x+b.x, a.y+b.y);}
    template <typename N>
    inline Point2<N> operator - (const Point2<N>& a,   const Point2<N>& b){return Point2<N>(a.x-b.x, a.y-b.y);}
    template <typename N, typename T>
    inline Point2<N> operator * (const T a, const Point2<N>& b ){return Point2<N>(a*b.x, a*b.y);};
    template <typename N, typename T>
    inline Point2<N> operator * (const Point2<N>& b, const T a ){return Point2<N>(a*b.x, a*b.y);};
    template <typename N>
    inline Point2<N> operator * (const Point2<N>& a,   const Point2<N>& b ){return Point2<N>(a.x*b.x, a.y*b.y);};
    template <typename N>
    inline N operator &  (const Point2<N>& a,   const Point2<N>& b){return a.x*b.x + a.y*b.y;}
    template <typename N>
    inline N operator ^  (const Point2<N>& a,   const Point2<N>& b){return a.x*b.y - a.y*b.x;}

    template <typename N>
    inline int operator == (const Point2<N>& a, const Point2<N>& b)
    {return std::abs(a.x - b.x)
            + std::abs(a.y - b.y) < mmEps;}
    template <typename N>
    inline int operator != (const Point2<N>& a, const Point2<N>& b)
    {return std::abs(a.x - b.x)
            + std::abs(a.y - b.y) >= mmEps;}
    template <typename N>
    inline int operator > (const Point2<N>& a, const Point2<N>& b)
    {
        if (a.x > b.x) return 1;
        if (b.x > a.x) return 0;
        if (a.y > b.y) return 1;
        return 0;
    }
    template <typename N>
    inline bool operator < (const Point2<N>& a, const Point2<N>& b)
    {
        if (a.x < b.x) return true;
        if (b.x < a.x) return false;
        if (a.y < b.y) return true;
        return false;
    }

    template <typename N>
    inline std::istream& operator >> (std::istream& stream, Point2<N>& a)
    {   return stream >> a.x >> a.y;   }
    template <typename N>
    inline std::ostream& operator << (std::ostream& stream, const Point2<N>& a)
    {   return stream << a.x << ' ' << a.y;   }

    template <typename N>
    inline Point2<N> MinCoord (Point2<N> a, const Point2<N>& b)
    {
        if (a.x > b.x)
            a.x = b.x;
        if (a.y > b.y)
            a.y = b.y;
        return a;
    }
    template <typename N>
    inline Point2<N> MaxCoord (Point2<N> a, const Point2<N>& b)
    {
        if (a.x < b.x)
            a.x = b.x;
        if (a.y < b.y)
            a.y = b.y;
        return a;
    }
///      
///  x -  (-90, 90), y -  (-180, 180)
    template <typename N>
    inline N SphereLen (Point2<N> p1, const Point2<N>& p2)
    {
        double x = std::abs(p1.x + p2.x) * .5;
        double a = (x < 50.0) ? (111.33 - 0.0156*x*x - 0.023*x) : (135.35 - 0.00586*x*x - 0.978*x);
        double b = 110.44 + 0.014*x;
        double s = p1.x - p2.x;
        double temp = std::abs(p1.y - p2.y);
        double d = std::min(temp, 360.0 - temp);
        return sqrt(b*b*s*s + a*a*d*d);
    }
//-------------------------------------------types------------------------------------------------------

    typedef Point2<int> Point2i;
    typedef Point2<float> Point2f;
    typedef Point2<double> Point2d;
    typedef Point3<int> Point3i;
    typedef Point3<float> Point3f;
    typedef Point3<double> Point3d;
    typedef std::vector<Point2d> Points2d;

}//namespace Geom

#undef mEps
#undef mmEps
#undef mmEps2
#undef mEps3

#endif
