#include "get.h"
#include "dates.h"

namespace Pathfinder
{
    // храним ключи для пар "ключ-значение"
    // смысл - смотри в get.h
    const std::string Get::fromType     = "from_type";
    const std::string Get::fromID       = "from_id";
    const std::string Get::toType       = "to_type";
    const std::string Get::toID         = "to_id";
    const std::string Get::date         = "date";
    const std::string Get::transport    = "ttype";
    const std::string Get::boarding     = "boarding";
    const std::string Get::minDelay     = "min_delay";
    const std::string Get::maxDelay     = "max_delay";
    const std::string Get::maxTransfers = "max_transfers";
    const std::string Get::optimization = "optimize";
    const std::string Get::delta        = "delta";
    const std::string Get::station      = "station";
    const std::string Get::time         = "time";
    const std::string Get::iLoveSport   = "i_love_sport";
    const std::string Get::mode         = "mode";
    const std::string Get::timeLimit    = "time_limit";
    const std::string Get::canChangeInStartTown   = "can_change_in_start_town";
    const std::string Get::canChangeInFinishTown   = "can_change_in_finish_town";
    const std::string Get::canChangeInAnyTown   = "can_change_in_any_town";
    const std::string Get::usePrice   = "use_price";

#ifdef DEBUG_ROUTE
    const std::string Get::debugStation   = "debug_station";
    const std::string Get::debugRoute   = "debug_route";
#endif

    // загрузить (в том числе и распарсить)
    void
    Get::load(const std::string & in)
    {
        const std::string pingResource("GET /ping");

        if (in.substr(0, pingResource.length()) == pingResource) {
            _ping = true;
            return;
        }

        // вырезаем запрос
        const std::string query(_cutQuery(in));
        // получаем что-то, типа: "from_type=settlement&from_id=11643&to_type=settlement&to_id=8213&date=2008-04-01%2000:00:00&ttype=1&boarding=1440&min_delay=0&max_delay=480&max_transfers=3&optimize=time&delta=120"
        // или пустую строку
        std::istringstream ist(query);
        std::string portion;
        const char sep = '&';
        // читаем пары ключ-значения, разделенные символом '&', разбираем и добавляем
        while(std::getline(ist, portion, sep))
            _add(portion);

        // проверяем, что все, что нужно, прочитали
        #ifdef TEST_LOGIC
        PF_ASSERTION(_fromID >= 0, "valid from_id", _fromID);
        PF_ASSERTION(_toID >= 0, "valid to_id", _toID);
        PF_ASSERTION(_startTime >= 0, "valid date", _startTime);
        PF_ASSERTION(_transport != Transport::NONE && _transport <= Transport::ALL, "valid transport", _transport);
        PF_ASSERTION(_minDelay >= 0, "valid min_delay", _minDelay);
        PF_ASSERTION(_maxDelay >= 0, "valid max_delay", _maxDelay);
        PF_ASSERTION(_numOfChanges >= 0, "valid max_transfers", _numOfChanges);
        PF_ASSERTION(_delta >= 0, "valid delta", _delta);
        #endif
    }


    // вырезаем из строки get-запрос
    // в случае неудачи возвращаем пустую строку
    std::string
    Get::_cutQuery(const std::string & in)
    {
        // строка может иметь вид: "GET*?'query' *" или "GET*?'query'[\n]"
        std::string pattern("GET");
        std::string empty;
        // если не находим GET, то возвращаем пустую строку
        if (in.find(pattern) == std::string::npos)
            return empty;
        // начало запроса предваряется символом '?'
        const size_t begPos = in.find('?');
        if (begPos == std::string::npos)
            return empty;
        // запрос заканчиватся пробелом или концом строки
        size_t endPos = in.find(' ', begPos);
        if (endPos == std::string::npos)
            endPos = in.length();

        // проверка корректности начала и конца строки
        #ifdef TEST_LOGIC
        PF_ASSERTION(begPos < endPos, "valid positions", endPos);
        PF_ASSERTION(begPos >= 0 && begPos+1 < in.length(), "index is valid", begPos+1);
        PF_ASSERTION(endPos > 0 && endPos <= in.length(), "index is valid", endPos);
        #endif

        return in.substr(begPos+1, endPos-begPos);
    }


    // разбор и добавление пар ключ-значение
    bool
    Get::_add(const std::string & portion)
    {
        std::istringstream ist(portion);
        // разделитель пары ключ-значение
        const char sep = '=';

        // сюда сохраним ключ
        std::string keyword;
        // сюда сохраним значения (в зависимости от того - строка или нет)
        std::string tail;

        // прочитали ключ
        std::getline(ist, keyword, sep);
        // в зависимости от значения ключа решаем, что делать со значением
        //TODO: изящнее использовать для этого map
        if (keyword == fromType)
        {
            std::getline(ist, tail);
            _fromIsStation = (tail == station);
        }
        else if (keyword == fromID)
        {
            ist >> _fromID;
        }
        else if (keyword == toType)
        {
            std::getline(ist, tail);
            _toIsStation = (tail == station);
        }
        else if (keyword == toID)
        {
            ist >> _toID;
        }
        else if (keyword == date)
        {
            // читаем дату, за одним же проверяем сооствествие нашим ожиданиям

            //2008-04-10+00%3A00%3A00
            //2008-04-01%2000:00:00
            int year(0), month(0), dayOfMonth(0), hour(0), minute(0), second(0);
            char sep('\0');
            ist >> year;
            ist >> sep;
            #ifdef TEST_LOGIC
            PF_ASSERTION(sep == '-', "expected input 1", static_cast<int>(sep));
            #endif
            ist >> month;
            ist >> sep;
            #ifdef TEST_LOGIC
            PF_ASSERTION(sep == '-', "expected input 2", static_cast<int>(sep));
            #endif
            ist >> dayOfMonth;
            ist >> sep;
            if (sep == '%')
            {
                ist >> sep;
                #ifdef TEST_LOGIC
                PF_ASSERTION(sep == '2', "expected input", static_cast<int>(sep));
                #endif
                ist >> sep;
                #ifdef TEST_LOGIC
                PF_ASSERTION(sep == '0', "expected input", static_cast<int>(sep));
                #endif
                ist >> hour;
                ist >> sep;
                #ifdef TEST_LOGIC
                PF_ASSERTION(sep == ':', "expected input 1", static_cast<int>(sep));
                #endif
                ist >> minute;
                ist >> sep;
                #ifdef TEST_LOGIC
                PF_ASSERTION(sep == ':', "expected input 2", static_cast<int>(sep));
                #endif
                ist >> second;
            }
            else
            {
                #ifdef TEST_LOGIC
                PF_ASSERTION(sep == '+', "expected input", static_cast<int>(sep));
                #endif
                ist >> hour;
                ist >> sep;
                #ifdef TEST_LOGIC
                PF_ASSERTION(sep == '%', "expected input", static_cast<int>(sep));
                #endif
                ist >> sep;
                #ifdef TEST_LOGIC
                PF_ASSERTION(sep == '3', "expected input", static_cast<int>(sep));
                #endif
                ist >> sep;
                #ifdef TEST_LOGIC
                PF_ASSERTION(sep == 'A', "expected input", static_cast<int>(sep));
                #endif
                ist >> minute;
                ist >> sep;
                #ifdef TEST_LOGIC
                PF_ASSERTION(sep == '%', "expected input", static_cast<int>(sep));
                #endif
                ist >> sep;
                #ifdef TEST_LOGIC
                PF_ASSERTION(sep == '3', "expected input", static_cast<int>(sep));
                #endif
                ist >> sep;
                #ifdef TEST_LOGIC
                PF_ASSERTION(sep == 'A', "expected input", static_cast<int>(sep));
                #endif
                ist >> second;
            }

            // переводим в минуты начиная с 1-го января ORIGIN_YEAR года
            _startTime = minute + hour * MINUTES_IN_HOUR + Dates::toDay(year, month - 1, dayOfMonth - 1) * MINUTES_IN_DAY;
            char temp[100];
            sprintf(temp, "%04d-%02d-%02d", year, month, dayOfMonth);
            _dateOfStart = temp;
        }
        else if (keyword == transport)
        {
            int value(-1);
            ist >> value;
            _transport = (Transport::Type)(_transport | Transport::fromInt(value));
        }
        else if (keyword == boarding)
        {
            ist >> _boarding;
        }
        else if (keyword == minDelay)
        {
            ist >> _minDelay;
        }
        else if (keyword == maxDelay)
        {
            ist >> _maxDelay;
        }
        else if (keyword == maxTransfers)
        {
            ist >> _numOfChanges;
            if (_numOfChanges < 0 ||  _numOfChanges > MAX_CHANGES)
                _numOfChanges = MAX_CHANGES;
            _numOfChanges = 3;
        }
        else if (keyword == optimization)
        {
            std::getline(ist, tail);
            _optimizeTime = (tail == time);
        }
        else if (keyword == delta)
        {
            ist >> _delta;
        }
        else if (keyword == iLoveSport)
        {
            ist >> _iLoveSport;
        }
        else if (keyword == mode)
        {
            ist >> _mode;
        }
        else if (keyword == timeLimit)
        {
            ist >> _timeLimit;
        }
        else if (keyword == canChangeInStartTown)
        {
            ist >> _canChangeInStartTown;
        }
        else if (keyword == canChangeInFinishTown)
        {
            ist >> _canChangeInFinishTown;
        }
        else if (keyword == canChangeInAnyTown)
        {
            ist >> _canChangeInAnyTown;
        }
        else if (keyword == usePrice)
        {
            ist >> _usePrice;
        }
#ifdef DEBUG_ROUTE
        else if (keyword == debugStation)
        {
            int _debugStation;
            ist >> _debugStation;
            _debugStationIDs.push_back(_debugStation);
        }
        else if (keyword == debugRoute)
        {
            std::string _debugRoute;
            ist >> _debugRoute;
            _debugRoutes.push_back(_debugRoute);
        }
#endif
        else
        {
            // проверка, что ничего неизвестного не прочитали
            #ifdef TEST_LOGIC
            PF_ASSERTION(false, portion.c_str(), 0);
            #endif
            return false;
        }
        return true;
    }
}

