#include "path.h"

#include "stationtown.h"
#include "recode.h"
#include "dates.h"

namespace Pathfinder
{
    //логирование
    std::ostream& Path::toString(std::ostream& ost) const {
        return ost << *_lastVertex;
    }

    struct ResultSegment {
        Station departureStation, arrivalStation;
        int departureTime, arrivalTime;
        std::string departureTimeName, arrivalTimeName;
        std::string thread;
        double price;
        int discomfort;
        std::string discomfortString;

        void toXML(std::ostringstream& ost, const std::string & indent){
            ost << indent;
            ost << "<route ";
            ost << "price=\"" << price << "\" ";
            ost << "tr=\"" << discomfort
#ifdef DISCOMFORT_STR_ON
                    << " (" << discomfortString << ")"
#endif
                << "\" ";
            std::string startDate("1900-01-01");
            if (departureTime >= 0)
                startDate = departureTimeName.substr(0, 10);
            ost << "start_date=\"" << startDate << "\" ";

            ost << "thread_id=\"" << thread << "\" ";
            ost << "departure_datetime=\"" << departureTimeName << "\" ";
            ost << "departure_station_id=\"" << departureStation << "\" ";
            ost << "arrival_datetime=\"" << arrivalTimeName << "\" ";
            ost << "arrival_station_id=\"" << arrivalStation << "\" ";
            ost << "></route>\n";
        }
    };
    typedef std::vector<ResultSegment> ResultSegments;

    void getTimeName(std::string& res, const int time) {
        res = (time < 0 ? "void" : Dates::toString(time));
    }

    void getResultSegments(ResultSegments& resultSegments, const StationGraph& SG,
                           std::vector<const Vertex *>& flippedPath) {
        const int n = static_cast<int>(flippedPath.size()); //чтобы при вычитании не получилось очень большое число

        // пробегаем по пути, определяем когда происходит пересадка и записываем в ost
        const ThreadStart* predThread = 0;
        Station departureStation = flippedPath[n-1]->getEndStation();
        for (int i = n-2; i >= 0; i--) {
            const ThreadStart* thread = flippedPath[i]->getThreadStart();
            if (predThread != thread) {
                ResultSegment segment;
                segment.thread = (thread == 0 ? "NULL" : SG.getThreadOrigWithoutSubThread(thread->id));
                segment.departureStation = SG.st.getStationOrig(departureStation);
                segment.departureTime = flippedPath[i]->getBeginStationTime();
                getTimeName(segment.departureTimeName, segment.departureTime);

                //пропускаем непересадочные станции
                int j;
                for (j = i-1; j >= 0 && flippedPath[j]->getThreadStart() == thread; j--);
                const Vertex* arrivalVertex = flippedPath[j+1];

                segment.arrivalStation = SG.st.getStationOrig(arrivalVertex->getEndStation());
                segment.arrivalTime = arrivalVertex->getEndStationTime();
                getTimeName(segment.arrivalTimeName, segment.arrivalTime);


                segment.price = arrivalVertex->getSegmentPriceForPrint();
                if (segment.departureStation == segment.arrivalStation)
                    segment.price = 0;
                else if (thread && j >= 0)
                    segment.price = flippedPath[j]->getSegmentPriceForPrint();
                segment.discomfort = arrivalVertex->getObt().discomfort;
#ifdef DISCOMFORT_STR_ON
                segment.discomfortString = arrivalVertex->getObt().discomfortString;
#endif
                resultSegments.push_back(segment);
                departureStation = arrivalVertex->getEndStation();
            }
            predThread = thread;
        }
    }
    // форматируем маршрут в XML
    // indent - хранит отступ
    void Path::toXML(std::ostringstream& ost, const std::string & indent, const StationGraph& SG) const
    {
        ost << "\t\t<variant ";
        ost << "price=\"" << getObt().getPriceForPrint() << "\" ";
        ost << "time=\"" << getObt().weight << "\" ";
        ost << "tr=\"" << getObt().discomfort << "\"";
        ost << ">\n";

        ResultSegments resultsSegments;
        std::vector<const Vertex*> flippedPath;
        getFlippedPath(flippedPath);
        getResultSegments(resultsSegments, SG, flippedPath);
        for (ResultSegments::iterator it = resultsSegments.begin(); it != resultsSegments.end(); ++it)
            it->toXML(ost, indent);

        ost << "\t\t</variant>\n";
    }

    // форматируем маршрут в XML
    // indent - хранит отступ
    std::string Path::toXML(const std::string & indent, const StationGraph& SG) const
    {
        std::ostringstream ost;
        toXML(ost, indent, SG);
        return ost.str();
    }

    std::string Path::toDebugString(const std::string & indent, const StationGraph& SG) const
    {
        // обратный порядок меняем на прямой
        Vertex * it = _lastVertex;
        std::vector<Vertex *> flippedPath;
        while(it != 0)
        {
            flippedPath.push_back(it);
            it = it->getPrevious();
        }

        std::ostringstream ost;

        const int n = static_cast<int>(flippedPath.size());
        std::string sub;
        const std::string end = "\" ";

        double price = getObt().getPriceForPrint();
        int sTime = flippedPath.back()->getBeginStationTime();
        std::string sTimeName = Dates::toString(sTime);
        std::string sDate = sTimeName.substr(0, 10);
        ost << "\t\tprice=\"" << price << "\" time=\"" << getObt().weight << "\" tr=\"" << getObt().discomfort//trPoint 
            << "\" from=\"" << flippedPath.back()->getEndStationInfo()->name << "\" to=\"" << flippedPath[0]->getEndStationInfo()->name
            << "\" date=\"" << sTime << "\"\n";

        // пробегаем по пути, определяем когда происходит пересадка и записываем в ost
        const ThreadStart* predThread = 0;
        std::string predStation = flippedPath[n-1]->getEndStationInfo()->name;
        for (int i = n-2; i >= 0; i--)
        {
            Vertex* curr = flippedPath[i];
            const ThreadStart* thread = curr->getThreadStart();
            if (predThread != thread)
            {
                std::string departureStation = flippedPath[i+1]->getEndStationInfo()->name;
                int departureTime = curr->getBeginStationTime();
                std::string departureTimeName("void");
                std::string startDate("1900-01-01");
                if (departureTime >= 0)
                {
                    departureTimeName = Dates::toString(departureTime);
                    startDate = departureTimeName.substr(0, 10);
                }

                //пропускаем непересадочные станции
                int j;
                for (j = i-1; j >= 0 && flippedPath[j]->getThreadStart() == thread; j--);
                i = j+1;
                curr = flippedPath[i];

                ost << indent;
                ost << "price=\"" << curr->getObt().getPriceForPrint() << end;
                ost << "tr=\"" << curr->getObt().discomfort << end;
                ost << "time=\"" << curr->getObt().weight << end;
                ost << "thread=\"";
                if (thread == 0)
                    ost << "NULL";
                else
                    ost << SG.getThreadOrig(thread->id);
                ost << end;
                ost << "departure=\"" << departureStation << " " << departureTimeName << end;
            

                std::string arrivalStation = curr->getEndStationInfo()->name;
                predStation = arrivalStation;
                int arrivalTime = curr->getEndStationTime();
                std::string arrivalTimeName = (arrivalTime < 0 ? "void" : Dates::toString(arrivalTime));

                ost << "arrival=\"" << arrivalStation << " " << arrivalTimeName << end;
                ost << "\n";
            }
            predThread = thread;

        }
        ost << "\n";
        return ost.str();
    }

}

