#ifndef PATH_H
#define PATH_H

#include "vertex.h"
#include "estimation.h"

namespace Pathfinder
{
    class StationTown;

    // храним найденный маршрут (списком)
    class Path
    {
        friend class Result;

    //=---------------------------DATA
    private:
        // последний сегмент
        mutable Vertex* _lastVertex;

    //=---------------------------STRUCTORs

    public:
        // конструктор
        Path(Vertex* v = 0):_lastVertex(v)
        {}

        Path(const Path & path):_lastVertex(path._lastVertex)
        {}

        // оператор присваивания
        Path & operator= (const Path & path)
        {
            _lastVertex = path._lastVertex;
            return *this;
        }

        // делаем swap
        void swap(Path & path)
        {
            std::swap(path._lastVertex, _lastVertex);
        }

        Path copy()
        {
            Path res;
            res._lastVertex = _lastVertex;
//			_lastVertex->increaseAllReferences();
            return res;
        }

        void changeLastVertex(Vertex* v)
        {
            _lastVertex = v;
        }

        // пустой?
        inline bool empty() const { return _lastVertex == 0; }

//	private:
        // уменьшаем количество ссылок для всех вершин пути (при удалении пути)
        void withdraw()
        {
            _lastVertex->decreaseAllReferences();
            _lastVertex = 0;
        }


    //=---------------------------SELECTORs

    public:
        // метод сравнения, основанный на времени в пути
        inline static bool cmpByObt(const Path & path0, const Path & path1)
        {
            return Vertex::cmpByObt(*path0._lastVertex, *path1._lastVertex);
        }
        // метод сравнения, основанный на количестве пересадок
        inline static bool cmpByChanges(const Path & path0, const Path & path1)
        {
            return Vertex::cmpByChanges(*path0._lastVertex, *path1._lastVertex);
        }
        inline static bool cmpByChangesAndTime(const Path & path0, const Path & path1)
        {
            if (path0._lastVertex == 0 && path1._lastVertex == 0)
                return false;
            if (path0._lastVertex != 0 && path1._lastVertex == 0)
                return true;
            if (path0._lastVertex == 0 && path1._lastVertex != 0)
                return false;
            return Vertex::cmpByChanges(*path0._lastVertex, *path1._lastVertex);
        }

        // сейчас сравниваем так
        inline bool operator<(const Path& wp) const
        {
            return (*_lastVertex) < (*wp._lastVertex);
        }
        inline static bool cmp(const Path & path0, const Path & path1)
        {
            return Vertex::cmp(*path0._lastVertex, *path1._lastVertex);//(*path0._lastVertex) < (*path1._lastVertex);
        }
        inline static bool cmpByEst(const Path & path0, const Path & path1)
        {
            return Vertex::cmpByEst(*path0._lastVertex, *path1._lastVertex);
        }

        // ----------------- методы доступа: ---------------------------
    public:

        inline Vertex* getLastVertex() const { return _lastVertex; }
        inline const ThreadStart* getLastThread() const { return _lastVertex->getThreadStart(); }
        inline const TripInfo* getLastTripInfo() const { return _lastVertex->getTripInfo(); }
        inline int getNumberOfChanges() const { return _lastVertex->getObt().changes; }
        inline int getWeight() const { return _lastVertex->getObt().weight; }
        inline bool growable() const { return _lastVertex->getGrowable(); }

        inline const Estimation& getObt() const {return _lastVertex->getObt(); }
        inline Estimation& getObtForChange() { return _lastVertex->getObtForChange(); }
        inline const Estimation& getEst() const {return _lastVertex->getEst(); }
        void setZeroEst()
        {
            _lastVertex->setZeroEst();
        }
        inline Estimation getTotalEst() const {return _lastVertex->getTotalEst(); }

        // вернуть время прибытия на последнюю станцию
        inline int getLastTime() const
        {
            return _lastVertex->getEndStationTime();
        }
        // вернуть время прибытия на последнюю станцию
        inline int getLastDay() const
        {
            return _lastVertex->getEndStationTime() / MINUTES_IN_DAY;
        }
        // вернуть время в пути (вместе с остановками)
        int getTotalTime() const
        {
            return _lastVertex->getTotalTime();
        }

        //собирает все транспорты из пути
        Transport::Type allTransports() const
        {
            return _lastVertex->allTransports();
        }

        // проходит путь через станцию station?
        inline bool containStation(const StationInfo* si) const
        {
            return _lastVertex->containStation(si);
        }

        // проходит путь хотя бы через одну из станций?
        bool containAnyStation(const CostationsSet& stations) const
        {
            return _lastVertex->containStations(stations);
        }

        // проходит путь хотя бы через одну из станций?
        bool containAnyStationExceptThread(const CostationsSet& stations, const ThreadStart* thread) const
        {
            return _lastVertex->containAnyStationExceptThread(stations, thread);
        }

        // содержит путь нитку thread?
        bool containThread(const std::string& threadName) const
        {
            return _lastVertex->containThread(threadName);
        }

        inline bool lastVertexIsTeleportation()
        {
            if (!getLastVertex()->getPrevious())
                return false;
            Station station = getLastVertex()->getEndStation();
            Station predStation = getLastVertex()->getPrevious()->getEndStation();
            return station!=predStation;
        }

        // проверка корректности
        bool isValid() const { return true; }

        const ThreadStart* getPredThread() const
        {
            const ThreadStart* thread = getLastVertex()->getThreadStart();
            Vertex* v = _lastVertex;
            for (; v && (v->getThreadStart()==0 || v->getThreadStart()==thread); v=v->getPrevious());
            if (v)
                return v->getThreadStart();
            return 0;
        }
        const Vertex* getStartVertex() const
        {
            const ThreadStart* thread = getLastVertex()->getThreadStart();
            Vertex* v = _lastVertex;
            for (; v && v->getPrevious() && v->getPrevious()->getThreadStart()==thread; v=v->getPrevious());
            if (v)
                return v;
            return 0;
        }

        //=---------------------------MODIFIERs

    public:
        // добавить вершину в путь и оценку снизу на оставшееся расстояние
        void add(Vertex * vertex)
        {
            _lastVertex = _lastVertex->addVertex(vertex);
            _lastVertex->calcObt();
        }

        // форматируем маршрут в XML
        void toXML(std::ostringstream& ost, const std::string & indent, const StationGraph& SG) const;
        std::string toXML(const std::string& indent, const StationGraph& SG) const;
        std::string toDebugString(const std::string & indent, const StationGraph& SG) const;
        std::ostream& toString(std::ostream& ost) const;

    private:
        // обратный порядок меняем на прямой
        void getFlippedPath(std::vector<const Vertex*>& flippedPath) const {
            const Vertex * it = _lastVertex;
            while (it) {
                flippedPath.push_back(it);
                it = it->getPrevious();
            }
        }
    };

    typedef std::vector<Path> Pathes;
}
#endif

