#ifndef _PATH_STORAGE_H_
#define _PATH_STORAGE_H_

#include "tripinfo.h"
#include "vertex.h"
#include "path.h"
#include "priority_queue.h"
#include "result.h"
#include "condition.h"
#include "tariff.h"

namespace Pathfinder
{

    //Хранилище всех кусочков всех путей
    typedef std::vector<Vertex> StorageVertex;

    // Фронт строящихся путей
    typedef priority_queue<Path> PathFront;

    //Короткое ребро + день. Однозначно определяет время и место прибытия, а также на чем едем.
    typedef std::vector<Vertex*> VertexList;

    struct GrowableItem
    {
        Vertex* vertex;
        const TripInfo* badTI;
        GrowableItem() : vertex(0), badTI(0) {}
    };
    typedef std::vector<GrowableItem> GrowableList;

    class BestPathsList
    {
        VertexList v;
        int deletedNumber;

    public:
        BestPathsList()
            : deletedNumber(0)
        {}
    
        void clear()
        {
            v.clear();
            deletedNumber = 0;
        }

        int getVertexNumber()
        {
            return v.size();
        }
        int getDeletedNumber()
        {
            return deletedNumber;
        }
        int getGoodVertexNumber()
        {
            int res = 0;
            for (VertexList::iterator it = v.begin(); it != v.end(); ++it)
                if ((*it)->getMakeLonger())
                    res++;
            return res;
        }
        int getBadVertexNumber()
        {
            int res = 0;
            for (VertexList::iterator it = v.begin(); it != v.end(); ++it)
                if (!(*it)->getMakeLonger())
                    res++;
            return res;
        }

        void getGrowableList(GrowableList& res)
        {
            VertexList::iterator itEnd = v.end();
            for (VertexList::iterator it = v.begin(); it != itEnd; ++it)
                if ((*it)->getGrowable() && ((*it)->getMakeLonger()))
                {
                    res.push_back(GrowableItem());
                    res.back().vertex = *it;
                }
        }

        const VertexList& getVertexList()
        {
            return v;
        }

        const Vertex* getVertex(const size_t i)
        {
            return v[i];
        }

        //путь не стоит пытаться продолжать, если есть 
        //путь через этот же путевой сегмент с лучшими характеристиками (намного лучшими)
        bool checkVertex(Vertex* vertex)
        {
            Estimation estV = vertex->getObt();
            bool used = false;
            bool deleted = false;
            VertexList::iterator itEnd = v.end();
            for (VertexList::iterator it = v.begin(); it != itEnd; ++it)
            {
                Vertex* curr = *it;
                if (curr == vertex)
                {
                    used = true;
                    break;
                }
                Estimation estCurr = curr->getObt();
                Estimation::BetterItem better = estV.better4(estCurr);
                if (better == Estimation::BETTER)
                {
                    if (curr->getMakeLonger())
                        deleted = true;
                    curr->setMakeLonger(false);
                }
                else if (better & (Estimation::WORSE | Estimation::ANY))
                {
                    vertex->setMakeLonger(false);
                }
            }
            if (!used && vertex->getMakeLonger())
                v.push_back(vertex);
            if (deleted)
            {
                int j = 0;
                for (int i = 0; i < v.size(); i++)
                    if (v[i]->getMakeLonger())
                    {
                        v[j] = v[i];
                        j++;
                    }
                deletedNumber += v.size() - j;
                v.resize(j);
            }
            return vertex->getMakeLonger();
        }

        //путь не стоит пытаться продолжать, если есть 
        //путь через этот же путевой сегмент с лучшими характеристиками (намного лучшими)
        bool checkVertex(Vertex* vertex, const StationInfoPtrs& debugStations, const std::vector<const Route*>& debugRoutes)
        {
            Estimation estV = vertex->getObt();
            bool used = false;
            bool deleted = false;
            VertexList::iterator itEnd = v.end();
            for (VertexList::iterator it = v.begin(); it != itEnd; ++it)
            {
                Vertex* curr = *it;
                if (curr == vertex)
                {
                    used = true;
                    break;
                }
                Estimation estCurr = curr->getObt();
                Estimation::BetterItem better = estV.better4(estCurr);
                if (better == Estimation::BETTER)
                {
                    if (curr->getMakeLonger())
                        deleted = true;
                    curr->setMakeLonger(false);
                }
                else if (better & (Estimation::WORSE | Estimation::ANY))
                {
                    vertex->setMakeLonger(false);
#ifdef DEBUG_ROUTE
                    bool fl = vertex->isDebugingThread(debugStations, debugRoutes);
                    if (fl)
                    {
                        std::fstream out(DEBUG_LOG_FILE, std::ios::out | std::ios::app);
                        out << "LOC_TR check false path " << *vertex;
                        out << " by " << *curr;
                        out << std::endl;
                    }
#endif
                }
                
            }
            if (!used && vertex->getMakeLonger())
                v.push_back(vertex);
            if (deleted)
            {
                int j = 0;
                for (int i = 0; i < v.size(); i++)
                    if (v[i]->getMakeLonger())
                    {
                        v[j] = v[i];
                        j++;
                    }
                deletedNumber += v.size() - j;
                v.resize(j);
            }
            return vertex->getMakeLonger();
        }

    };

    //Список концов путей с концом на нашем коротком ребре в конкретный день
    struct StationCalcInfo
    {
        const StationInfo* si;
        BestPathsList bestPaths;
        Estimation est;
        Estimation estAndGo;
        int front;
        StationCalcInfo()
            : si(0), front(10000)
        {}
        void clear()
        {
            bestPaths.clear();
            est = Estimation();
            estAndGo = Estimation();
            front = 10000;
        }
        int getEstChange()
        {
            if (est.empty())
                return -1;
            return est.changes;
        }
        
        int getMinBeginStationTime(const GrowableList& growableList) const
        {
            int res = growableList[0].vertex->getBeginStationTime();
            GrowableList::const_iterator itEnd = growableList.end(); 
            for (GrowableList::const_iterator it = growableList.begin()+1; it != itEnd; ++it)
                res = std::min(res, it->vertex->getBeginStationTime());
            return res;
        }
        int getMaxEndStationTime(const GrowableList& growableList) const
        {
            int res = growableList[0].vertex->getEndStationTime();
            GrowableList::const_iterator itEnd = growableList.end(); 
            for (GrowableList::const_iterator it = growableList.begin()+1; it != itEnd; ++it)
                res = std::max(res, it->vertex->getEndStationTime());
            return res;
        }

    };
    typedef std::vector<StationCalcInfo> Station2Info;

    struct TripInfoCalcInfo
    {
        const TripInfo* ti;
        BestPathsList bestPaths;
        //если мы только что прошли по этому ребру, сколько еще осталось идти???
        Estimation est;
        int front;
        TripInfoCalcInfo()
            : ti(0), front(10000)
        {}
        void clear()
        {
            bestPaths.clear();
            est = Estimation();
            front = 10000;
        }
        int getEstChange()
        {
            if (est.empty())
                return -1;
            return est.changes;
        }

    };
    typedef std::vector<TripInfoCalcInfo> Edge2Info;


    namespace PFResult
    {
        enum Type
        {
            ok                       = 0,
            bad_input                = 1,
            trivial                  = 2,
            impossible               = 3,
            long_way                 = 4,
            mem_vertex               = 5,
            mem_path                 = 6,
            norm                     = 7,
            prolong_teleportation    = 8,
            bad_path                 = 9,
            result                   = 10,
            max_wait                 = 11,
            max_res                  = 12,
            max_time                 = 13,
            max_w_sc                 = 14
        };

        inline std::string toString(const Type r)
        {
            switch (r)
            {
                case ok: return "ok";
                case bad_input: return "bad-input";
                case trivial: return "trivial";
                case impossible: return "impossible";
                case long_way: return "long-way";
                case mem_vertex: return "mem-vertex";
                case mem_path: return "mem-path";
                case norm: return "norm";
                case prolong_teleportation: return "prolong-teleportation";
                case bad_path: return "bad-path";
                case max_wait: return "max-wait";
                case max_res: return "max-res";
                case max_time: return "max-time";
                case max_w_sc: return "max-w-sc";
                case result: return "result";
            }
            return "unknown";
        }
    }

    namespace PFMode
    {
        enum Mode
        {
            unknown    = 0,
            common     = 1,
            wizard     = 2,
            many_days  = 3
        };

        inline Mode fromString(const std::string& s)
        {
            if (s == "many_days")
                return many_days;
            else if (s == "wizard")
                return wizard;
            return common;
        }
        inline std::string toString(const Mode r)
        {
            switch (r)
            {
                case common: return "common";
                case wizard: return "wizard";
                case many_days: return "many_days";
                default: return "unknown";
            }
            return "unknown";
        }
    }

    class PathStorage
    {
    public:
        StorageVertex _storageVertex;
        PathFront _pathFront;
        Edge2Info _edge2Info;
        Station2Info _station2Info;
        Result _result;
        TariffStorage _tariffStorage;

        // характеристики поиска
        // максимальное количество путей во фронте
        int _maxPathNumber;
        // сколько было путей во фронте, когда нашли последний маршрут
        int _lastUseful;
        // сколько было путей
        int _allPathNumber;
        // максимально-возможные времена работы на разных этапах
        PFMode::Mode _mode;
        int _timeLimitInit;
        int _timeLimitFirst;
        int _timeLimit;
        // прерывались по MAX_TIME на каком-то из этапов
        bool _executionWasStoppedByTimeLimit;

        size_t _stationNumber;
        size_t _tripInfoNumber;

        Timer _timer;

#ifdef DEBUG_ROUTE
        bool _needDebugLog;
        StationInfoPtrs _debugStations;
        std::vector<const Route*> _debugRoutes;
#endif

        PathStorage()
        {
            _storageVertex.resize(MAX_NUM_OF_VERTICES);
            _stationNumber = 0;
            _tripInfoNumber = 0;
#ifdef DEBUG_ROUTE
            _needDebugLog = false;
#endif
        }
        void startWork(const std::string& mode, const int newTimeLimit)
        {
            _mode = PFMode::fromString(mode);
            if (_mode == PFMode::common)
            {
                _timeLimitInit    = 440000; //в микросекундах
                _timeLimitFirst   = 470000;
                _timeLimit        = 490000;
            }
            else if (_mode == PFMode::wizard)
            {
                _timeLimitInit    = 32000;
                _timeLimitFirst   = 36000;
                _timeLimit        = 39000;
            }
            else if (_mode == PFMode::many_days)
            {
                _timeLimitInit    = 500000;
                _timeLimitFirst   = 700000;
                _timeLimit        = 990000;
            }
            if (newTimeLimit > 0) //пропорционально изменяем все времена
            {
                _timeLimitInit = (int)((double)_timeLimitInit*newTimeLimit/_timeLimit);
                _timeLimitFirst = (int)((double)_timeLimitFirst*newTimeLimit/_timeLimit);
                _timeLimit= newTimeLimit;
            }
            _timer.stop();
            _timer.start();
        }
        void clear()
        {
            _maxPathNumber = 0;
            _lastUseful = 0;
            _allPathNumber = 0;
            _executionWasStoppedByTimeLimit = false;
            _storageVertex.clear();
            _pathFront.clear();
            for (int i = 0; i < _station2Info.size(); i++)
                _station2Info[i].clear();
            _result.clear();
            _finishThreads.clear();
            _startThreads.clear();
            for (int i = 0; i < _edge2Info.size(); i++)
                _edge2Info[i].clear();
#ifdef DEBUG_ROUTE
            _debugStations.clear();
            _debugRoutes.clear();
            _needDebugLog = false;
#endif
        }
        void setStationNumber(size_t sn, size_t tin, const StationInfos& stationInfos, const TripInfos& tripInfos)
        {
            bool flag = (_stationNumber!=sn || _tripInfoNumber!=tin);
            _stationNumber = sn;
            _tripInfoNumber = tin;
            if (flag)
                clear();
            _station2Info.resize(_stationNumber);
            for (int i = 0; i < _station2Info.size(); i++)
                _station2Info[i].si = &stationInfos[i];
            _edge2Info.resize(_tripInfoNumber);
            for (int i = 0; i < _edge2Info.size(); i++)
                _edge2Info[i].ti = &tripInfos[i];
        }

        inline Vertex* getFreeVertex()
        {
            if (_storageVertex.size() >= MAX_NUM_OF_VERTICES)
            {
                return 0;
            }
            _storageVertex.resize(_storageVertex.size()+1);
            return &_storageVertex.back();
        }

        inline void updateLastUseful()
        {
            _lastUseful = _allPathNumber;
        }

        inline void updateMaxPathNumber()
        {
            _maxPathNumber = std::max(_maxPathNumber, (int)_pathFront.size());
        }

        inline void stopExecutionByTimeLimit()
        {
            _executionWasStoppedByTimeLimit = true;
        }
        inline bool wasExecutionStoppedByTimeLimit()
        {
            return _executionWasStoppedByTimeLimit;
        }

        bool addPathToFront(Path& path)
        {
            if (_pathFront.size() >= MAX_NUM_OF_PATHS)
                return false;
#ifdef DEBUG_PRINT
if (path.getObt().weight < 0)
{
    std::fstream out(DEBUG_LOG_FILE, std::ios::out | std::ios::app);
    out << "Weight ERROR! w < 0" << std::endl;
}
#endif
            assert(path.getObt().weight >= 0);
            
            path.getLastVertex()->increaseAllReferences();
            _pathFront.push(path, Path::cmp);
            updateMaxPathNumber();
            ++_allPathNumber;

            return true;
        }

        //обновляем _edge2Info для новых вешин пути и проверяем существующие
        bool checkPath(const Path& path)
        {
#ifdef DEBUG_ROUTE
            bool fl = false;
            if (this->_needDebugLog)
            {
                fl = path.getLastVertex()->isDebugingThread(this->_debugStations, this->_debugRoutes);
            }
#endif
            bool makeLonger = true;
            Vertex* predv = 0;
            for (Vertex* v = path.getLastVertex(); v; v = v->getPrevious())
            {
                if (v->getTripInfo()) //ездовое ребро
                {
                    if (predv==0)
#ifdef DEBUG_ROUTE
                        if (fl)
                            makeLonger &= _edge2Info[v->getTripInfo()->id].bestPaths.checkVertex(v, _debugStations, _debugRoutes);
                        else
#endif
                            makeLonger &= _edge2Info[v->getTripInfo()->id].bestPaths.checkVertex(v);
                }
                else //телепортационное ребро
                {
#ifdef DEBUG_ROUTE
                        if (fl)
                            makeLonger &= _station2Info[v->getEndStation()].bestPaths.checkVertex(v, _debugStations, _debugRoutes);                    
                        else
#endif
                            makeLonger &= _station2Info[v->getEndStation()].bestPaths.checkVertex(v);                    
                }
                if (predv==0)
                    break;
                predv = v;
            }
#ifdef DEBUG_ROUTE
            if (this->_needDebugLog)
            {
                bool fl = path.getLastVertex()->isDebugingThread(this->_debugStations, this->_debugRoutes);
                if (fl)
                {
                    std::fstream out(DEBUG_LOG_FILE, std::ios::out | std::ios::app);
                    out << "LOC_TR check " << "path";
                    path.toString(out);
                }
                if (fl)
                {
                    std::fstream out(DEBUG_LOG_FILE, std::ios::out | std::ios::app);
                    out << "makeLonger=" << makeLonger << std::endl;
                }
            }
#endif
            return makeLonger;
        }

        // выкидываем путь и, возможно, уточняем отсечения
        void erasePath(Path& path)
        {
            //если так случилось, что этот путь закончен и он не привел к реализации какого-либо результата
            for (Vertex* v = path.getLastVertex(); v && v->getNumberOfReferences() == 1; v = v->getPrevious())
            {
                v->setMakeLonger(false);
                if (v->getTripInfo())
                {
#ifdef DEBUG_ROUTE
                    if (_needDebugLog)
                        _edge2Info[v->getTripInfo()->id].bestPaths.checkVertex(v, _debugStations, _debugRoutes);
                    else
#endif
                        _edge2Info[v->getTripInfo()->id].bestPaths.checkVertex(v);
                }
                else
                {
#ifdef DEBUG_ROUTE
                    if (_needDebugLog)
                        _station2Info[v->getEndStation()].bestPaths.checkVertex(v, _debugStations, _debugRoutes);
                    else
#endif
                        _station2Info[v->getEndStation()].bestPaths.checkVertex(v);
                }
            }
            path.withdraw();
        }

        bool isValid()
        {
            return _storageVertex.capacity() >= MAX_NUM_OF_VERTICES;
        }

        bool goodPath(const Path &path, const int changes)
        {
            const Vertex* end = path.getLastVertex();
            if (!end->checkMakeLonger(changes) )  //MAX_CHANGES
                return false;
            if ( !_result.checkByResultTotalEst(path) )
                return false;
            if ( !checkPath(path) )
                return false;
            return true;
        }

        int addPathToResult(Path& path, const Condition & condition)
        {
            // добавляем путь к ответу
            _result.addPathCopy(path);
            updateLastUseful();
            return true;
        }

        PFResult::Type checkMaxConditions()
        {
            // обновляем максимальное количество путей во фронте
            updateMaxPathNumber();

            // если путей перебрали много и ничего не нашли, то выходим
            if (_allPathNumber > MAX_WAIT && _result.empty())
                return PFResult::max_wait;

            _timer.stop();
            if (_timer.getTime() > _timeLimit) {
                stopExecutionByTimeLimit();
                return PFResult::max_time;
            }

            // порог "усталости":
            // если путей перебрали много и уже что-то нашли, то выходим
            if (_allPathNumber > MAX_WAIT_SUCCESS && (_result.empty() == false))
                return PFResult::max_w_sc;

            // если путей нашли много, то выходим
            if (_result.size() > MAX_RES)
                return PFResult::max_res;

            return PFResult::ok;
        }

        void pathFrontPop()
        {
            _pathFront.pop(Path::cmp);
        }

        typedef std::pair<const ThreadStart*, const TripInfo*> Thread2TI;
        typedef std::multimap<const ThreadStart*, const TripInfo*> FinishThreads;
        typedef std::pair<FinishThreads::const_iterator, FinishThreads::const_iterator> FIIterator;
        FinishThreads _finishThreads;
        FinishThreads _startThreads;

        bool add2finishThreads(const ThreadStart* thread, const TripInfo* ti)
        {
            FIIterator fiit = _finishThreads.equal_range(thread);
            if (fiit.first == _finishThreads.end())
                _finishThreads.insert( Thread2TI(thread, ti) );
            else
            {
                int maj = fiit.first->second->stationInfoTo->majority;
                if (ti->stationInfoTo->majority == maj)
                    _finishThreads.insert( Thread2TI(thread, ti) );
                else if (ti->stationInfoTo->majority < maj)
                {
                    _finishThreads.erase(thread);
                    _finishThreads.insert( Thread2TI(thread, ti) );
                }
                else 
                    return false;
            }
            return true;
        }
        const TripInfo* getFinishThread(const ThreadStart* thread, const TripInfo* tiAfter) const
        {
            FIIterator fiit = _finishThreads.equal_range(thread);
            const TripInfo* res = 0;
            for (FinishThreads::const_iterator it = fiit.first; it != fiit.second; ++it)
                if (it->second > tiAfter && (res==0 || it->second < res))
                    res = it->second;
            return res;
        }
        bool add2startThreads(const ThreadStart* thread, const TripInfo* ti)
        {
            FIIterator fiit = _startThreads.equal_range(thread);
            if (fiit.first == _startThreads.end())
            {
                _startThreads.insert( Thread2TI(thread, ti) );
                return true;
            }
            else
            {
                int maj = fiit.first->second->stationInfoFrom->majority;
                if (ti->stationInfoFrom->majority == maj)
                {
                    _startThreads.insert( Thread2TI(thread, ti) );
                    return true;
                }
                else if (ti->stationInfoFrom->majority < maj)
                {
                    _startThreads.erase(thread);
                    _startThreads.insert( Thread2TI(thread, ti) );
                    return true;
                }
            }
            return false;
        }
        const TripInfo* getStartThread(const ThreadStart* thread, const TripInfo* tiBefore) const
        {
            FIIterator fiit = _startThreads.equal_range(thread);
            const TripInfo* res = 0;
            for (FinishThreads::const_iterator it = fiit.first; it != fiit.second; ++it)
                if (it->second < tiBefore && (res==0 || it->second > res))
                    res = it->second;
            return res;
        }

    public:
        // инициализация массива оценок снизу на вес маршрута до станции назначения на указанных типах транспорта
        int initEstimations(const StationGraph& sg, const Condition & condition, const Transport::Type& transport
            , const size_t firstBegDay, const size_t lastBegDay, const size_t lastTripDay);

        inline const Estimation& getStationEst(const Station& station) const
        {
            return _station2Info[station].est;
        }
        inline const Estimation& getStationEst(const StationInfo* si) const
        {
            return _station2Info[si->id1].est;
        }
        inline const Estimation& getStationEstAndGo(const StationInfo* si) const
        {
            return _station2Info[si->id1].estAndGo;
        }
        inline Estimation getMinStationEst(const StationInfoPtrs& stations) const
        {
            Estimation res;
            StationInfoPtrs::const_iterator itEnd = stations.end();
            for (StationInfoPtrs::const_iterator it = stations.begin(); it != itEnd; ++it)
                res = std::min(res, _station2Info[(*it)->id1].est);
            return res;
        }
        inline const Estimation& getTripInfoEst(const size_t id) const
        {
            return _edge2Info[id].est;
        }

        inline const Estimation& getTripInfoEst(const TripInfo* tripinfo) const
        {
            return _edge2Info[tripinfo->id].est;
        }
        inline Estimation& setTripInfoEst(const TripInfo* tripinfo, const Estimation& est)
        {
            return _edge2Info[tripinfo->id].est = est;
        }

        void searchDebugOutput(FILE* out)
        {
            int goodS = 0, badS = 0, deletedS = 0;
            for (Station2Info::iterator it = _station2Info.begin(); it != _station2Info.end(); ++it)
                if (it->bestPaths.getVertexNumber())
                {
                    int good = it->bestPaths.getGoodVertexNumber();
                    int bad = it->bestPaths.getBadVertexNumber();
                    int deleted = it->bestPaths.getDeletedNumber();
                    fprintf(out, "%d\t%s\t%d\t%d\t%d\n", it->si->id, it->si->name.c_str()
                        , good, bad, deleted);
                    goodS += good;
                    badS += bad;
                    deletedS += deleted;
                }
            int goodE = 0, badE = 0, deletedE = 0;
            for (Edge2Info::iterator it = _edge2Info.begin(); it != _edge2Info.end(); ++it)
                if (it->bestPaths.getVertexNumber())
                {
                    int good = it->bestPaths.getGoodVertexNumber();
                    int bad = it->bestPaths.getBadVertexNumber();
                    int deleted = it->bestPaths.getDeletedNumber();
                    fprintf(out, "%s\t%s\t%s\t%d\t%d\t%d\n", it->ti->getName().c_str()
                        , it->ti->stationInfoFrom->name.c_str(),  it->ti->stationInfoTo->name.c_str()
                        , good, bad, deleted);
                    goodE += good;
                    badE += bad;
                    deletedE += deleted;
                }        
            
        }

    private:
        // сохраняем вычисленные оценки
        void _saveEstimations(const char * filePath) const;
    };

}

#endif
