#include <string.h>
#include <assert.h>

#include "stationgraph.h"

namespace Pathfinder
{
    inline bool isFirstStationInThread(const std::string& threadUid, const ThreadStarts& _threadStarts)
    {
        return _threadStarts.empty() || _threadStarts.back().name != threadUid;
    }

    // определяет избыточные нитки МЦК
    inline bool isExcessMCZDthread(const std::string& threadUid, const int stationID)
    {
        if (!startWith(threadUid, std::string("MCZK_")))
            return false;

        return stationID != StationInfo::nizegorodskayaStationId && stationID != StationInfo::streshnevoStationId;
    }

    bool StationGraph::lastThreadIsSingleEdge() const
    {
        if (_tripInfos.size() <= 1)
            return true;
        const TripInfo& ti = *(_tripInfos.end()-1);
        const TripInfo& predTI = *(_tripInfos.end()-2);
        return ti.thread != predTI.thread;
    }

    void StationGraph::clearPredThreadTail()
    {
        if (_tripInfos.empty())
            return;

        TripInfo& ti = _tripInfos.back();
        if (ti.stationInfoTo == 0 || ti.duration < 0) {
            if (lastThreadIsSingleEdge())
                _threadStart.pop_back();
            _tripInfos.pop_back();
        }
    }

    bool StationGraph::loadTripInfo(std::string buffer, TripInfo &tripinfo, std::string &threadUid, std::string &route,
                                    const std::string &LOG_FILE, const int count, int &mainThread)
    {
        // станция 0    станция 1         станция 2
        //    >------------>=================>-------

        // структура файла с данными имеет следующий формат:
        // station1_id    - станция отправления
        // station2_id    - станция назначения
        // is_technical_stop - является ли остановка на станции 1 технической
        // thread_uid      - идентификатор нитки
        // departure_time - время отправления (по Москве) со станции 1
        // stop_time      - время стоянки на станции 1
        // duration       - время в пути
        // route_id       - номер рейса (текстовый, с русскими буквами)
        // t_type_id      - идентификатор типа транспорта
        // year_days      - маска хождения транспорта по Москве (длина маски равна 366=DAYS_IN_YEAR_LEAP)

        // станции отправления и назначения
        Station fromOrig(-1), toOrig(-1);
        // время отправления: час, минута, секунда
        int hour(-1), minute(-1), second(-1);
        // время в пути (прибытие на станцию 2 минус отправление со станции 1)
        int duration(-1), stoptime(-1), is_technical_stop(0);
        // тип транспорта
        int transport(-2);
        // маска хождения транспорта по Москве
        std::string yearMask;

        char cthread[200], croute[200], cyearMask[800];

        int err = sscanf(buffer.c_str(), "%d %d %d %s %d:%d:%d %d %d %[^\t] %d %d %s"
                , &fromOrig, &toOrig
                , &is_technical_stop, cthread
                , &hour, &minute, &second
                , &stoptime, &duration
                , croute, &transport, &mainThread, cyearMask);
        if (err != 13) {
            message(LOG_FILE, "thegraph reading ERROR with line: ", count);
            return false;
        }
        threadUid = cthread;
        route = croute;
        yearMask = cyearMask;

        // инициализируем маску дней хождения
        Departures times;
        times.initGoodDayMask(yearMask);

        // начинаем строить ребро
        // устанавливаем время
        times.setTime(hour, minute, second);
        tripinfo.departures = times;
        tripinfo.duration = duration;
        tripinfo.stopTimeBefore = stoptime;
        tripinfo.is_technical_stop = (bool)is_technical_stop;

        // заполняем
        tripinfo.transport = Transport::fromInt(transport);
        tripinfo.id = _tripInfos.size();

        Station from = st.getStationInnerID(fromOrig);
        Station to = st.getStationInnerID(toOrig);
        if (from >= 0)
            tripinfo.stationInfoFrom = &st.stationInfos[from];
        if (to >= 0)
            tripinfo.stationInfoTo = &st.stationInfos[to];

        return true;
    }

    bool StationGraph::checkTripInfo(const std::string &LOG_FILE, TripInfo &tripinfo, std::string &threadUid,
                                     const int count, int &durationErr, int &stoptimeErr)
    {
        bool loadedCorrect = true;
        if (tripinfo.duration < 0)
        {
            durationErr++;
            loadedCorrect = false;
            message(LOG_FILE, "strange threadUid (duration < 0): ",  threadUid);
        }
        if (tripinfo.stopTimeBefore < 0)
        {
            stoptimeErr++;
            loadedCorrect = false;
            message(LOG_FILE, "strange threadUid (stoptime < 0): ",  threadUid);
        }
        if (tripinfo.stationInfoFrom == 0 || tripinfo.stationInfoTo == 0)
        {
            message(LOG_FILE, "thegraph reading ERROR (from < 0 || to < 0) with line: ", count);
            loadedCorrect = false;
        }

        return loadedCorrect;
    }

    bool StationGraph::checkPredTripInfo(const std::string &LOG_FILE, TripInfo &ti, TripInfo &predTI,
                                         const int count, int &depTimeErr) {
        if ((predTI.departures.getTime() + predTI.duration + ti.stopTimeBefore) % (24 * 60) != ti.departures.getTime()) {
            message(LOG_FILE, "thegraph reading ERROR (depTime) withline: ", count);
            depTimeErr++;
            return false;
        }
        return true;
    }


    int StationGraph::load(const std::string& LOG_FILE, const char* filePath, const Transport::Type tr, const bool forRailway)
    {
        std::ifstream fin(filePath);
        if (fin.fail())
            return -1;

        //статистика ошибок
        int durationErr = 0, stoptimeErr = 0, depTimeErr = 0, mainThread = 0;

        // читаем
        std::string s;
        int count(0);
        std::string excessMCZDthreadUID;
        for (count = 0; std::getline(fin, s); count++)
        {
            TripInfo tripinfo;
            // идентификатор нитки
            std::string threadUid, route;

            if (!loadTripInfo(s, tripinfo, threadUid, route, LOG_FILE, count, mainThread))
                continue;

            if (!(tripinfo.transport & tr))
                continue;

            bool loadedCorrect = checkTripInfo(LOG_FILE, tripinfo, threadUid, count, durationErr, stoptimeErr);

            bool is_first_station_in_thread = isFirstStationInThread(threadUid, _threadStart);

            if (!forRailway) {
                if (!is_first_station_in_thread) {
                    TripInfo& predTI = _tripInfos.back();

                    loadedCorrect &= checkPredTripInfo(LOG_FILE, tripinfo, predTI, count, depTimeErr);
                    _threadStart.back().loadedCorrect &= loadedCorrect;

                    // если с нашим ребром что-то не так - отрицательное время стоянки или в пути, или нет остановки,
                    // или у предыдущего ребра отрицательное время в пути, или неизвестная станция To,
                    // то клеим наше ребро к предыдущему
                    if (!loadedCorrect || (tripinfo.stopTimeBefore == 0 && tripinfo.transport != Transport::BUS)
                        || tripinfo.is_technical_stop || predTI.duration < 0 || predTI.stationInfoTo == 0)
                    {
                        predTI.stationInfoTo = tripinfo.stationInfoTo;
                        st.stationInfos[tripinfo.stationInfoTo->id1].transports = (Transport::Type)(predTI.stationInfoTo->transports | tripinfo.transport);
                        predTI.duration += tripinfo.duration;
                        predTI.duration += tripinfo.stopTimeBefore;
                        continue;
                    }
                }
                else
                    clearPredThreadTail();
            }

            st.stationInfos[tripinfo.stationInfoFrom->id1].transports = (Transport::Type)(tripinfo.stationInfoFrom->transports | tripinfo.transport);
            st.stationInfos[tripinfo.stationInfoTo->id1].transports = (Transport::Type)(tripinfo.stationInfoTo->transports | tripinfo.transport);

            if (is_first_station_in_thread)
                if (isExcessMCZDthread(threadUid, tripinfo.stationInfoFrom->id))
                    excessMCZDthreadUID = threadUid;

            if (threadUid != excessMCZDthreadUID)
            {
                if (is_first_station_in_thread)
                {
                    _threadStart.push_back(ThreadStart((int)_threadStart.size(), (int)_tripInfos.size()
                            , threadUid, route, tripinfo.transport, mainThread));
                    // инициализируем маску дней хождения
                    _threadStart.back().departures = tripinfo.departures;
                    tripinfo.durationStart2From = 0;
                }
                else
                    tripinfo.durationStart2From = _tripInfos.back().durationStart2To() + tripinfo.stopTimeBefore;

                tripinfo.thread = (int)_threadStart.size()-1;
                tripinfo.threadStart = &_threadStart.back();
                _tripInfos.push_back(tripinfo);
            }
        }
        if (!forRailway)
            clearPredThreadTail();

        message(LOG_FILE, "thegraph reading errors (duration < 0): ", durationErr);
        message(LOG_FILE, "thegraph reading errors (stoptime < 0): ", stoptimeErr);
        message(LOG_FILE, "thegraph reading errors (depTimeErr not increase): ",  depTimeErr);

        return (int)_tripInfos.size();
    }


//<?xml version="1.0" encoding="Windows-1251"?>
//
//<channel type="trainroute" ver="1">
//<train number="0001А" route="С-Петербург Гл. - Москва Окт.">
//  <station e2code="2004001" name="С-Петербург Гл." arrival="-" stop_time="-" departure="23:55" />
//  <station e2code="2004108" name="С-Петербург Т.М." arrival="-" stop_time="-" departure="00:02" />
//  <station e2code="9990157" name="5 Км" arrival="-" stop_time="-" departure="00:04" />
//...
//  <station e2code="2005480" name="Москва-Окт-Тов." arrival="-" stop_time="-" departure="07:49" />
//  <station e2code="2006004" name="Москва Окт." arrival="07:55" stop_time="-" departure="-" />
//</train>
//
//<train number="0002А" route="Москва Окт. - С-Петербург Гл.">
//  <station e2code="2006004" name="Москва Окт." arrival="-" stop_time="-" departure="23:55" />
//  <station e2code="2005480" name="Москва-Окт-Тов." arrival="-" stop_time="-" departure="00:01" />
//  <station e2code="2006100" name="Ховрино" arrival="-" stop_time="-" departure="00:11" />
//...
//  <station e2code="2004108" name="С-Петербург Т.М." arrival="-" stop_time="-" departure="07:47" />
//  <station e2code="2004001" name="С-Петербург Гл." arrival="07:55" stop_time="-" departure="-" />
//</train>

    // загружаем данные из файла
    int StationGraph::loadXML(const std::string& LOG_FILE, const char* filePath, const char* recodeStationsFile)
    {
        fillTripInfoSeq();
        TechBegin2TIid techmap;
        collectTechStations(techmap);

        Recode<int, int> xml2station;
        xml2station.load(recodeStationsFile);

        // если файл не читается, то возвращаем -1
        FILE* fin = fopen(filePath, "rt");
        if (!fin)
            return -1;

        int count(0);

        // станция 0    станция 1         станция 2
        //    >------------>=================>-------

        // структура файла с данными имеет следующий формат:
        // station1_id    - станция отправления
        // station2_id    - станция назначения
        // thread_uid     - идентификатор нитки
        // departure_time - время отправления (по Москве) со станции 1
        // stoptime       - время остановки
        // duration       - время в пути
        // route_id       - идентификатор маршрута
        // transport      - идентификатор типа транспорта
        // year_days      - маска хождения транспорта по Москве (длина маски равна 366=DAYS_IN_YEAR_LEAP)

        // читаем
        char w1[10000], tail1[10000], name[1000], arr[100], stop[100], dep[100];
        int err = 0;
        int inserted = 0;
        for (count = 0; fscanf(fin, "%s%[^\n]", w1, tail1) > 0; count++)
        {
            std::string w(w1); 
            std::string tail(tail1);
            if (w == "<train") {
                //tail ==  number="0001А" route="С-Петербург Гл. - Москва Окт.">
                sscanf(tail1, " number=\"%[^\"]", w1);
                std::string thread = w1;
                int wlen = strlen(w1);
                if (wlen >= 4 && w1[0] == '0') {
                    if (w1[wlen - 1] >= '0' && w1[wlen - 1] <= '9')
                        thread = w1 + 1;
                    else if (wlen >= 5)
                        thread = w1 + 1;
                }
                std::string route = thread;
                thread = "!" + thread;
                int thread_id = -1;
                int maxThread = _recodeThread.maxA;
                while (maxThread == _recodeThread.maxA) {
                    thread_id = _recodeThread.addNext(thread);
                    if (maxThread == _recodeThread.maxA) {
                        thread += "_";
                    }
                }

                int preddep = -1, predstop, arrival, stoptime, duration, departure, hour, minute;
                Station fromOrig(-1), toOrig(-1);
                for (int stationCount = 0; fscanf(fin, "%s%[^\n]", w1, tail1) > 0 && std::string(w1) == "<station";) {
                    //<station e2code="2004001" name="С-Петербург Гл." arrival="-" stop_time="-" departure="23:55" />
                    sscanf(tail1,
                           " e2code=\"%d \" name=\"%[^\"]\" arrival=\"%[^\"]\" stop_time=\"%[^\"]\" departure=\"%[^\"]\"",
                           &toOrig, name, arr, stop, dep);

                    std::map<int, int>::const_iterator it = xml2station.imapback.find(toOrig);
                    if (it == xml2station.imapback.end()) {
                        //printf("No such stations %d - %s with %s\n", toOrig, name, route.c_str());
                        message(LOG_FILE,
                                (std::string("XML: No such station in route ") + route + ": " + name + " - ").c_str(),
                                toOrig);
                        err++;
                        continue;
                    }

                    stationCount++;
                    toOrig = it->second;

                    if (std::string(dep) == "-" && std::string(arr) == "-") {
                        if (std::string(stop) == "-")
                            stoptime = 0;
                        arrival = preddep + 1;
                        departure = arrival + stoptime;
                    }
                    else {
                        if (std::string(stop) == "-")
                            stoptime = 0;
                        else {
                            sscanf(stop, "%d", &stoptime);
                        }
                        if (std::string(dep) == "-")
                            departure = 0;
                        else {
                            for (char *c = dep; *c; c++)
                                if (*c == ':')
                                    *c = ' ';
                            sscanf(dep, "%d %d", &hour, &minute);
                            departure = hour * MINUTES_IN_HOUR + minute;
                        }
                        if (std::string(arr) == "-")
                            arrival = departure - stoptime;
                        else {
                            for (char *c = arr; *c; c++)
                                if (*c == ':')
                                    *c = ' ';
                            sscanf(arr, "%d %d", &hour, &minute);
                            arrival = hour * MINUTES_IN_HOUR + minute;
                        }

                    }

                    duration = 0;
                    if (preddep >= 0)
                        duration = arrival - preddep;
                    if (duration < 0)
                        duration += 24 * MINUTES_IN_HOUR;

                    if (stationCount <= 1)
                        _threadStart.push_back(
                                ThreadStart(thread_id, (int) _tripInfos.size(), thread, route, Transport::TRAIN, 1));
                    else {
                        Departures times;
                        TripInfo tripinfo;
                        // начинаем строить ребро
                        // устанавливаем время
                        times.setTime(preddep);
                        tripinfo.departures = times;
                        tripinfo.duration = duration;
                        tripinfo.stopTimeBefore = predstop;

                        tripinfo.thread = thread_id;
                        tripinfo.threadStart = &_threadStart.back();
                        tripinfo.transport = Transport::TRAIN;
                        Station from = st.getStationInnerID(fromOrig);
                        Station to = st.getStationInnerID(toOrig);
                        //assert(from >= 0 && to >= 0);
                        if (from < 0 || to < 0) {
                            message(LOG_FILE, "thegraph XML reading ERROR (from < 0 || to < 0) with line: ", count);
                            message(LOG_FILE, "XML from=", from);
                            message(LOG_FILE, "XML to=", to);
                            continue;
                        }
                        tripinfo.id = _tripInfos.size();
                        tripinfo.stationInfoFrom = &st.stationInfos[from];
                        tripinfo.stationInfoTo = &st.stationInfos[to];
                        st.stationInfos[from].transports = (Transport::Type) (tripinfo.stationInfoFrom->transports |
                                                                              tripinfo.transport);
                        st.stationInfos[to].transports = (Transport::Type) (tripinfo.stationInfoTo->transports |
                                                                            tripinfo.transport);

                        int i = 0;
                        for (i = (int) _tripInfos.size() - 1; i >= 0 && _tripInfos[i].thread == tripinfo.thread &&
                                                              _tripInfos[i].stationInfoTo !=
                                                              tripinfo.stationInfoTo; i--);
                        if (i >= 0 && _tripInfos[i].thread == tripinfo.thread)
                            _tripInfos.resize(i);

                        _tripInfos.push_back(tripinfo);
                    }

                    predstop = stoptime;
                    preddep = departure;
                    fromOrig = toOrig;
                }

                if (_threadStart.empty())
                    continue;

                //если последняя нитка пустая, удаляем ее
                if (_tripInfos.back().thread != _threadStart.back().id) {
                    _recodeThread.deleteLastItem();
                    _threadStart.pop_back();
                    continue;
                }

                inserted += insertTechStationsInLastThread(LOG_FILE, techmap);
            }
        }
        message(LOG_FILE, "tech stations inserted %d", inserted);
        message(LOG_FILE, "errors %d", err);

        return count;
    }

    int StationGraph::insertTechStationsInLastThread(const std::string& LOG_FILE, const TechBegin2TIid& techmap)
    {
        int inserted = 0;
        const ThreadStart *ts = &_threadStart.back();
        for (int i = ts->startID; i < _tripInfos.size(); i++) {
            TripInfos::iterator ti = _tripInfos.begin() + i;
            //вставка технических станций
            std::pair<TechIt, TechIt> p = techmap.equal_range(TechBegin(ts->routeName, ti->stationInfoFrom));
            TechIt techIt = p.first;
            TripInfos::iterator forInsertIt;
            for (; techIt != p.second &&
                   (forInsertIt = techSuit(_tripInfos.begin() + techIt->second, ti)) ==
                   _tripInfos.end(); ++techIt);
            if (techIt != p.second) {
                TripInfos::iterator tiTech = _tripInfos.begin() + techIt->second;
                const ThreadStart *tsTech = &_threadStart[tiTech->thread];
                char temp[1000] = "";
                sprintf(temp, "route %s with %s , tech station (%d %s) between (%d %s) and (%d %s)",
                        ts->routeName.c_str(), tsTech->name.c_str(), tiTech->stationInfoFrom->id,
                        tiTech->stationInfoFrom->name.c_str(), forInsertIt->stationInfoFrom->id,
                        forInsertIt->stationInfoFrom->name.c_str(), forInsertIt->stationInfoTo->id,
                        forInsertIt->stationInfoTo->name.c_str()
                );
                message(LOG_FILE, "%s", temp);
                insertAfter(tiTech, forInsertIt);
                inserted++;
            }
        }
        return inserted;
    }


    bool timeBetween(const int a, const int b, const int c)
    {
        if (a <= c)
        {
            if (a < b && b < c)
                return true;
            return false;
        }
        if (b > a || b < c)
            return true;
        return false;
    }

    int timeMinus(const int a, const int b)
    {
        if (a > b)
            return a-b;
        return b+(MINUTES_IN_DAY-a);
    }

    TripInfos::iterator StationGraph::techSuit(TripInfos::iterator techIt, TripInfos::iterator beforeIt)
    {
        for (TripInfos::iterator it = beforeIt; it != _tripInfos.end() 
            && _threadStart[it->thread].route==_threadStart[beforeIt->thread].route; ++it)
            if (techIt->stationInfoFrom == it->stationInfoFrom || techIt->stationInfoFrom == it->stationInfoTo)
                return _tripInfos.end();
        for (TripInfos::iterator it = beforeIt; it != _tripInfos.end() 
            && _threadStart[it->thread].route==_threadStart[beforeIt->thread].route; ++it)
        {
            if (techIt->stationInfoFrom != it->stationInfoFrom && techIt->stationInfoFrom != it->stationInfoTo)
            if (timeBetween(it->departures.getTime(), techIt->departures.getTime(), it->departures.getTime()+it->duration))
            {
                //кроме подходящего времени, техническая линия когда-то должна встретиться с helper-линией
                TripInfos::iterator afterTechIt = techIt+1;
                const StationInfo* siAfterTech = 0;
                for (; afterTechIt != _tripInfos.end() && afterTechIt->thread==techIt->thread
                    && afterTechIt->is_technical_stop; ++afterTechIt);
                if (afterTechIt != _tripInfos.end() && afterTechIt->thread==techIt->thread)
                    siAfterTech = afterTechIt->stationInfoFrom;
                else
                    siAfterTech = (afterTechIt-1)->stationInfoTo;

                TripInfos::iterator afterIt = it;
                for (; afterIt != _tripInfos.end() && afterIt->thread==beforeIt->thread
                    && afterIt->stationInfoTo != siAfterTech; ++afterIt);
                if (afterIt != _tripInfos.end() && afterIt->thread==beforeIt->thread)
                    return it;

                return _tripInfos.end();
            }
        }

        return _tripInfos.end();
    }

    TripInfos::iterator StationGraph::insertAfter(TripInfos::iterator techIt, TripInfos::iterator helperIt)
    {
        TripInfo ti = *techIt;
        ti.stationInfoTo = helperIt->stationInfoTo;
        ti.duration = timeMinus(helperIt->departures.getTime()+helperIt->duration, techIt->departures.getTime());
        ti.thread = helperIt->thread;
        ti.threadStart = helperIt->threadStart;

        helperIt->stationInfoTo = techIt->stationInfoFrom;
        helperIt->duration -= ti.duration+ti.stopTimeBefore;

        return _tripInfos.insert(helperIt+1, ti);
    }

    void StationGraph::createAllStructures(const std::string& LOG_FILE)
    {
        fillTripInfoSeq();
        createGraph(_graph, false/*not inv graph*/);
        createGraph(_invGraph, true/*inv graph*/);
        st.initStationToCostations();
        fillIsGoodForStart();
        for (ThreadStarts::iterator it = _threadStart.begin(); it != _threadStart.end(); ++it)
        {
            it->updateInfo();
            RouteMap::iterator rit = _routeMap.find(it->routeName);
            int id = 0;
            if (rit == _routeMap.end())
            {
                id = _routes.size();
                _routeMap[it->routeName] = id;
                _routes.push_back(Route(it->routeName, id));
            }
            else
            {
                id = rit->second;
            }
            std::vector<const ThreadStart*>::iterator tit = _routes[id].threads.begin();
            for (; tit != _routes[id].threads.end() && (*tit)->id != it->id; ++tit);
            if (tit == _routes[id].threads.end())
                _routes[id].threads.push_back(&(*it));
        }
        for (ThreadStarts::iterator it = _threadStart.begin(); it != _threadStart.end(); ++it)
        {
            it->route = &_routes[_routeMap[it->routeName]];
        }
        fillTripInfoSai();
    }

    void StationGraph::createGraph(Graph& g, const bool inv)
    {
        g.clear();
        g.resize(st.getStationNumber());
        for (TripInfos::iterator t = _tripInfos.begin(); t != _tripInfos.end(); ++t)
        {
            // добавляем ребро (from, to, tripinfo) в граф
            const StationInfo* from = t->stationInfoFrom;
            const StationInfo* to = t->stationInfoTo;
            if (inv)
                std::swap(from, to);

            const size_t n = std::max(from->id1, to->id1)+1;
            if (g.size() < n)
                g.resize(n);

            Neighbours::iterator itn = g[from->id1].sais.begin();
            Neighbours::iterator itEnd = g[from->id1].sais.end();
            for (; itn != itEnd && itn->stationTo != to; ++itn);
            if (itn != itEnd)
            {
                itn->info.push_back(&(*t));
                itn->transports = (Transport::Type)(itn->transports | t->transport);
            }
            else
            {
                Info info;
                info.push_back(&(*t));
                g[from->id1].sais.push_back(StationAndInfo(to, info));
                g[from->id1].sais.back().transports = t->transport;
            }
        }
    }

    void StationGraph::fillTripInfoSai()
    {
        for(size_t i = 0; i < _graph.size(); ++i)
            for (size_t j = 0; j < _graph[i].sais.size(); ++j)
            {
                for(size_t k = 0; k < _graph[i].sais[j].info.size(); ++k)
                {
                    _graph[i].sais[j].info[k]->sai = &_graph[i].sais[j];
                }
            }
    }

    void StationGraph::fillTripInfoSeq()
    {
        //теперь _tripInfos заполнен и больше не будет перемещаться по памяти
        //можно спокойно пользоваться const TripInfo*
        TripInfo* predTI = 0;
        int id = 0;
        for (TripInfos::iterator it = _tripInfos.begin(); it != _tripInfos.end(); ++it, id++)
        {
            if (predTI && predTI->thread == it->thread)
            {
                predTI->nextTI = &(*it);
                it->predTI = predTI;
            }
            else
            {
                if (predTI)
                    predTI->nextTI = 0;
                it->predTI = 0;
                _threadStart[it->thread].startTI = &(*it);
                _recodeThread.addNext(_threadStart[it->thread].name);
            }

            it->threadStart = &_threadStart[it->thread];
            it->id = id;
            predTI = &(*it);
        }
    }

    struct MajTI
    {
        int majority;
        int time;
        const TripInfo* ti;
        TeleportLink* tlink;
        MajTI(int maj = 4, int t = 0, const TripInfo* ti0 = 0, TeleportLink* tl = 0)
            : majority(maj), time(t), ti(ti0), tlink(tl) 
        {}
    };
    typedef std::map<const StationInfo*, MajTI> si2majMap;
    void StationGraph::fillIsGoodForStart()
    {
        for (ThreadStarts::iterator it = _threadStart.begin(); it != _threadStart.end(); ++it)
        {
            si2majMap si2majStart;
            si2majMap si2majFinish;
            for (const TripInfo* ti = it->startTI; ti; ti = ti->nextTI)
            {
                for (TeleportLinkPtrs::iterator tit = st.stationInfos[ti->stationInfoFrom->id1].towns.begin(); tit != ti->stationInfoFrom->towns.end(); ++tit)
                {
                    if (ti->stopOnFirstStation())
                    {
                        si2majMap::iterator find = si2majStart.find((*tit)->town);
                        MajTI majti(ti->stationInfoFrom->majority, ti->durationStart2From-(*tit)->time2center, ti, *tit);
                        if (find == si2majStart.end() 
                                || find->second.majority > majti.majority
                                || (find->second.majority == majti.majority && find->second.time < majti.time))
                            si2majStart[(*tit)->town] = majti;
                    }
                }
                for (TeleportLinkPtrs::iterator tit = st.stationInfos[ti->stationInfoTo->id1].towns.begin(); tit != ti->stationInfoTo->towns.end(); ++tit)
                    if (ti->stopOnLastStation())
                    {
                        si2majMap::iterator find = si2majFinish.find((*tit)->town);
                        MajTI majti(ti->stationInfoTo->majority, ti->durationStart2To()+(*tit)->time2center, ti, *tit);
                        if (find == si2majFinish.end() 
                                || find->second.majority > majti.majority
                                || (find->second.majority == majti.majority && find->second.time > majti.time))
                            si2majFinish[(*tit)->town] = majti;
                    }
            }
            for (si2majMap::iterator itMap = si2majStart.begin(); itMap != si2majStart.end(); ++itMap)
            {
                itMap->second.tlink->isGoodForStart = true;
                _tripInfos[itMap->second.ti->id].townsGoodForIn.push_back(itMap->second.tlink);
                _graph[itMap->first->id1].startTIs.push_back(itMap->second.ti);
            }
            for (si2majMap::iterator itMap = si2majFinish.begin(); itMap != si2majFinish.end(); ++itMap)
            {
                itMap->second.tlink->isGoodForFinish = true;
                _tripInfos[itMap->second.ti->id].townsGoodForOut.push_back(itMap->second.tlink);
                _invGraph[itMap->first->id1].startTIs.push_back(itMap->second.ti);
            }
            for (const TripInfo* ti = it->startTI; ti; ti = ti->nextTI)
            {
                if (ti->stopOnFirstStation())
                    _graph[ti->stationInfoFrom->id1].startTIs.push_back(ti);
                if (ti->stopOnLastStation())
                    _invGraph[ti->stationInfoTo->id1].startTIs.push_back(ti);
                //проблема Домодедово
                if (ti->stationInfoFrom->transports & Transport::PLANE)
                    if (ti->transport & Transport::TRAIN_OR_LOCAL_TRAIN)
                    {
                        //_tripInfos[ti->id].isGoodForStart = false;
                        _tripInfos[ti->id].townsGoodForIn.clear();
                        _tripInfos[ti->id].townsGoodForOut.clear();
                    }
            }
        }
    }


    int StationGraph::collectTechStations(TechBegin2TIid& map)
    {
        int res = 0;
        if (_tripInfos.empty())
            return res;
        for (TripInfos::iterator t = _tripInfos.begin()+1; t != _tripInfos.end(); ++t)
        {
            if (t->is_technical_stop && (t-1)->thread == t->thread)
            {
                map.insert(std::pair<TechBegin, size_t>(TechBegin(_threadStart[t->thread].routeName, (t-1)->stationInfoFrom), t->id));
                res++;
            }
        }
        return res;
    }

    int StationGraph::loadIntervalThreadsInfo(const std::string& LOG_FILE, const char* filePath)
    {
        FILE* fin = fopen(filePath, "rt");
        if (!fin)
            return -1;
        char threadName[100], begin_date[100], end_date[100], density[100];
        int count;
        for (count = 0; fscanf(fin, "%s%s%s%[^\t]%*s", threadName, begin_date, end_date, density) > 0; count++)
        {
            int tid = getThreadInnerID(threadName);
            if (tid >= 0)
            {
                ThreadStart* ts = &_threadStart[tid];
                ts->interval_begin_time = Dates::fromStringTime(begin_date);
                ts->interval_end_time = Dates::fromStringTime(end_date);
                int i = -2, j = 0;
                for (char* c = density; *c; c++, j++)
                    if (*c < '0' || *c > '9')
                        *c = ' ';
                    else
                        if (j!=i+1)
                            i = j;
                int delta = 60;
                if (sscanf(density+i, "%d", &delta) == 1 && delta)
                    ts->interval_delta = delta;
            }
            else
                message(LOG_FILE, "interval reading ERROR with line: ", count);

        }
        fclose(fin);
        return count;
    }

    void StationGraph::save(std::ofstream& fout) const
    {
        for (TripInfos::const_iterator t = _tripInfos.begin(); t != _tripInfos.end(); ++t) {
            fout << t->id << " ";
            if (t->stationInfoFrom)
                fout << t->stationInfoFrom->id << " ";
            if (t->stationInfoTo)
                fout << t->stationInfoTo->id << " ";
            fout << t->thread << " ";
            fout << t->duration << " ";
            fout << t->durationStart2From << " " << t->stopTimeBefore << " " << t->is_technical_stop << std::endl;
        }
        fout << std::endl;
        for (ThreadStarts::const_iterator it = _threadStart.begin(); it != _threadStart.end(); ++it) {
            fout << it->id << " " << it->loadedCorrect << " " << it->name << " ";
            fout << it->startID << " " << it->tiNumber << " " << it->stopTime << " " << it->moveTime << std::endl;
        }
    }

}
