#ifndef STATIONGRAPH_H
#define STATIONGRAPH_H

#include "tripinfo.h"
#include "recode.h"
#include "stationtown.h"
#include "geom.h"
#include "curve.h"

namespace Pathfinder
{
    static inline bool endingWith(const char* s, const char* end)
    {
        size_t ns = strlen(s);
        size_t ne = strlen(end);
        if (ns >= ne)
        {
            return strcmp(s + ns - ne, end) == 0;
        }
        return false;
    }

    static inline bool startWith(const std::string s, const std::string& start)
    {
        return std::equal(start.begin(), start.end(), s.begin());
    }

    // пара: станция, сообщения
    struct StationAndInfo
    {
        const StationInfo* stationTo;
        Info info;
        Transport::Type transports;

        int nShortEdgeInside;
        const TripInfo* detailedTI;
        Geom::Curve curve;
        std::map<std::string, Geom::Curve> routeCurves;
        std::vector<const StationInfo*> innerStations;
        double dist;

        StationAndInfo() : stationTo(0), nShortEdgeInside(0), detailedTI(0), dist(-1) {}
        StationAndInfo(const StationInfo* st, const Info& inf) : stationTo(st), info(inf), nShortEdgeInside(1), detailedTI(0), dist(-1) {}
    };
    // окрестность вершины: станции и сообщения
    typedef std::vector<StationAndInfo> Neighbours;
    struct BestPrice
    {
        const StationInfo* to;
        int price;
        BestPrice(const StationInfo* si=0, int p=0)
                : to(si), price(p)
        {}
        inline bool operator<(const BestPrice& t) const
        {
            return to < t.to;
        }
    };
    typedef std::vector<BestPrice> BestPrices;
    class StationRasp
    {
    public:
        Neighbours sais;
        TripInfoPtrs startTIs;
        BestPrices bestPrices;
        int getBestPrice(const StationInfo* to) const
        {
            if (bestPrices.empty())
                return MAX_PRICE;
            BestPrices::const_iterator it = std::lower_bound(bestPrices.begin(), bestPrices.end(), BestPrice(to));
            if (it != bestPrices.end() && it->to == to)
                return it->price;
            return MAX_PRICE;
        }
    };
    // граф сообщений, хранится как окрестностей, индекс совпадает с идентификатором станции
    typedef std::vector<StationRasp> Graph;

    class StationGraph
    {
    public:
        Graph _graph;
        Graph _invGraph;
        TripInfos _tripInfos;
        Recode<int, std::string> _recodeThread;
        StationTown st;

        // для нитки храним станцию, с которой нитка стартует
        ThreadStarts _threadStart;
        Routes _routes;
        RouteMap _routeMap;

        int getBestPrice(const StationInfo* from, const StationInfo* to) const
        {
            return _graph[from->id1].getBestPrice(to);
        }
        int getPrice(const TripInfo* tripinfo, const StationInfo* to, const int time) const
        {
            Departures DEP;
            const size_t day(time / MINUTES_IN_DAY);
            const size_t d1 = DEP.toOurDay(day);
            SearchHelper SH(d1);

            int predPrice = MAX_PRICE;
            int price = MAX_PRICE;
            const Tariff* tariff = tripinfo->minTariff;
            for (; tariff && tariff <= tripinfo->maxTariff; tariff++)
            {
                if (tariff->to == to)
                {
                    if ((tariff->departures == 0 || tariff->departures->isGoodDay(SH)) && (tariff->date < 0 || tariff->date >= day || tariff == tripinfo->maxTariff))
                    {
                        price = tariff->price;
                        break;
                    }
                    predPrice = tariff->price;
                }
            }
            if (tariff && tariff->date >= 0 && tariff->date > day && predPrice < MAX_PRICE)
                price = std::min(price, predPrice);
            return price;
        }

        int load(const std::string& LOG_FILE, const char* filePath, const Transport::Type tr = Transport::ALL, const bool forRailway = false);
        bool loadTripInfo(std::string buffer, TripInfo &tripinfo, std::string &threadUid, std::string &route,
                          const std::string &LOG_FILE, const int count, int &mainThread);
        bool checkTripInfo(const std::string &LOG_FILE, TripInfo &tripinfo, std::string &threadUid,
                           const int count, int &durationErr, int &stoptimeErr);
        bool checkPredTripInfo(const std::string &LOG_FILE, TripInfo &ti, TripInfo &predTI,
                               const int count, int &depTimeErr);
        void clearPredThreadTail();
        bool lastThreadIsSingleEdge() const;

        int loadXML(const std::string& LOG_FILE, const char* filePath, const char* recodeStationsFile);
        int loadIntervalThreadsInfo(const std::string& LOG_FILE, const char* filePath);
        void save(std::ofstream& fout) const;

        void createGraph(Graph& g, const bool inv);
        void fillTripInfoSeq();
        void fillIsGoodForStart();
        void fillTripInfoSai();
        void calcIsGoodForStart();

        void createAllStructures(const std::string& LOG_FILE);

        void createAllStructures();

        inline size_t getStationNumber() const { return st.getStationNumber(); }
        inline size_t getThreadNumber() const { return _threadStart.size(); }
        inline std::string getThreadOrig(const Thread id) const
        {
            std::map<int, std::string>::const_iterator it = _recodeThread.imap.find(id);
            if (it == _recodeThread.imap.end())
                return "";
            return it->second;
        }
        inline std::string getThreadOrigWithoutSubThread(const Thread id) const
        {
            std::map<int, std::string>::const_iterator it = _recodeThread.imap.find(id);
            if (it == _recodeThread.imap.end())
                return "";
            if (it->second.empty() || *(it->second.end()-1) != ')')
                return it->second;
            std::string res = it->second;
            std::string::iterator sit = res.end()-1;
            for (; sit != res.begin() && (*sit) != '('; --sit){
            }
            if (sit != res.begin())
                res.erase(sit, res.end());
            return res;
        }
        inline Thread getThreadInnerID(const std::string threadName) const
        {
            std::map<std::string, int>::const_iterator it = _recodeThread.imapback.find(threadName);
            if (it == _recodeThread.imapback.end())
                return -1;
            return it->second;
        }
        size_t getThreadsBegining(const Neighbours& neigs, const Transport::Type trFilter);
        size_t getThreadsEnding(const Neighbours& neigs, const Transport::Type trFilter);

        size_t saveRoadsAsPolishNear(std::ostream& out, const Transport::Type trFilter, const char* color, const bool paintLongEdge = true);
        size_t saveRoadsAsPolishFar(std::ostream& out, const Transport::Type trFilter, const char* color);
        size_t saveCurveRoadsAsPolish(std::ostream& out, const Transport::Type trFilter, const char* color, const bool paintLongEdge = true);
        size_t filterBadStationLocation(std::ostream& log, const Transport::Type trFilter);
        size_t interpolateEmptyStationLocation(std::ostream& log, const Transport::Type trFilter);
        size_t fillShortestPaths(std::ostream& log, const Transport::Type trFilter);
        size_t fillLongCurves(std::ostream& log, const Transport::Type trFilter);
        size_t fillLongCurves2(std::ostream& log, const Transport::Type transports);
        size_t fillLongCurvesByThread(std::ostream& log, const TripInfo* startTI, const Transport::Type transports);
        size_t fillCurves(std::ostream& log, Geom::CurveMap& curveMap, const double eps, const Transport::Type trFilter);
        size_t calcDists();
        size_t fillNearestNeigs(const Transport::Type trFilter);
        size_t fillNearestNeigs2(const Transport::Type trFilter);
        size_t calcLongestPointPath(std::ostream& log, const Transport::Type trFilter);
        int saveThreadsAsPolish(std::ostream& log, std::ostream& out, const Transport::Type trFilter, const char* color, const int minMoveTime, const int maxMoveTime, const int notAll, const int shift, const bool drawHelpers = false, const bool drawNotHelpers = true)
        {
            int res = 0;
            int time = 0;
            for (ThreadStarts::iterator it = _threadStart.begin(); it != _threadStart.end(); ++it, time++)
                if (it->transports & trFilter)
                if (it->moveTime >= minMoveTime && it->moveTime <= maxMoveTime && ((time+shift) % notAll) == 0)
                {
                    if (drawHelpers && it->startTI->isHelper())
                    {
                        it->saveAsPolish(log, out, color);
                    }
                    if (drawNotHelpers && !it->startTI->isHelper())
                    {
                        it->saveAsPolish(log, out, color);
                    }
                }
            return res;
        }
        size_t saveThreadsAsPolish_peace(std::ostream& log, const Transport::Type trFilter, const char* color);
        size_t saveEdge2Curve(std::ostream& out, const Transport::Type trFilter);

        void printStat(std::ostream& out)
        {
            std::vector<int> nodosity;
            nodosity.resize(st.stationInfos.size(), 0);
            for (ThreadStarts::iterator it = _threadStart.begin(); it != _threadStart.end(); ++it)
            {
                for (const TripInfo* ti = it->startTI; ti; ti = ti->nextTI)
                    if (ti->stationInfoFrom->isKURB() && ti->stationInfoTo->isKURB())
                    {
                        if (ti->predTI == 0)
                            nodosity[ti->stationInfoFrom->id1] += 100;
                        if (ti->nextTI == 0)
                            nodosity[ti->stationInfoTo->id1] += 100;
                        nodosity[ti->stationInfoFrom->id1] += ti->stopTimeBefore;
                    }
            }
            for (int i = 0; i < st.stationInfos.size(); i++)
                if (!st.isTown(i))
                {
                    out << nodosity[i] << "\t" << st.stationInfos[i].id << "\t" << st.stationInfos[i].name << "\t" ;
                    if (st.stationInfos[i].towns.empty())
                        out << "NULL";
                    else
                        out << st.stationInfos[i].towns[0]->town->name;
                    out << std::endl;
                }
        }

        struct TechBegin
        {
            std::string route;
            const StationInfo* st;
            TechBegin() : st(0) {}
            TechBegin(const std::string& r, const StationInfo* s) : route(r), st(s) {}
            inline bool operator<(const TechBegin& ws) const
            {
                if (route != ws.route)
                    return route < ws.route;
                return st < ws.st;
            }
        };
        typedef std::multimap<TechBegin, size_t> TechBegin2TIid;
        typedef TechBegin2TIid::const_iterator TechIt;
        int collectTechStations(TechBegin2TIid&);

        int insertTechStationsInLastThread(const std::string& LOG_FILE, const TechBegin2TIid& techmap);
        TripInfos::iterator techSuit(TripInfos::iterator techIt, TripInfos::iterator afterIt);
        TripInfos::iterator insertAfter(TripInfos::iterator techIt, TripInfos::iterator afterIt);

    };

}

#endif
