#ifndef STATIONTOWN_H
#define STATIONTOWN_H

#include "pf.h"
#include "recode.h"
#include "geom.h"
#include "transport.h"
#include "dates.h"

namespace Geom
{
    struct Curve;
}

namespace Pathfinder
{
    typedef int Station;
    typedef std::vector<Station> Stations;

    // идентификатор города
    typedef int Town;
    typedef std::vector<Town> Towns;
    struct StationInfo;
    class TripInfo;
    typedef std::vector<const StationInfo*> StationInfoPtrs;
    typedef std::set<const StationInfo*> CostationsSet;
    struct TeleportLink
    {
        enum Type
        {
            NONE    = 0x00,
            START    = 0x01,
            TELEPORT    = 0x02,
            ALL        = 0x03
        };
        
        Station stationID1;
        const StationInfo* si;
        const StationInfo* town;
        Type type;
        int time2center;
        bool isGoodForStart;
        bool isGoodForFinish;

        TeleportLink(const Station id1, const Type t = ALL, const int time = 25)
            : stationID1(id1), si(0), town(0), type(t), time2center(time), isGoodForStart(false), isGoodForFinish(false)
        {}
        TeleportLink(const StationInfo* si1, const StationInfo* town1, const Type t = ALL, const int time = 25);
        // оператор сравнения
        inline bool operator<(const TeleportLink& ws) const
        {
            return stationID1 < ws.stationID1;
        }
    };
    typedef std::vector<TeleportLink> TeleportLinks;
    typedef std::vector<TeleportLink*> TeleportLinkPtrs;
    typedef std::vector<const TeleportLink*> TeleportLinkConstPtrs;

    struct StationInfo
    {
        //внутренний номер
        Station id1;
        //внешний
        Station id;
        std::string name;
        //связи с городами
        TeleportLinkPtrs towns;
        int majority;
        bool isGoodForStart;
        //costations у города - это ВСЕ станции в городе
        //costations у станции - связи станция-станция
        TeleportLinks costations;
        CostationsSet costationsSet;
        std::vector<const StationInfo*> nearestStations;
        Geom::Point2d location;
        Geom::Point2d oldLocation;
        Transport::Type transports;
        const Geom::Curve* curve;
        std::vector<const Geom::Curve*> neigCurves;
        int country_id;
        const TimeZone* timeZone;
        StationInfo() : id1(0), id(0), majority(-1), isGoodForStart(false), transports(Transport::NONE)
            , curve(0), country_id(-1), timeZone(0)
        {} 

        bool canChange() const
        {
            return majority <= 3;
        }

        bool canTeleport() const
        {
            return majority <= 2;
        }
        bool isTown() const
        {
            return majority < 0;
        }
        bool canLast() const
        {
            return majority <= 2;
        }
        bool isKURB() const
        {
            if (country_id == 187/*Украина*/
                || country_id == 225/*Россия*/
                || country_id == 149/*Белларусия*/
                || country_id == 159/*Казахстан*/
                )
                return true;
            return false;
        }

        //станции на противоположных концах кольца МЦК - Нижегородская и Стрешнево
        enum {
            nizegorodskayaStationId = 9855163,
            streshnevoStationId = 9855178,
        };

        int getNumCostationsGoodForIn(const StationInfo* exceptSI, const Transport::Type transport) const
        {
            int res = 0;
            TeleportLinks::const_iterator itEnd = costations.end();
            for (TeleportLinks::const_iterator it = costations.begin(); it != itEnd; ++it)
                if (it->si != exceptSI && it->isGoodForStart && it->si->transports & transport)
                    res++;
            return res;
        }

        int getNumCostationsGoodForOut(const StationInfo* exceptSI, const Transport::Type transport) const
        {
            int res = 0;
            TeleportLinks::const_iterator itEnd = costations.end();
            for (TeleportLinks::const_iterator it = costations.begin(); it != itEnd; ++it)
                if (it->si != exceptSI && it->isGoodForFinish && it->si->transports & transport)
                    res++;
            return res;
        }

        void debugPrint(std::ostream& out) const
        {
            out << id << " " << name << " maj=" << majority << " tr=" << transports;
            if (timeZone)
                out << " tz=" << timeZone->name;
            out << " towns=(";
            TeleportLinkPtrs::const_iterator itEnd = towns.end();
            for (TeleportLinkPtrs::const_iterator it = towns.begin(); it != itEnd; ++it)
                if ((*it)->town)
                    out << (*it)->town->id << " " << (*it)->town->name << " time2center=" << (*it)->time2center 
                        << " isGoodForStart=" << (*it)->isGoodForStart << " isGoodForFinish=" << (*it)->isGoodForFinish << " | ";
            out << ")";
            out << " teleport=(";
            TeleportLinks::const_iterator itcEnd = costations.end();
            for (TeleportLinks::const_iterator itc = costations.begin(); itc != itcEnd; ++itc)
                if (itc->si)
                    out << itc->town->id << " " << itc->town->name << " maj=" << itc->town->majority << " time=" << itc->time2center << " | ";
            out << ")";
        }
    };
    typedef std::vector<StationInfo> StationInfos;

    // храним информацию о том в каком городе есть какие станции
    class StationTown
    {
    private:
        size_t minStationID;
    
    public:
        StationTown() : minStationID(0)
        {}
        StationInfos stationInfos;
        Recode<int, int> recodeStation;
        TimeZoneStorage timeZoneStorage;

        inline size_t getStationNumber() const { return stationInfos.size(); }
        inline size_t getStationMajority(const Station id) const { return stationInfos[id].majority; }
//        inline const StationInfoPtrs& getCostations(const Station id) const { return stationInfos[id].costations; }

        inline Station getStationOrig(const Station id) const
        {
            std::map<int, int>::const_iterator it = recodeStation.imap.find(id);
            if (it == recodeStation.imap.end())
                return -1;
            return it->second;
        }
        inline Station getStationInnerID(const Station orig) const
        {
            std::map<int, int>::const_iterator it = recodeStation.imapback.find(orig);
            if (it == recodeStation.imapback.end())
                return -1;
            return it->second;
        }
        inline Station getStationInnerIDWithoutGraph(const Station orig) 
        {
            int maxA = recodeStation.maxA;
            Station id1 = recodeStation.addNext(orig);
            //std::map<int, int>::const_iterator it = recodeStation.imapback.find(orig);
            if (maxA != recodeStation.maxA)
            {
                stationInfos.push_back(StationInfo());
                stationInfos[id1].id1 = id1;
                stationInfos[id1].id = orig;
            }
            return id1;
        }
        inline const StationInfo* getStationByOrig(const Station orig) const
        {
            std::map<int, int>::const_iterator it = recodeStation.imapback.find(orig);
            if (it == recodeStation.imapback.end())
                return 0;
            if (it->second >= 0 && it->second < stationInfos.size())
                return &stationInfos[it->second];
            return 0;
        }
        inline TeleportLink* getTeleportLink(const StationInfo* si, StationInfo* town)
        {
            TeleportLinks::iterator it = town->costations.begin();
            for (; it != town->costations.end() && it->si != si; ++it);
            if (it != town->costations.end())
                return &(*it);
            return 0;
        }

        size_t loadTowns(const char* file)
        {
            stationInfos.clear();
            FILE* fin = fopen(file, "rt");
            Station orig;
            char temp[200];
            while (fscanf(fin, "%d\t%[^\t\n]\n", &orig, temp) == 2)
            {
                Station id = recodeStation.addNext(orig);
                if (id >= (Station)stationInfos.size())
                    stationInfos.push_back(StationInfo());
                stationInfos[id].id1 = id;
                stationInfos[id].id = orig;
                stationInfos[id].majority = -1;
                stationInfos[id].name = temp;
//                stationInfos[id].towns.push_back(id);
                stationInfos[id].isGoodForStart = false;
            }
            minStationID = stationInfos.size();
            return stationInfos.size();
        }
        size_t loadStations(const char* file)
        {
            FILE* fin = fopen(file, "rt");
            StationInfo si;
            int id;
            int maj;
            char town[20], name[200], country[40], time_zone[200];
            while (fscanf(fin, "%d\t%[^\t]\t%s\t%d\t%s\t%s", &id, name, town, &maj, country, time_zone) == 6)
            {
                si.id = id;
                si.majority = maj;
                int maxA = recodeStation.maxA;
                Station id1 = recodeStation.addNext(si.id);
                if (maxA != recodeStation.maxA)
                    stationInfos.push_back(StationInfo());
                if (town[0] != 'N')
                {
                    Town t = atoi(town);
                    Town tid = getStationInnerID(t);
                    stationInfos[tid].costations.push_back(TeleportLink(id1, TeleportLink::ALL));
//                    stationInfos[id].towns.push_back(&stationInfos[tid].costations.back());
                }
                stationInfos[id1].id1 = id1;
                stationInfos[id1].id = si.id;
                stationInfos[id1].majority = si.majority;
                stationInfos[id1].name = name;
                stationInfos[id1].isGoodForStart = false;//stationInfos[id].canLast();
                stationInfos[id1].country_id = (country[0] == 'N') ? -1 : atoi(country);
                stationInfos[id1].timeZone = timeZoneStorage.getTimeZone(time_zone);
            }
            return stationInfos.size();
        }

        size_t loadGeo(const char* file)
        {
            FILE* fin = fopen(file, "rt");
            size_t res = 0;
            char xstr[100], ystr[100];
            int tempid;
            while (fscanf(fin, "%d %s %s", &tempid, xstr, ystr) == 3)
            {
                Station id = recodeStation.addNext(tempid);
                if (id < (int)stationInfos.size() && xstr[0] != 'N' && ystr[0] != 'N')
                {
                    stationInfos[id].location.x = atof(xstr);
                    stationInfos[id].location.y = atof(ystr);
                    stationInfos[id].oldLocation = stationInfos[id].location;
                    res++;
                }
            }
            return res;
        }

        size_t loadTime2Center(const char* file)
        {
            for (size_t i = 0; i < minStationID; i++)
            {
                TeleportLinks::iterator it = stationInfos[i].costations.begin();
                for (; it != stationInfos[i].costations.end(); ++it)
                {
                    it->si = &stationInfos[it->stationID1];
                    it->town = &stationInfos[i];
                }
            }

            FILE* fin = fopen(file, "rt");
            if (!fin)
                return 0;
            size_t res = 0;
            int stationOrig, townOrig, time;
            while (fscanf(fin, "%d %d %d", &townOrig, &stationOrig, &time) == 3)
            {
                Station stationID = recodeStation.addNext(stationOrig);
                Station townID = recodeStation.addNext(townOrig);
                if (stationID < (int)stationInfos.size() && townID < (int)stationInfos.size() )
                {
                    if (isTown(townID))
                    {
                        TeleportLink* link = getTeleportLink(&stationInfos[stationID], &stationInfos[townID]);
                        if (link)
                            link->time2center = time;
                        else
                            stationInfos[townID].costations.push_back(TeleportLink(&stationInfos[stationID], &stationInfos[townID], TeleportLink::ALL, time));
                    }    
                    else
                    {
                        stationInfos[stationID].costations.push_back(TeleportLink(&stationInfos[stationID], &stationInfos[townID], TeleportLink::TELEPORT, time));
                        stationInfos[townID].costations.push_back(TeleportLink(&stationInfos[townID], &stationInfos[stationID], TeleportLink::TELEPORT, time));
                    }
                }
            }
            return res;
        }



        size_t saveStationsAsPolish(std::ostream& out, const Transport::Type trFilter, const char* color);

        // проверяем, что идентификатор "является" идентификатором города
        bool isTown(size_t id) const { return id < minStationID; }
        bool isTown(const StationInfo* si) const { return si->id1 < minStationID; }

        const StationInfo* fromOneTown(const StationInfo* s0, const StationInfo* s1) const
        {
            if (s0 == s1)
                return s0;
            if (isTown(s0) && isTown(s1))
                return 0;
            if (s0->towns.empty() || s1->towns.empty())
                return 0;
            TeleportLinkPtrs::const_iterator itEnd0 = s0->towns.end();
            TeleportLinkPtrs::const_iterator itEnd1 = s1->towns.end();
            for (TeleportLinkPtrs::const_iterator it0 = s0->towns.begin(); it0 != itEnd0; ++it0)
                for (TeleportLinkPtrs::const_iterator it1 = s1->towns.begin(); it1 != itEnd1; ++it1)
                    if ((*it0)->town == (*it1)->town)
                        return (*it0)->town;
            return 0;
        }

        const StationInfo* fromTowns(const StationInfo* station, const StationInfoPtrs& towns) const
        {
            if (isTown(station))
                return 0;
            for (TeleportLinkPtrs::const_iterator it0 = station->towns.begin(); it0 != station->towns.end(); ++it0)
                for (StationInfoPtrs::const_iterator it1 = towns.begin(); it1 != towns.end(); ++it1)
                    if ((*it0)->town == *it1)
                        return (*it0)->town;
            return 0;
        }

        const StationInfo* fromStationsTowns(const StationInfo* station, const StationInfoPtrs& stations) const
        {
            if (isTown(station))
                return 0;
            for (TeleportLinkPtrs::const_iterator it0 = station->towns.begin(); it0 != station->towns.end(); ++it0)
                for (StationInfoPtrs::const_iterator it1 = stations.begin(); it1 != stations.end(); ++it1)
                    for (TeleportLinkPtrs::const_iterator it2 = (*it1)->towns.begin(); it2 != (*it1)->towns.end(); ++it2)
                        if ((*it0)->town == (*it2)->town)
                            return (*it0)->town;
            return 0;
        }

        //формирует список вершин, из которых может начинатсья и заканчиваться маршрут
        void allStationsFrom_startOnly(const StationInfo* id, StationInfoPtrs& res) const
        {
            res.clear();
            if (id == 0)
                return;
            if (isTown(id->id1))
            {
                TeleportLinks::const_iterator itEnd = stationInfos[id->id1].costations.end();
                for (TeleportLinks::const_iterator it = stationInfos[id->id1].costations.begin(); it != itEnd; ++it)
                    if (it->isGoodForStart && (it->type&TeleportLink::START))// si->isGoodForStart )
                        res.push_back(it->si);
            }
            else
                res.push_back(id);
        }
        
        void allStationsFrom_finishOnly(const StationInfo* id, StationInfoPtrs& res) const
        {
            res.clear();
            if (id == 0)
                return;
            if (isTown(id->id1))
            {
                TeleportLinks::const_iterator itEnd = stationInfos[id->id1].costations.end();
                for (TeleportLinks::const_iterator it = stationInfos[id->id1].costations.begin(); it != itEnd; ++it)
                    if (it->isGoodForFinish && (it->type&TeleportLink::START))// si->isGoodForStart )
                        res.push_back(it->si);
            }
            else
                res.push_back(id);
        }
        
        void addTownsWith(const StationInfoPtrs& s1, StationInfoPtrs& res) const
        {
            std::set<const StationInfo*> s;
            StationInfoPtrs::const_iterator itEnd = s1.end();
            for (StationInfoPtrs::const_iterator it = s1.begin(); it != itEnd; ++it)
                for (TeleportLinkPtrs::const_iterator t = (*it)->towns.begin(); t != (*it)->towns.end(); ++t)
                    s.insert((*t)->town);
            for (std::set<const StationInfo*>::iterator it = s.begin(); it != s.end(); ++it)
                res.push_back(*it);
        }
        
        // init station to costations map
        void initStationToCostations()
        {
            size_t n = stationInfos.size();
            for (size_t i = 0; i < minStationID; i++)
            {
                std::sort(stationInfos[i].costations.begin(), stationInfos[i].costations.end());
                TeleportLinks::iterator itEnd = stationInfos[i].costations.end();
                for (TeleportLinks::iterator it = stationInfos[i].costations.begin(); it != itEnd; ++it)
                {
                    it->si = &stationInfos[it->stationID1];
                    it->town = &stationInfos[i];
                    stationInfos[it->stationID1].towns.push_back(&(*it));
                }
            }
            for (size_t i = 0/*minStationID*/; i < n; i++)
            {
                TeleportLinks::iterator itEnd = stationInfos[i].costations.end();
                for (TeleportLinks::iterator it = stationInfos[i].costations.begin(); it != itEnd; ++it)
                {
                    if (it->si->transports)
                        stationInfos[i].costationsSet.insert(it->si);
                    if (it->si->transports & Transport::PLANE)
                        it->time2center = 45;
                }
            }
            //for (size_t i = 0; i < n; i++)
            //{
            //    CostationsSet::const_iterator itEnd1 = stationInfos[i].costationsSet.end();
            //    for (CostationsSet::const_iterator it = stationInfos[i].costationsSet.begin(); it != itEnd1; ++it)
            //    {
            //        //stationInfos[i].costations.push_back(*it);
            //    }
            //}
        }

        void saveCostations(const std::string& LOG_FILE)
        {
            char temp[1000];
            for (size_t i = 0; i < minStationID; i++)
            {
                TeleportLinks::iterator itEnd = stationInfos[i].costations.end();
                for (TeleportLinks::iterator it = stationInfos[i].costations.begin(); it != itEnd; ++it)
                {
                    sprintf(temp, "town=(%d %s) station=(%d %s maj=%d) in=%d out=%d", it->town->id, it->town->name.c_str(), it->si->id, it->si->name.c_str(), it->si->majority, (int)it->isGoodForStart, (int)it->isGoodForFinish);
                    message(LOG_FILE, "", temp);
                }
            }
        }

        size_t checkSameLocation(std::ostream& log, const Transport::Type trFilter)
        {
            size_t res = 0;
            for (StationInfos::iterator it1 = stationInfos.begin(); it1 != stationInfos.end(); ++it1)
            {
                if (!it1->location.IsZero() && it1->transports&trFilter)
                {
                    if (it1->curve == 0)
                    {
                        log << it1->id << "\t" << it1->name.c_str() << "\t" << "station_without_curve\t" << it1->location << std::endl; 
                    }
                    for (StationInfos::iterator it2 = it1; it2 != stationInfos.end(); ++it2)
                    {
                        if (it1 != it2 && it2->transports&trFilter)
                        {
                            if (it1->location == it2->location)
                            {
                                log << it1->id << "\t" << it1->name << "\tsame_loc\t" << it2->id << "\t" << it2->name << "\t" << it1->location << std::endl;
                                res++;
                            }
                        }
                    }
                }
            }
            return res;
        }

        size_t incBusStationMajority()
        {
            size_t res = 0;
            for (StationInfos::iterator it = stationInfos.begin(); it != stationInfos.end(); ++it)
            {
                if (it->transports == Transport::BUS && it->majority == 3)
                {
                    it->majority = 2;
                    it->isGoodForStart = true;
                    res++;
                }
            }
            return res;
        }


    };
}
#endif

