#include <cstring>
#include "stationgraph.h"
#include "dates.h"

namespace Pathfinder {

    int TariffStorage::LoadAeroexTariff(const std::string &LOG_FILE, const char *filePath, StationGraph &SG) {
        usedStations.resize(SG.st.stationInfos.size(), false);

        std::ifstream fin(filePath);
        if (fin.fail())
            return -1;

        Station fromOrig(-1), toOrig(-1);
        int reverse, transport;
        double price;
        for (int count = 0; fin >> fromOrig >> toOrig >> price >> reverse >> transport; count++) {
            Station from = SG.st.getStationInnerID(fromOrig);
            Station to = SG.st.getStationInnerID(toOrig);
            if (from < 0 || to < 0) {
                message(LOG_FILE, "thegraph reading ERROR (from < 0 || to < 0) with line: ", count);
                continue;
            }
            Transport::Type tr(Transport::LOCAL_TRAIN);
            if (transport == 2)
                tr = Transport::TRAIN_OR_LOCAL_TRAIN; //express

            Tariff tariff(0, &SG.st.stationInfos[from], &SG.st.stationInfos[to], (int) (price * 100 + .5), 0, tr);
            tariffs.push_back(tariff);
            if (reverse == 1) {
                Tariff tariff(0, &SG.st.stationInfos[to], &SG.st.stationInfos[from], (int) (price * 100 + .5), 0, tr);
                tariffs.push_back(tariff);
            }
        }
        return tariffs.size();
    }

    int TariffStorage::LoadThreadTariff(const std::string &LOG_FILE, const char *filePath, StationGraph &SG) {
        std::ifstream fin(filePath);
        if (fin.fail())
            return -1;

        std::string threadName, yearMask, currency;
        Station fromOrig(-1), toOrig(-1);
        double price;
        for (int count = 0; fin >> threadName >> fromOrig >> toOrig >> price >> yearMask >> currency; count++) {
            Station thread = SG.getThreadInnerID(threadName);
            if (thread < 0) {
                message(LOG_FILE, "thegraph reading ERROR (thread < 0) with line: ", count);
                continue;
            }
            Station from = SG.st.getStationInnerID(fromOrig);
            Station to = SG.st.getStationInnerID(toOrig);
            if (from < 0 || to < 0) {
                message(LOG_FILE, "thegraph reading ERROR (from < 0 || to < 0) with line: ", count);
                continue;
            }

            Tariff tariff(&SG._threadStart[thread], &SG.st.stationInfos[from], &SG.st.stationInfos[to],
                          (int) (price * 100 + .5), 0, Transport::NONE);
            if (yearMask != "NULL") {
                if (departures.empty())
                    departures.reserve(2001);
                Departures times;
                times.initGoodDayMask(yearMask);
                int i = 0;
                for (; i < departures.size() && !(departures[i] == times); i++);
                if (i < departures.size()) {
                    tariff.departures = &departures[i];
                }
                else {
                    if (departures.size() >= 2000) {
                        message(LOG_FILE, "ERROR! Not enought 2000 elements for tariffs departures: ",
                                departures.size());
                        return -1;
                    }
                    departures.push_back(times);
                    tariff.departures = &departures.back();
                }
            }
            tariffs.push_back(tariff);
        }
        return tariffs.size();
    }

    int TariffStorage::SaveTariffs(const std::string &LOG_FILE, const char *filePath, StationGraph &SG) {
        std::ofstream fout(filePath);
        if (fout.fail())
            return -1;
        for (int i = 0; i < tariffs.size(); i++) {
            std::string t = "";
            if (tariffs[i].threadStart)
                t = tariffs[i].threadStart->name;
            fout << tariffs[i].from->id << "\t" << tariffs[i].to->id << "\t" << tariffs[i].route
            << "\t" << t << "\t" << tariffs[i].tr << "\t" << tariffs[i].date << "\t" << tariffs[i].price << "\t" <<
            tariffs[i].departures << std::endl;
        }
        return tariffs.size();
    }

    int TariffStorage::SaveThreadTariffs(const std::string &LOG_FILE, const char *filePath, StationGraph &SG) {
        std::ofstream fout(filePath);
        if (fout.fail())
            return -1;
        int res = 0;
        for (TripInfos::const_iterator it = SG._tripInfos.begin(); it != SG._tripInfos.end(); ++it) {
            if (it->minTariff) {
                fout << it->threadStart->name << "\t" << it->minTariff->from->id << "\t" << it->minTariff->to->id
                << "\t" << it->minTariff->price << "\t" << (it->maxTariff - it->minTariff) << "\t" <<
                it->minTariff->departures << std::endl;
                res++;
            }
        }
        return res;
    }

    int TariffStorage::MinMaxTariff2TripInfo(StationGraph &SG) {
        sort();
        for (int i = 0; i < tariffs.size(); i++) {
            const ThreadStart *ts = tariffs[i].threadStart;
            const Tariff *begTariff = &tariffs[i];
            if (ts) //статические цены - автобусы
            {
                const TripInfo *tripinfo = ts->startTI;
                for (; tripinfo && tripinfo->stationInfoFrom != begTariff->from; tripinfo = tripinfo->nextTI);
                if (tripinfo) {
                    for (i = i + 1; i < tariffs.size() && tariffs[i].threadStart == begTariff->threadStart &&
                                    tariffs[i].from == begTariff->from; i++);
                    i--;
                    const Tariff *endTariff = &tariffs[i];

                    TripInfo *ti = &SG._tripInfos[tripinfo->id];
                    GetMinMaxTariff2TripInfo(begTariff, endTariff, ti);
                }
            }
            else if (begTariff->route) //цены от демона - самолеты и поезда
            {
                for (i = i + 1; i < tariffs.size() && tariffs[i].from == begTariff->from &&
                                tariffs[i].route == begTariff->route; i++);
                i--;
                const Tariff *endTariff = &tariffs[i];

                const TripInfoPtrs &info(SG._graph[tariffs[i].from->id1].startTIs);
                TripInfoPtrs::const_iterator infoEnd = info.end();
                for (TripInfoPtrs::const_iterator tip = info.begin(); tip != infoEnd; ++tip) {
                    const TripInfo *tripinfo = *tip;
                    if ((tripinfo->threadStart->route == begTariff->route)) {
                        TripInfo *ti = &SG._tripInfos[tripinfo->id];
                        GetMinMaxTariff2TripInfo(begTariff, endTariff, ti);
                    }
                }
            }
            else //электрички
            {
                for (i = i + 1;
                     i < tariffs.size() && tariffs[i].from == begTariff->from && tariffs[i].tr == begTariff->tr; i++);
                i--;
                const Tariff *endTariff = &tariffs[i];

                const TripInfoPtrs &info(SG._graph[tariffs[i].from->id1].startTIs);
                TripInfoPtrs::const_iterator infoEnd = info.end();
                for (TripInfoPtrs::const_iterator tip = info.begin(); tip != infoEnd; ++tip) {
                    const TripInfo *tripinfo = *tip;
                    if ((tripinfo->transport == begTariff->tr)) {
                        TripInfo *ti = &SG._tripInfos[tripinfo->id];
                        GetMinMaxTariff2TripInfo(begTariff, endTariff, ti);
                    }
                }
            }
        }
        return 0;
    }

    int TariffStorage::GetMinMaxTariff2TripInfo(const Tariff *begTariff, const Tariff *endTariff, TripInfo *tripinfo) {
        for (const TripInfo *ti = tripinfo; ti; ti = ti->nextTI) {
            usedStations[ti->stationInfoTo->id1] = true;
        }
        for (const Tariff *tariff = begTariff; tariff <= endTariff; tariff++) {
            if (usedStations[tariff->to->id1]) {
                if (tripinfo->minTariff == 0) {
                    tripinfo->minTariff = tariff;
                    break;
                }
            }
        }
        for (const Tariff *tariff = endTariff; tariff >= begTariff; tariff--) {
            if (usedStations[tariff->to->id1]) {
                if (tripinfo->maxTariff == 0) {
                    tripinfo->maxTariff = tariff;
                    break;
                }
            }
        }

        for (const TripInfo *ti = tripinfo; ti; ti = ti->nextTI) {
            usedStations[ti->stationInfoTo->id1] = false;
        }
        return 0;
    }

    typedef std::map<const StationInfo *, int> BestPriceMap;

    ///заполняем цены для оценок
    ///обязательно запускать ДО MinMaxTariff2TripInfo
    int TariffStorage::FillBestPrice(StationGraph &SG) {
        int res = 0;
        sort();
        BestPriceMap bestPriceMap;
        for (int i = 0; i < tariffs.size(); i++) {
            BestPriceMap::iterator fit = bestPriceMap.find(tariffs[i].to);
            if (fit == bestPriceMap.end() || fit->second > tariffs[i].price)
                bestPriceMap[tariffs[i].to] = tariffs[i].price;
            if (i == tariffs.size() - 1 || tariffs[i].from != tariffs[i + 1].from) {
                for (BestPriceMap::iterator it = bestPriceMap.begin(); it != bestPriceMap.end(); ++it) {
//				std::cout << "i = " << i << ", (from == NULL) = " << (tariffs[i].from == 0) << " " << tariffs[i].from << std::endl;
                    Station id = tariffs[i].from->id1;
//				std::cout << " from=" << id << std::endl;
                    SG._graph[id].bestPrices.push_back(BestPrice(it->first, it->second));
                    SG._invGraph[it->first->id1].bestPrices.push_back(BestPrice(tariffs[i].from, it->second));
                }
                res += bestPriceMap.size();
                bestPriceMap.clear();
            }
        }
        for (int i = 0; i < SG._invGraph.size(); i++) {
            std::sort(SG._invGraph[i].bestPrices.begin(), SG._invGraph[i].bestPrices.end());
        }
        return res;
    }

    int DumpSeatPriceLogParser::LoadCurrencyRates(const char *filePath) {
        std::ifstream fin(filePath);
        int res = 0;
        std::string currency;
        double rate;
        for (; fin >> currency >> rate; res++)
            currencyRates[currency] = rate;
        return res;
    }


    int DumpSeatPriceLogParser::LoadLog(const std::string &LOG_FILE, std::istream &fin, StationGraph &SG) {
        std::string str;
        int count = 0;
        char buf[100];
        for (count = 0; std::getline(fin, str, '\n'); count++) {
            message(LOG_FILE, "%s", str);
            char *tok = strtok(&str[0], "\t=");
            TariffLogKey tariffLogKey;
            TariffLogValue tariffLogValue;
            while (tok) {
                char *predtok = tok;
                if (!tok)
                    break;
                tok = strtok(NULL, "\t=");
                if (!tok)
                    break;
                if (strcmp(predtok, "timestamp") == 0) {
                    tariffLogValue.timestamp = Dates::fromString(tok);
                    if (tariffLogValue.timestamp < 0) {
                        message(LOG_FILE, "bad timestamp ", tok);
                        break;
                    }
                }
                else if (strcmp(predtok, "type") == 0) {
                    if (tok[0] == 'p'/*plane*/)
                        tariffLogKey.transport = Transport::PLANE;
                    else if (tok[0] == 't'/*train*/)
                        tariffLogKey.transport = Transport::TRAIN;
                    else if (tok[0] == 'b' && tok[1] == 'u'/*bus*/)
                        tariffLogKey.transport = Transport::BUS;
                    else
                        message(LOG_FILE, "unknown transport in log ", tok);
                }
                else if (strcmp(predtok, "date_forward") == 0) {
                    tariffLogKey.date = Dates::fromStringDate(tok);
                }
                else if (strcmp(predtok, "date_backward") == 0) {
                    if (tok[0] != 'N'/*None*/)
                        break;
                }
                else if (strcmp(predtok, "object_from_id") == 0) {
                    Station from = SG.st.getStationInnerID(atoi(tok));
                    if (from < 0)
                        break;
                    tariffLogKey.from = &SG.st.stationInfos[from];
                }
                else if (strcmp(predtok, "object_to_id") == 0) {
                    Station to = SG.st.getStationInnerID(atoi(tok));
                    if (to < 0)
                        break;
                    tariffLogKey.to = &SG.st.stationInfos[to];
                }
                else if (strcmp(predtok, "route_number") == 0 || strcmp(predtok, "route_uid") == 0) {
                    if (strchr(tok, ';'))
                        break;
                    RouteMap::iterator it = SG._routeMap.find(tok);
                    if (it == SG._routeMap.end())
                        break;
                    tariffLogKey.route = &SG._routes[it->second];
                }
                else if (endingWith(predtok, "tariff") || endingWith(predtok, "price")) {
                    double d;
                    int n = sscanf(tok, "%lf %s", &d, buf);
                    double rate = 1;
                    if (n == 2) {
                        CurrencyRates::iterator it = currencyRates.find(std::string(buf));
                        if (it == currencyRates.end()) {
                            message(LOG_FILE, "unknown currency in log %s", tok);
                            break;
                        }
                        rate = it->second;
                    }
                    int price = d * rate * 100 + 0.5;
                    tariffLogValue.price = std::min(tariffLogValue.price, price);
                }
            }//while (tok)
            if (tariffLogKey.empty() || tariffLogValue.empty())
                continue;
            TariffLogMap::iterator it = tariffLogMap.find(tariffLogKey);
            if (it == tariffLogMap.end())
                tariffLogMap[tariffLogKey] = tariffLogValue;
            else
                it->second = std::min(it->second, tariffLogValue);
        }
        return count;
    }

    int DumpSeatPriceLogParser::LoadLogWithoutGraph(const std::string &LOG_FILE, std::istream &fin, StationGraph &SG) {
        std::string str;
        int count = 0;
        char buf[100];
        for (count = 0; std::getline(fin, str, '\n'); count++) {
            char *tok = strtok(&str[0], "\t=");
            TariffLogKeyWithoutGraph tariffLogKey;
            TariffLogValue tariffLogValue;
            while (tok) {
                char *predtok = tok;
                if (!tok)
                    break;
                tok = strtok(NULL, "\t=");
                if (!tok)
                    break;
                if (strcmp(predtok, "timestamp") == 0) {
                    tariffLogValue.timestamp = Dates::fromString(tok);
                }
                else if (strcmp(predtok, "type") == 0) {
                    if (tok[0] == 'p'/*plane*/)
                        tariffLogKey.transport = Transport::PLANE;
                    else if (tok[0] == 't'/*train*/)
                        tariffLogKey.transport = Transport::TRAIN;
                    else if (tok[0] == 'b'/*bus*/)
                        tariffLogKey.transport = Transport::BUS;
                    else
                        message(LOG_FILE, "unknown transport in log %s", tok);
                }
                else if (strcmp(predtok, "date_forward") == 0) {
                    tariffLogKey.date = Dates::fromStringDate(tok);
                }
                else if (strcmp(predtok, "date_backward") == 0) {
                    if (tok[0] != 'N'/*None*/)
                        break;
                }
                else if (strcmp(predtok, "object_from_id") == 0) {
                    Station from = SG.st.getStationInnerIDWithoutGraph(atoi(tok));
                    if (from < 0)
                        break;
                    tariffLogKey.fromID = from;
                }
                else if (strcmp(predtok, "object_to_id") == 0) {
                    Station to = SG.st.getStationInnerIDWithoutGraph(atoi(tok));
                    if (to < 0)
                        break;
                    tariffLogKey.toID = to;
                }
                else if (strcmp(predtok, "route_number") == 0 || strcmp(predtok, "route_uid") == 0) {
                    if (strchr(tok, ';'))
                        break;
                    RouteMap::iterator it = SG._routeMap.find(tok);
                    if (it == SG._routeMap.end()) {
                        int nr = SG._routeMap.size();
                        SG._routeMap[tok] = nr;
                        SG._routes.push_back(Route());
                        SG._routes.back().name = tok;
                        SG._routes.back().id = nr;
                        tariffLogKey.routeID = nr;
                    }
                    else
                        tariffLogKey.routeID = it->second;
                }
                else if (endingWith(predtok, "tariff") || endingWith(predtok, "price")) {
                    double d;
                    int n = sscanf(tok, "%lf %s", &d, buf);
                    double rate = 1;
                    if (n == 2) {
                        CurrencyRates::iterator it = currencyRates.find(std::string(buf));
                        if (it == currencyRates.end()) {
                            message(LOG_FILE, "unknown currency in log %s", tok);
                            break;
                        }
                        rate = it->second;
                    }
                    int price = d * rate * 100 + 0.5;
                    tariffLogValue.price = std::min(tariffLogValue.price, price);
                }
            }//while (tok)
            if (tariffLogKey.empty() || tariffLogValue.empty())
                continue;
            TariffLogMapWithoutGraph::iterator it = tariffLogMapWithoutGraph.find(tariffLogKey);
            if (it == tariffLogMapWithoutGraph.end())
                tariffLogMapWithoutGraph[tariffLogKey] = tariffLogValue;
            else
                it->second = std::min(it->second, tariffLogValue);
        }
        return count;
    }

    int DumpSeatPriceLogParser::LoadLog(const std::string &LOG_FILE, const char *filePath, StationGraph &SG) {
        std::ifstream fin(filePath);
        if (fin.fail())
            return -1;
        return LoadLog(LOG_FILE, fin, SG);
    }

    int DumpSeatPriceLogParser::SaveLog(const std::string &LOG_FILE, const char *filePath, StationGraph &SG) {
        std::ofstream fout(filePath);
        if (fout.fail())
            return -1;
        for (TariffLogMap::iterator it = tariffLogMap.begin(); it != tariffLogMap.end(); ++it) {
            fout << "object_from_id=" << it->first.from->id << "\tobject_to_id=" << it->first.to->id;
            fout << "\ttype=" << Transport::toString(it->first.transport) << "\troute_number=" << it->first.route->name;
            fout << "\tdate_forward=" << Dates::toStringDate(it->first.date) << "\ttimestamp=" <<
            Dates::toString(it->second.timestamp, true, true)
            << "\ttariff=" << it->second.price / 100.0 << std::endl;

        }
        return 0;
    }

    int DumpSeatPriceLogParser::SaveLogWithoutGraph(const std::string &LOG_FILE, const char *filePath,
                                                    StationGraph &SG) {
        std::ofstream fout(filePath);
        if (fout.fail())
            return -1;
        for (TariffLogMapWithoutGraph::iterator it = tariffLogMapWithoutGraph.begin();
             it != tariffLogMapWithoutGraph.end(); ++it) {
            fout << "object_from_id=" << SG.st.stationInfos[it->first.fromID].id << "\tobject_to_id=" <<
            SG.st.stationInfos[it->first.toID].id;
            fout << "\ttype=" << Transport::toString(it->first.transport) << "\troute_number=" <<
            SG._routes[it->first.routeID].name;
            fout << "\tdate_forward=" << Dates::toStringDate(it->first.date) << "\ttimestamp=" <<
            Dates::toString(it->second.timestamp, true, true)
            << "\ttariff=" << it->second.price / 100.0 << std::endl;

        }
        return 0;
    }

    int TariffStorage::LogTariff2TariffStorage(TariffLogMap &map, StationGraph &SG) {
        for (TariffLogMap::iterator tariffit = map.begin(); tariffit != map.end(); ++tariffit) {
            if (SG.st.isTown(tariffit->first.from))
                SG._routes[tariffit->first.route->id].town2stations[tariffit->first.from] = StationInfoPtrs();
            if (SG.st.isTown(tariffit->first.to))
                SG._routes[tariffit->first.route->id].town2stations[tariffit->first.to] = StationInfoPtrs();
        }
        for (ThreadStarts::iterator it = SG._threadStart.begin(); it != SG._threadStart.end(); ++it) {
            if (!it->route->town2stations.empty()) {
                for (const TripInfo *tit = it->startTI; tit; tit = tit->nextTI) {
                    for (TeleportLinkPtrs::const_iterator town = tit->stationInfoFrom->towns.begin();
                         town != tit->stationInfoFrom->towns.end(); ++town) {
                        Town2Stations::iterator townst = SG._routes[it->route->id].town2stations.find((*town)->town);
                        if (townst != it->route->town2stations.end()) {
                            StationInfoPtrs::iterator sit = std::find(townst->second.begin(), townst->second.end(),
                                                                      (*town)->si);
                            if (sit == townst->second.end())
                                townst->second.push_back((*town)->si);
                        }
                    }
                    if (tit->nextTI == 0) {
                        for (TeleportLinkPtrs::const_iterator town = tit->stationInfoTo->towns.begin();
                             town != tit->stationInfoTo->towns.end(); ++town) {
                            Town2Stations::iterator townst = SG._routes[it->route->id].town2stations.find(
                                    (*town)->town);
                            if (townst != it->route->town2stations.end()) {
                                StationInfoPtrs::iterator sit = std::find(townst->second.begin(), townst->second.end(),
                                                                          (*town)->si);
                                if (sit == townst->second.end())
                                    townst->second.push_back((*town)->si);
                            }
                        }
                    }
                }
            }
        }

        int res = 0;
        for (TariffLogMap::iterator it = map.begin(); it != map.end(); ++it) {
            Tariff tariff(0, it->first.from, it->first.to, it->second.price, 0, it->first.transport, it->first.date);
            tariff.route = it->first.route;
            tariff.from = tariff.route->town2station(tariff.from);
            tariff.to = tariff.route->town2station(tariff.to);
            if (tariff.from == 0 || tariff.to == 0)
                continue;
            if (tariffs.empty())
                tariffs.push_back(tariff);
            else if (tariffs.back().from == tariff.from && tariffs.back().to == tariff.to
                     && tariffs.back().route == tariff.route) {
                if (tariffs.back().price != tariff.price)
                    tariffs.push_back(tariff);
            }
            else
                tariffs.push_back(tariff);

            res++;
        }
        return res;
    }


}
