#ifndef TariffH
#define TariffH

#include "tripinfo.h"

namespace Pathfinder
{

    class StationGraph;
    struct StationInfo;
    class Departures;

    typedef std::map<std::string, double> CurrencyRates;

    struct Tariff
    {
        const ThreadStart* threadStart;
        const StationInfo* from;
        const StationInfo* to;
        int price;
        const Departures* departures;
        Transport::Type tr;
        const Route* route;
        int date;
        Tariff(const ThreadStart* ts = 0, const StationInfo* f = 0, const StationInfo* t = 0, const int p = 0,
               const Departures* dep = 0, Transport::Type tr0 = Transport::NONE, int d = -1, const Route* r = 0)
                :threadStart(ts), from(f), to(t), price(p), departures(dep), tr(tr0), route(0), date(d)
        {}
        inline bool operator<(const Tariff& t) const
        {
            if (from != t.from)
                return from < t.from;
            if (threadStart != t.threadStart)
                return threadStart < t.threadStart;
            if (tr != t.tr)
                return tr < t.tr;
            if (route != t.route)
                return route < t.route;
            if (to != t.to)
                return to < t.to;
            if (date != t.date)
                return date < t.date;
            return price < t.price;
        }
        inline static bool cmpByFrom(const Tariff& t0, const Tariff& t1)
        {
            return t0.from < t1.from;
        }

    };

    struct TariffLogKey
    {
        const StationInfo* from;
        const StationInfo* to;
        const Route* route;
        int date;
        Transport::Type transport;
        TariffLogKey(const StationInfo* f = 0, const StationInfo* t = 0, int d = -1, Transport::Type tr = Transport::NONE)
                : from(f), to(t), route(0), date(d), transport(tr)
        {}
        bool empty()
        {
            if (!from || !to || !route || date < 0 || transport == Transport::NONE)
                return true;
            return false;
        }
        inline bool operator<(const TariffLogKey& t) const
        {
            if (from != t.from)
                return from < t.from;
            if (to != t.to)
                return to < t.to;
            if (date != t.date)
                return date < t.date;
            if (transport != t.transport)
                return transport < t.transport;
            return route < t.route;
        }
    };

    struct TariffLogKeyWithoutGraph
    {
        Station fromID;
        Station toID;
        int routeID;
        int date;
        Transport::Type transport;
        TariffLogKeyWithoutGraph(int d = -1, Transport::Type tr = Transport::NONE)
                : fromID(-1), toID(-1), routeID(-1), date(d), transport(tr)
        {}
        bool empty()
        {
            if (fromID<0 || toID<0 || routeID<0 || date < 0 || transport == Transport::NONE)
                return true;
            return false;
        }
        inline bool operator<(const TariffLogKeyWithoutGraph& t) const
        {
            if (fromID != t.fromID)
                return fromID < t.fromID;
            if (toID != t.toID)
                return toID < t.toID;
            if (date != t.date)
                return date < t.date;
            if (transport != t.transport)
                return transport < t.transport;
            return routeID < t.routeID;
        }
    };

    struct TariffLogValue
    {
        int price;
        int timestamp;
        TariffLogValue(int p = MAX_PRICE, int t = -1)
                : price(p), timestamp(t)
        {}
        bool empty()
        {
            if (price == MAX_PRICE || timestamp < 0)
                return true;
            return false;
        }
        inline bool operator<(const TariffLogValue& t) const
        {
            if (timestamp > t.timestamp)
                return true;
            if (timestamp == t.timestamp && price < t.price)
                return true;
            return false;
        }
    };
    typedef std::map<TariffLogKey, TariffLogValue> TariffLogMap;
    typedef std::map<TariffLogKeyWithoutGraph, TariffLogValue> TariffLogMapWithoutGraph;

    class DumpSeatPriceLogParser
    {
    public:
        TariffLogMap tariffLogMap;
        TariffLogMapWithoutGraph tariffLogMapWithoutGraph;
        CurrencyRates currencyRates;
    public:
        int LoadCurrencyRates(const char* filePath);
        int LoadLog(const std::string& LOG_FILE, const char* filePath, StationGraph& SG);
        int LoadLog(const std::string& LOG_FILE, std::istream& fin, StationGraph& SG);
        int LoadLogWithoutGraph(const std::string& LOG_FILE, std::istream& fin, StationGraph& SG);
        int SaveLog(const std::string& LOG_FILE, const char* filePath, StationGraph& SG);
        int SaveLogWithoutGraph(const std::string& LOG_FILE, const char* filePath, StationGraph& SG);
        int Log2Tariffs();
    };

    class TariffStorage
    {
        std::vector<Tariff> tariffs;
        std::vector<Departures> departures;
        std::vector<bool> usedStations;
    public:
        int LoadAeroexTariff(const std::string& LOG_FILE, const char* filePath, StationGraph& SG);
        int LoadThreadTariff(const std::string& LOG_FILE, const char* filePath, StationGraph& SG);
        int SaveTariffs(const std::string& LOG_FILE, const char* filePath, StationGraph& SG);
        int SaveThreadTariffs(const std::string& LOG_FILE, const char* filePath, StationGraph& SG);
        void sort()
        {
            std::stable_sort(tariffs.begin(), tariffs.end());
        }
        void sortByFrom()
        {
            std::stable_sort(tariffs.begin(), tariffs.end(), Tariff::cmpByFrom);
        }
        bool empty() const
        {
            return tariffs.empty();
        }

        int MinMaxTariff2TripInfo(StationGraph& SG);
        int GetMinMaxTariff2TripInfo(const Tariff* begTariff, const Tariff* endTariff, TripInfo* tripinfo);
        ///   MinMaxTariff2TripInfo
        int FillBestPrice(StationGraph& SG);

        int LogTariff2TariffStorage(TariffLogMap& map, StationGraph& SG);

    };

}

#endif

