#include "thegraph.h"

namespace Pathfinder
{
    //=--------------------------TheGraph
    std::string TheGraph::search(const std::string &LOG_FILE, Condition &condition) const
    {
        std::string resInit = search1(LOG_FILE, condition);
        if (!resInit.empty())
            return resInit;
        std::string resFirstStep = search2(LOG_FILE, condition);
        if (!resFirstStep.empty())
            return resFirstStep;
        std::string resMainStep = search3(LOG_FILE, condition);
        if ((resMainStep.empty() || resMainStep=="norm") && _pathStorage.wasExecutionStoppedByTimeLimit())
            resMainStep = PFResult::toString(PFResult::max_time);
        return resMainStep;
    }

    // поиск (дополнительно передается список ниток, которые не используем)
    std::string TheGraph::search1(const std::string &LOG_FILE, Condition &condition) const
    {
        // здесь будем хранить фронт - т.е. набор текущих недостроенных путей

        if (condition._from.empty() || condition._to.empty())
            return "bad input";

        message(LOG_FILE, "init begin with conditions: ", condition.toString().c_str());

        // первая итерация
        const PFResult::Type resInit = _searchInit(condition);

        message(LOG_FILE, "res init: ", PFResult::toString(resInit).c_str());
        return "";
    }

    std::string TheGraph::search2(const std::string &LOG_FILE, const Condition &condition) const
    {
        // первая итерация
        const PFResult::Type resFirstStep = _firstStep(condition);

        message(LOG_FILE, "first step: ", PFResult::toString(resFirstStep).c_str());

		return "";
	}
	 
    std::string TheGraph::search3(const std::string &LOG_FILE, const Condition &condition) const
    {
       // остальные итерации
        const PFResult::Type resMain = _searchMain(condition);
#ifdef DEBUG_ROUTE
        if (_pathStorage._needDebugLog)
        {
            std::fstream out(DEBUG_LOG_FILE, std::ios::out | std::ios::app);
            for (StationInfoPtrs::iterator dsit = _pathStorage._debugStations.begin(); dsit != _pathStorage._debugStations.end(); ++dsit)            
            {
                out << "result for station " << (*dsit)->id << " " << (*dsit)->name << std::endl;
                stationResultToDebugString(out, *dsit);
                out << std::endl;
            }
        }
#endif

#ifndef FOR_WIZARD
        _pathStorage._result.cutByOneDay(LOG_FILE, condition.getTime(), condition.getTime()+condition.getBoarding());
#endif

        return PFResult::toString(resMain);
    }

    // поиск, первая итерация
    // возвращаем сообщение об ошибке
    PFResult::Type TheGraph::_searchInit(Condition &condition) const
    {
        // если они плохие, то возвращаемся
        if (condition.getFrom().empty() || condition.getTo().empty())
            return PFResult::bad_input;

        // если станция отправления и станция назначения находятся в одном городе, то возвращаем "trivial"
        for (StationInfoPtrs::const_iterator it1 = condition.getFrom().begin(); it1 != condition.getFrom().end(); ++it1)
        {
            for (StationInfoPtrs::const_iterator it2 = condition.getTo().begin(); it2 != condition.getTo().end(); ++it2)
                if (*it1 == *it2)
                    return PFResult::trivial;
            _pathStorage._station2Info[(*it1)->id1].front = 0;
        }

        // вычисляем оценку снизу на веса
        condition.initLocs();
        double dist = condition.getSphereDist();
        int tripLong = 10;
        if (dist > 0)
            tripLong = std::max(2, std::min(10, (int)(dist/(40*24)+0.5))); 
        int iFront = _pathStorage.initEstimations(SG, condition, condition.getTransports()
            , condition.begFirstDay(), condition.begLastDay(), condition.endTripDay(tripLong));

        // оценка веса маршрута до станции назначения
        const StationInfoPtrs& froms = condition.getFrom();
        const Estimation origin =_pathStorage.getMinStationEst(froms);
        int ch = origin.changes;
//        if (ch <= 3)
            ch++;

#ifdef DEBUG_ROUTE
        if (origin.empty() && _pathStorage._needDebugLog) {
            std::fstream out(DEBUG_LOG_FILE, std::ios::out | std::ios::app);
            out << "init res = impossible" << std::endl;
            if (_pathStorage.wasExecutionStoppedByTimeLimit())
                out << "time limit exceeded" << std::endl;
            for (Station2Info::iterator it = _pathStorage._station2Info.begin();
                 it != _pathStorage._station2Info.end(); ++it)
                if (!it->est.empty()) {
                    out << it->si->id << " " << it->si->name << " " << it->si->transports << " ";
                    it->est.toStream(out);
                    out << std::endl;
                }
        }
#endif

        if (iFront == MAX_CHANGES && origin.empty())
        {
            // станция назначения недостижима, сообщаем об этом
            return PFResult::impossible;
        }
        condition.setChanges(ch);
        if (iFront == MAX_CHANGES)
        {
            return PFResult::max_time;
        }
#ifdef DEBUG_ROUTE
        if (_pathStorage._needDebugLog)
        {
            std::fstream out(DEBUG_LOG_FILE, std::ios::out | std::ios::app);
            out << "dist=" << dist << " km TripLong=" << tripLong << " days" << std::endl;
            for (StationInfoPtrs::iterator dsit = _pathStorage._debugStations.begin(); dsit != _pathStorage._debugStations.end(); ++dsit)
            {
                out << (*dsit)->id << " " << (*dsit)->name << " ";
                _pathStorage.getStationEst(*dsit).toStream(out);
                out << std::endl;
            }
        }
#endif
        return PFResult::ok;
    }

    PFResult::Type TheGraph::_firstStep(const Condition &condition) const
    {
        // время для пересадки 
        const int begTime = condition.getTime();
        const int endTime = condition.getEndTime();

        // первая итерация
        const StationInfoPtrs& froms = condition.getFrom();
        for (size_t m = 0; m < froms.size(); ++m)
        {
            Vertex * beg = _pathStorage.getFreeVertex();
            if (beg == 0)
                return PFResult::mem_vertex;
            const StationInfo* currentStation = froms[m];

#ifdef DEBUG_ROUTE
            if (_pathStorage._needDebugLog) {
                std::fstream out(DEBUG_LOG_FILE, std::ios::out | std::ios::app);
                out << "START FULL from " << currentStation->name;
            }
            size_t oldFront = _pathStorage._pathFront.size();
#endif

            // инициализируем вершину
            // создаем вершину телепортации (бывает телепортация сама в себя)
            // начальное и конечное время в этой вершине - это время когда можно начать движение на следующем транспорте
            beg->setStation(currentStation, _pathStorage.getStationEst(currentStation->id1));
            beg->setInfo(begTime, endTime, Transport::NONE);
            if (SG.st.isTown(condition._fromStation))
                beg->setTeleportationTown(condition._fromStation);

            std::vector<int> departureTimes;
            Path currentPath;
            currentPath.add(beg);

            // начинаем перебирать соседей
            const Neighbours& neighbours = SG._graph[currentStation->id1].sais;
            Neighbours::const_iterator itEnd = neighbours.end();
            for (Neighbours::const_iterator it = neighbours.begin(); it != itEnd; ++it)
            {
                // зафиксировали соседа
                const StationAndInfo& sai(*it);

                const size_t tsz(sai.info.size());
                // перебираем сообщения
                for (size_t j = 0; j < tsz; ++j)
                {
                    const TripInfo* tripinfo = sai.info[j];
                    if (!tripinfo->stopOnFirstStation())
                        continue;
                    if (currentPath.getLastVertex()->getTeleportationTown() && !tripinfo->townGoodForStart(currentPath.getLastVertex()->getTeleportationTown(), TeleportLink::ALL))
                        continue;

                    // пропускаем лишний транспорт
                    if ((condition.getTransports() & tripinfo->transport) == Transport::NONE)
                        continue;

                    _pathStorage._timer.stop();
                    if (_pathStorage._timer.getTime() > _pathStorage._timeLimitFirst) {
                        _pathStorage.stopExecutionByTimeLimit();
                        return PFResult::ok;
                    }

                    const TripInfo* lastTI = 0;
                    for (const TripInfo* ti = tripinfo; ti; ti = ti->nextTI)
                    {
                        const Estimation& estT = _pathStorage.getTripInfoEst(ti);
                        if (!estT.empty())
                            lastTI = ti;
                        else
                        {
                            const Estimation& estS = _pathStorage.getStationEst(ti->stationInfoTo->id1);
                            if (!estS.empty())
                            {
                                Estimation& e = _pathStorage.setTripInfoEst(ti, estS);
                                e.changes++;
                                e.trPoint += Transport::getWeight(tripinfo->transport);
                                int transportDelay = Transport::getMinDelay(condition._minDelayOut, tripinfo->transport);
                                e.weight += transportDelay;
                                e.weight += tripinfo->duration;
                                lastTI = ti;
                            }
                        }
                    }

                    if (lastTI && lastTI >= tripinfo)
                    {
                        const TripInfo* bestTI = lastTI;
                        Estimation bestEst( _pathStorage.getTripInfoEst(lastTI) );
                        for (const TripInfo* ti = lastTI->predTI; ti && ti >= tripinfo; ti = ti->predTI)
                        {
                            bestEst.weight += ti->duration + ti->nextTI->stopTimeBefore;
                            const Estimation& currEst = _pathStorage.getTripInfoEst(ti);
                            if (bestEst < currEst)
                                _pathStorage.setTripInfoEst(ti, bestEst);
                            if (currEst < bestEst)
                            {
                                bestEst = currEst;
                            }
                        }

                        int begTime1 = std::max(begTime, currentPath.getLastVertex()->getBeginStationTime());
                        
                        // выбираем время отправления и запоминаем время прибытия
                        // перебираем дни
                        int nd = tripinfo->getDepartureTimesBetween(begTime1, endTime, departureTimes);
//#ifdef DEBUG_PRINT
//{
//    std::fstream out("log.txt", std::ios::out | std::ios::app);
//    out << "departures " << tripinfo->threadStart->name << "  begTime=" << begTime1 << " endTime=" << endTime << " nd=" << nd 
//        << "int_beg=" << tripinfo->threadStart->interval_begin_time  
//        << "int_end=" << tripinfo->threadStart->interval_end_time  
//        << "int_delta=" << tripinfo->threadStart->interval_delta 
//        << std::endl;
//}
//#endif
                        for (int idt = 0; idt < nd; idt++)
                        {
                            int departureTime = departureTimes[idt];

                            Path path = currentPath.copy();
                            PFResult::Type res = PFResult::ok;
                            const Vertex* startVertex = 0;
                            for (const TripInfo* ti = tripinfo; ti && ti<=bestTI && res == PFResult::ok; ti = ti->nextTI)
                            {
                                if (ti == tripinfo)
                                    res = _addEdge(path, ti, departureTime, condition, startVertex);
                                else
                                    res = _prolongEdge(path, condition, startVertex);
                                if (startVertex == 0)
                                    startVertex = path.getLastVertex();
                                if (res == PFResult::result || res == PFResult::bad_path)
                                    continue;
                                if (res == PFResult::ok)
                                {
                                    const Estimation& est = _pathStorage.getTripInfoEst(ti->id);
                                    if (!est.empty())
                                        res = _addExit(path, condition, startVertex);
                                }
                                if (res != PFResult::ok)
                                    return res;
                            }
                            if (bestTI < lastTI && res == PFResult::ok)
                            {
                                if (!_pathStorage.addPathToFront(path))
                                    return PFResult::mem_path;
                            }
                        }
                    }
                }
            }
#ifdef DEBUG_ROUTE
            if (_pathStorage._needDebugLog){
                std::fstream out(DEBUG_LOG_FILE, std::ios::out | std::ios::app);
                out << " frontInc=" << (_pathStorage._pathFront.size()-oldFront) << std::endl;
            }
#endif
        }
        return PFResult::ok;
    }

    // поиск маршрутов (последующие итерации)
    // функция _searchMain вызывается после _searchInit
    // вход: условия поиск (condition), дополнительные параметры (optimization),
    //       в result хаписываются результаты работы поиска (там уже может что-то быть от _searchInit),
    //       frontPtr - фронт, оставшийся после _searchInit,
    // выход: результаты хранятся в result, кроме того возвращается строка, описывающая причину завершения поиска:
    //TODO: указать возможные строки и их смысл
    PFResult::Type TheGraph::_searchMain(const Condition &condition) const
    {
        // основной цикл
        while ( !_pathStorage._pathFront.empty() )
        {
            PFResult::Type res = _pathStorage.checkMaxConditions();
            if (res != PFResult::ok)
                return res;

            // выталкиваем путь из фронта для продолжения
            Path currentPath = _pathStorage._pathFront.top();
            _pathStorage.pathFrontPop();

            PFResult::Type stepres = _searchMainStep(currentPath, condition);
            if (stepres != PFResult::ok)
                return stepres;

            _pathStorage.erasePath(currentPath);
        }
        // все прошло нормально, обработали все пути
        return PFResult::norm;
    }

    //добавляем к текущему пересадочному ребру, все телепортационные ребра
    PFResult::Type TheGraph::_addTeleportations(Path &currentPath, const Condition &condition) const
    {
        const StationInfo* si = currentPath.getLastVertex()->getEndStationInfo();
        const StationInfo* predStation = 0;
        if (currentPath.getLastVertex()->getPrevious())
            predStation = currentPath.getLastVertex()->getPrevious()->getEndStationInfo();
        //телепортацию делаем только из пересадочного ребра, изменяя последний вертекс
        //самый первый вертекс пути - пересадочный, но делать из него телепортации не надо, они уже есть в очереди
        if (si!=predStation || predStation == 0)
            return PFResult::ok;
        //если предыдущим ездовым ребром мы не выехали из города, где были
        //не стоит делать телепортации сейчас, они уже сделаны
        //пересадочное же ребро - уместно
        const StationInfo* predPredStation = 0;  
        if (currentPath.getLastVertex()->getPrevious()->getPrevious())
            predPredStation = currentPath.getLastVertex()->getPrevious()->getPrevious()->getEndStationInfo();

        // генерируем маршруты
        for (size_t x = 0; x < si->costations.size(); ++x)
            if (si != si->costations[x].town)
        {
            const StationInfo* cosi = si->costations[x].town;
            const int teleportTime = si->costations[x].time2center;
//#ifdef DEBUG_ROUTE
//            if (_pathStorage._needDebugLog) {
//                std::fstream out(DEBUG_LOG_FILE, std::ios::out | std::ios::app);
//                out << "TELE CO from " << si->name << " to " << cosi->name;
//            }
//            size_t oldFront = _pathStorage._pathFront.size();
//#endif

            // готовим новый путевой сегмент
            Vertex* end = _pathStorage.getFreeVertex();
            // если места нет, возвращаемся
            if (end == 0)
                return PFResult::mem_vertex;
            *end = *currentPath.getLastVertex();

            // добавляем путевой сегмент к пути
            end->setStation(cosi, _pathStorage.getStationEst(cosi->id1));
            end->incTeleportations();
            end->getObtForChange().weight += teleportTime;
            end->addBeginStationTime(teleportTime);
            if (end->getSegmentPrice() < 0)
                end->setSegmentPrice(0);

            Path path = currentPath.copy();
            path.changeLastVertex(end);

            if (!_pathStorage.goodPath(path, condition.getChanges()))
                continue;

            // выделяем память
            if (!_pathStorage.addPathToFront(path))
            {
                // если места нет, возвращаемся
                return PFResult::mem_path;
            }
//#ifdef DEBUG_ROUTE
//            if (_pathStorage._needDebugLog){
//                std::fstream out(DEBUG_LOG_FILE, std::ios::out | std::ios::app);
//                out << " frontInc=" << (_pathStorage._pathFront.size()-oldFront) << std::endl;
//            }
//#endif
        }
        return PFResult::ok;
    }

    //================================================================================================

    //добавляем к текущему пересадочному ребру, телепортационное ребро в центр города
    PFResult::Type TheGraph::_addTeleportation2TownCenter(Path &currentPath, const Condition &condition,
                                                          const Vertex *startVertex) const
    {
#ifdef DEBUG_ROUTE
        if (_pathStorage._needDebugLog) {
            std::fstream out(DEBUG_LOG_FILE, std::ios::out | std::ios::app);
            out << "TELE TO CENTER from " << currentPath.getLastVertex()->getEndStationInfo()->name;
        }
        size_t oldFront = _pathStorage._pathFront.size();
#endif
        if (currentPath.getLastVertex()->getNoTeleportations())
            return PFResult::ok;
        const TripInfo* tripinfo = currentPath.getLastVertex()->getPrevious()->getTripInfo();//.getLastTripInfo();
        if (tripinfo == 0)
            return PFResult::ok;
        const StationInfo* si = currentPath.getLastVertex()->getEndStationInfo();
        // идем в центры всех городов
        TeleportLinkPtrs::const_iterator itEnd = tripinfo->townsGoodForOut.end();
        for (TeleportLinkPtrs::const_iterator it = tripinfo->townsGoodForOut.begin(); it != itEnd; ++it)
        {
            const StationInfo* town = (*it)->town;
            int numCostationsGoodForIn = town->getNumCostationsGoodForIn(si, condition.getTransports());
            if (numCostationsGoodForIn < 1)
                continue;
            // готовим новый путевой сегмент
            Vertex* end = _pathStorage.getFreeVertex();
            // если места нет, возвращаемся
            if (end == 0)
                return PFResult::mem_vertex;
            *end = *currentPath.getLastVertex();

            // добавляем путевой сегмент к пути
            end->setStation(town, _pathStorage.getStationEst(town->id1));
            end->incTeleportations();
            end->getObtForChange().weight += (*it)->time2center;
#ifdef DISCOMFORT_ON
            end->addDiscomfortChecked( (*it)->time2center, "teleport" );
#endif
            end->addBeginStationTime((*it)->time2center);
            if (end->getSegmentPrice() < 0)
                end->setSegmentPrice(0);

            Path path = currentPath.copy();
            path.changeLastVertex(end);

            if (!_pathStorage.goodPath(path, condition.getChanges()))
                return PFResult::ok;

            // выделяем память
            if (!_pathStorage.addPathToFront(path))
            {
                // если места нет, возвращаемся
                return PFResult::mem_path;
            }
        }
#ifdef DEBUG_ROUTE
        if (_pathStorage._needDebugLog){
            std::fstream out(DEBUG_LOG_FILE, std::ios::out | std::ios::app);
            out << " frontInc=" << (_pathStorage._pathFront.size()-oldFront) << std::endl;
        }
#endif
        return PFResult::ok;
    }

    //добавляем к текущему пересадочному ребру, телепортационное ребро в центр города
    PFResult::Type TheGraph::_addTeleportationFromTownCenter(Path &currentPath, const Condition &condition) const
    {
        const StationInfo* predStation = currentPath.getLastVertex()->getPrevious()->getEndStationInfo();
        const StationInfo* town = currentPath.getLastVertex()->getEndStationInfo();
#ifdef DEBUG_ROUTE
        if (_pathStorage._needDebugLog) {
            std::fstream out(DEBUG_LOG_FILE, std::ios::out | std::ios::app);
            out << "TELE FROM CENTER of " << town->name << std::endl;
        }
        size_t oldFront = _pathStorage._pathFront.size();
#endif
        // идем в центры всех городов
        TeleportLinks::const_iterator itEnd = town->costations.end();
        for (TeleportLinks::const_iterator it = town->costations.begin(); it != itEnd; ++it)
        if (it->isGoodForStart && it->si != predStation) {
            const StationInfo *si = it->si;
            // готовим новый путевой сегмент
            Vertex *end = _pathStorage.getFreeVertex();
            // если места нет, возвращаемся
            if (end == 0)
                return PFResult::mem_vertex;

            // добавляем путевой сегмент к пути
            end->setStation(si, _pathStorage.getStationEstAndGo(si));
            int minDelay = it->time2center;
            end->setInfo(currentPath.getLastVertex()->getBeginStationTime() + minDelay, currentPath.getLastTime(),
                         Transport::NONE);
            end->getObtForChange().weight += minDelay;
            end->addPriceChecked(TELEPORTATION_PRICE);
            end->setSegmentPrice(TELEPORTATION_PRICE);
            end->addDiscomfortChecked(minDelay, "teleport");
            Path path = currentPath.copy();
            path.add(end);

            if (!_pathStorage.goodPath(path, condition.getChanges()))
                continue;

            // выделяем память
            if (!_pathStorage.addPathToFront(path)) {
                // если места нет, возвращаемся
                return PFResult::mem_path;
            }
        }
#ifdef DEBUG_ROUTE
        if (_pathStorage._needDebugLog) {
            std::fstream out(DEBUG_LOG_FILE, std::ios::out | std::ios::app);
            out << " frontInc=" << (_pathStorage._pathFront.size()-oldFront) << std::endl;
        }
#endif
        return PFResult::ok;
    }

    //добавляем к текущему пересадочному ребру, телепортационное ребро в центр города
    PFResult::Type TheGraph::_addTeleportationFromTownCenterFull(Path &currentPath, const Condition &condition) const
    {
        const StationInfo* town = currentPath.getLastVertex()->getEndStationInfo();
        StationCalcInfo& stationCalcInfo = _pathStorage._station2Info[town->id1];
        GrowableList growableList;
        stationCalcInfo.bestPaths.getGrowableList(growableList);
        if (growableList.empty())
            return PFResult::ok;

#ifdef DEBUG_ROUTE
        if (_pathStorage._needDebugLog) {
            std::fstream out(DEBUG_LOG_FILE, std::ios::out | std::ios::app);
            out << "TELE FULL from " << town->name << " growables=" << growableList.size()
            << " obt=" << growableList[0].vertex->getObt()
            << " est=" << growableList[0].vertex->getEst() << std::endl;
        }
        size_t oldFront = _pathStorage._pathFront.size();
#endif

        GrowableList::iterator itEndGL = growableList.end(); 
        for (GrowableList::iterator it = growableList.begin(); it != itEndGL; ++it)
        {
            Path path(it->vertex);
            PFResult::Type res = _addTeleportationFromTownCenter(path, condition);
            if (res != PFResult::ok)
                return res;
        }

        for (GrowableList::iterator it = growableList.begin(); it != itEndGL; ++it)
            it->vertex->decreaseAllReferences();
#ifdef DEBUG_ROUTE
        if (_pathStorage._needDebugLog) {
            std::fstream out(DEBUG_LOG_FILE, std::ios::out | std::ios::app);
            out << " frontInc=" << (_pathStorage._pathFront.size()-oldFront) << std::endl;
        }
#endif
        return PFResult::ok;
    }

    //================================================================================================

    PFResult::Type TheGraph::_addExit(Path &currentPath, const Condition &condition, const Vertex *startVertex) const
    {
        const TripInfo* tripinfo = currentPath.getLastTripInfo();
        if (tripinfo == 0)
            return PFResult::ok;
        const StationInfo* neighbour = tripinfo->stationInfoTo;

        //если паравоз не останавливается, слезть с него не получится
        if (!tripinfo->stopOnLastStation())// nextTI && !tripinfo->nextTI->stopOnFirstStation())
            return PFResult::ok;

        bool canTeleport = condition._canTeleport;
        //если пересадки в городах начала и конца пути запрещены, можно только попасть в резалт
        if (!condition._canChangeInAnyTown && !condition._canChangeInStartTown && !condition._canChangeInFinishTown)
        {
            if (SG.st.fromTowns(neighbour, condition._endingTowns))
                return PFResult::ok;
        }
        else
        {
            const StationInfo* startTown = SG.st.fromStationsTowns(neighbour, condition._from);
            const StationInfo* finishTown = SG.st.fromStationsTowns(neighbour, condition._to);

            if (condition._canChangeInAnyTown) {
                // телепортироваться через центр города в городах высадки и посадки запрещено
                if (startTown || finishTown)
                    canTeleport = false;
            }
            else {
                if (startTown) {
                    if (!condition._canChangeInStartTown)
                        return PFResult::ok;
                    else
                        canTeleport = false;
                }
                if (finishTown) {
                    if (!condition._canChangeInFinishTown)
                        return PFResult::ok;
                    else
                        canTeleport = false;
                }
                if (!startTown && !finishTown && SG.st.fromOneTown(condition._from[0], condition._to[0]) != 0)
                    return PFResult::ok;
            }
        }
        if (!tripinfo->stationInfoTo->towns.empty() && tripinfo->townsGoodForOut.empty())
            canTeleport = false;

        // готовим новый путевой сегмент
        Vertex* end = _pathStorage.getFreeVertex();
        // если места нет, возвращаемся
        if (end == 0)
            return PFResult::mem_vertex;

        end->setStation(tripinfo->stationInfoTo, _pathStorage.getStationEstAndGo(tripinfo->stationInfoTo));
        int minDelay = Transport::getMinDelay(condition._minDelayOut, tripinfo->transport);
        end->setInfo(currentPath.getLastTime()+minDelay
            , currentPath.getLastTime()+minDelay+condition.getMaxDelay()
            , Transport::NONE);
        end->getObtForChange().weight = minDelay;

        end->setNoTeleportations(!canTeleport);
        const int price = (condition._usePrice) ? SG.getPrice(startVertex->getTripInfo(), neighbour, startVertex->getBeginStationTime()) : MAX_PRICE;
        end->addPriceChecked(price);
        end->setSegmentPrice(price);

#ifdef DISCOMFORT_ON
        int nightTime = 0;
        if (startVertex->getEndStationInfo()->timeZone)
            nightTime = startVertex->getEndStationInfo()->timeZone->getNightTime(startVertex->getBeginStationTime(), currentPath.getLastTime());
#ifdef DISCOMFORT_STR_ON
        std::string dStr;
        int d =  Transport::getDiscomfort(tripinfo->transport, currentPath.getLastTime()-startVertex->getBeginStationTime(), nightTime, tripinfo->threadStart->isMain(), dStr);
        end->setDiscomfortChecked(d, dStr.c_str());
#else
        int d =  Transport::getDiscomfort(tripinfo->transport, currentPath.getLastTime()-startVertex->getBeginStationTime(), nightTime, tripinfo->threadStart->isMain());
        end->setDiscomfortChecked(d);
#endif
#endif
        if (neighbour->majority > 1)
            end->addDiscomfortChecked(1, "maj>1");

        // добавляем путевой сегмент к пути
        Path path = currentPath.copy();
        path.add(end);

        PFResult::Type res = _addTeleportations(path, condition);
        if (res != PFResult::ok)
            return res;
        if (canTeleport)
        {
            res = _addTeleportation2TownCenter(path, condition, startVertex);
            if (res != PFResult::ok)
                return res;
        }

        if (!_pathStorage.goodPath(path, condition.getChanges()))
            return PFResult::ok;

        // выделяем память
        if (!_pathStorage.addPathToFront(path))
        {
            // если места нет, возвращаемся
            return PFResult::mem_path;
        }

        return PFResult::ok;
    }

    //================================================================================================

    //добавляем к текущему пути ребро, и все телепортационные ребра из конца ребра
    //телепортационное ребро может быть само в себя
    //но число пересадок увеличивается уже на телепортации
    PFResult::Type TheGraph::_prolongEdge(Path &currentPath, const Condition &condition, const Vertex *startVertex,
                                          const bool doCheck) const
    {
        if (currentPath.getLastThread() == 0)
            return PFResult::prolong_teleportation;

        const TripInfo* tripinfo = currentPath.getLastVertex()->getTripInfo()->nextTI;
        if (tripinfo == 0)
            return PFResult::bad_path;

        return _addEdge1(currentPath, tripinfo, currentPath.getLastTime(), condition, startVertex, doCheck,
                         currentPath.getLastTime() + tripinfo->stopTimeBefore);
    }

    //добавляем к текущему пути ребро, и все телепортационные ребра из конца ребра
    //телепортационное ребро может быть само в себя
    //но число пересадок увеличивается уже на телепортации
    PFResult::Type TheGraph::_addEdge(Path &currentPath, const TripInfo *tripinfo, int departureAfter,
                                      const Condition &condition, const Vertex *startVertex, const bool doCheck) const
    {
        int dt = tripinfo->getDepartureTimeAfter2(departureAfter);
        if (dt < 0)
        {
            return PFResult::bad_path;
        }

        PFResult::Type res = _addEdge1(currentPath, tripinfo, currentPath.getLastTime(), condition, startVertex,
                                       doCheck, dt);
        return res;
    }

    //================================================================================================

    //добавляем к текущему пути ребро, и все телепортационные ребра из конца ребра
    //телепортационное ребро может быть само в себя
    //но число пересадок увеличивается уже на телепортации
    PFResult::Type TheGraph::_addEdge1(Path &currentPath, const TripInfo *tripinfo, int departureAfter,
                                       const Condition &condition, const Vertex *startVertex, const bool doCheck,
                                       const int dt) const
    {
        int at = dt + tripinfo->duration;
        // готовим новый путевой сегмент
        Vertex* end = _pathStorage.getFreeVertex();
        // если места нет, возвращаемся
        if (end == 0)
        {
            return PFResult::mem_vertex;
        }
        const StationInfo* neighbour = tripinfo->stationInfoTo;
        end->setStation(neighbour, _pathStorage.getStationEst(neighbour->id1));
        end->setInfo(dt, at, tripinfo->transport);
        const Estimation& tiest = _pathStorage.getTripInfoEst(tripinfo);
        if (tiest.empty())
        {
            return PFResult::bad_path;
        }
        end->setTripInfo(tripinfo, tiest);
        if (end->getDuration() < 0)
        {
            return PFResult::bad_path;
        }
        // добавляем путевой сегмент к пути
        Path path = currentPath.copy();
        path.add(end);

        // если это последняя станция маршрута или на второй станции поезд останавливается, то можно вылезать
        if (tripinfo->stopOnLastStation())
            if (neighbour == condition._toStation
                || (condition._toStationIsTown && tripinfo->townGoodForFinish(condition._toStation, TeleportLink::START)))
            {
                if (startVertex == 0)
                    startVertex = path.getLastVertex();

                int price = (condition._usePrice) ? SG.getPrice(startVertex->getTripInfo(), tripinfo->stationInfoTo, startVertex->getBeginStationTime()) : MAX_PRICE;
                path.getObtForChange().plusPrice(price);
                path.getLastVertex()->setSegmentPrice(price);

#ifdef DISCOMFORT_ON
        int nightTime = 0;
        if (startVertex->getEndStationInfo()->timeZone)
            nightTime = startVertex->getEndStationInfo()->timeZone->getNightTime(startVertex->getBeginStationTime(), path.getLastTime());
#ifdef DISCOMFORT_STR_ON
                std::string dStr;
                int d =  Transport::getDiscomfort(tripinfo->transport, path.getLastTime()-startVertex->getBeginStationTime(), nightTime, tripinfo->threadStart->isMain(), dStr);
                end->addDiscomfortChecked(d, dStr.c_str());
#else
                int d =  Transport::getDiscomfort(tripinfo->transport, path.getLastTime()-startVertex->getBeginStationTime(), nightTime, tripinfo->threadStart->isMain());
                end->addDiscomfortChecked(d);
#endif
#endif

                if (_pathStorage.addPathToResult(path, condition))
                {
                    return PFResult::result;
                }

            }

        if (doCheck && !_pathStorage.goodPath(path, condition.getChanges()))
            return PFResult::bad_path;

        currentPath = path;
        return PFResult::ok;
    }

    //================================================================================================

    PFResult::Type TheGraph::_growPathWithThread(Path &currentPath, const TripInfo *tripinfo, int departureTime,
                                                     const StationInfo *sameTown, int minDelayIn, const Condition &condition,
                                                     bool &flag) const
    {
        const int begTime = currentPath.getLastVertex()->getBeginStationTime();
        const int endTime = currentPath.getLastVertex()->getEndStationTime();
        int begTime1 = begTime + minDelayIn;
        if (departureTime < begTime1 || departureTime > endTime)
            return PFResult::ok;
        flag = true;

        Path path(currentPath);
        PFResult::Type res = _addEdge(path, tripinfo, departureTime, condition, 0);
        const Vertex* startVertex = path.getLastVertex();
        if (res == PFResult::ok
            && (!sameTown || condition._canChangeInAnyTown
                || (condition._canChangeInStartTown/*в стартовом городе разрешены пересадки*/
                    && SG.st.fromOneTown(condition._from[0], tripinfo->stationInfoTo))))
        {
            res = _addExit(path, condition, startVertex);
            if (res != PFResult::ok)
                return res;                                
        }

        while (res == PFResult::ok /*&& sameTownTemp*/ && path.getLastTripInfo()->nextTI)
        {
            res = _prolongEdge(path, condition, startVertex);
            const TripInfo* ti = path.getLastTripInfo();
            const StationInfo* sameTownTemp = SG.st.fromOneTown(ti->stationInfoFrom, ti->stationInfoTo);
//            if (estChanges > 1)
            if (res == PFResult::ok)
            if (!sameTownTemp  || condition._canChangeInAnyTown
                //логика для гортранса
                || (condition._canChangeInStartTown/*в стартовом городе разрешены пересадки*/ 
                && SG.st.fromOneTown(condition._from[0], ti->stationInfoTo)/*текущий город - стартовый*/)
                )
            {
                res = _addExit(path, condition, startVertex);
                if (res != PFResult::ok)
                    return res;                                
            }
            if (!sameTownTemp)
            {
                const TripInfo* ti = path.getLastTripInfo();
                const StationInfo* neighbour = ti->stationInfoTo;
                if (path.getLastVertex()->getPrevious()->containAnyStationExceptThread(neighbour->costationsSet, ti->threadStart))
                    return PFResult::ok;
            }
        }
        if (res == PFResult::result || res == PFResult::bad_path || res == PFResult::ok)
            return PFResult::ok;
        return res;
    }

    //================================================================================================

    PFResult::Type TheGraph::_addEdgesFromStationFull(Path &currentPath, const Condition &condition) const
    {
        const StationInfo* currentStation = currentPath.getLastVertex()->getEndStationInfo();
        const Neighbours& neighbours = SG._graph[currentStation->id1].sais;
    
        StationCalcInfo& stationCalcInfo = _pathStorage._station2Info[currentStation->id1];
        GrowableList growableList;
        stationCalcInfo.bestPaths.getGrowableList(growableList);
        GrowableList::iterator itEndGL = growableList.end(); 

#ifdef DEBUG_ROUTE
        if (_pathStorage._needDebugLog) {
            std::fstream out(DEBUG_LOG_FILE, std::ios::out | std::ios::app);
            out << "FULL from " << currentStation->name << " growables=" << growableList.size()
                << " obt=" << growableList[0].vertex->getObt()
                << " est=" << growableList[0].vertex->getEst() << std::endl;
        }
        size_t oldFront = _pathStorage._pathFront.size();
#endif

        // время для пересадки 
        const int begTime = stationCalcInfo.getMinBeginStationTime(growableList);
        const int endTime = stationCalcInfo.getMaxEndStationTime(growableList);
        // то же самое в днях
        const int begDay = begTime / MINUTES_IN_DAY;
        const int endDay  = endTime / MINUTES_IN_DAY;

        // начинаем перебирать соседей
        Neighbours::const_iterator itEnd = neighbours.end();
        for (Neighbours::const_iterator it = neighbours.begin(); it != itEnd; ++it)
        {
            // зафиксировали соседа
            const StationAndInfo& sai(*it);
            const StationInfo* neighbour = sai.stationTo;

            //если уже следующим ребром мы сменим город, то можно проверить, чтоб не повториться
            const StationInfo* sameTown = SG.st.fromOneTown(currentStation, neighbour);

            // перебираем пути сообщений от froms[m] до neighbour,
            // выбираем наиболее подходящий
            const size_t tsz(sai.info.size());

            // перебираем сообщения
            for (size_t j = 0; j < tsz; ++j)
                    if (sai.info[j])
            {
                const TripInfo* tripinfo = sai.info[j];
                // пропускаем лишний транспорт
                if ((condition.getTransports() & tripinfo->transport) == Transport::NONE)
                    continue;

                if (!tripinfo->stopOnFirstStation())//isGoodForStart)
                    continue;
                const TripInfoCalcInfo& tici = _pathStorage._edge2Info[tripinfo->id];
                if (tici.est.empty())
                    continue;

                int goodGrowable = 0;
                for (GrowableList::iterator it = growableList.begin(); it != itEndGL; ++it)
                {
                    if (sameTown == 0)
                    {
                        // если путь не содержит этого города в этой нитке
                        // и содержит уже такой город в другой нитке, то прерываемся
                        if (it->vertex->containStations(neighbour->costationsSet))
                            it->badTI = tripinfo;
                        else
                            goodGrowable++;
                    }

                    if (it->vertex->getTeleportationTown() && !tripinfo->townGoodForStart(it->vertex->getTeleportationTown(), TeleportLink::ALL))
                        it->badTI = tripinfo;
                    else
                        goodGrowable++;
                }
                if (goodGrowable == 0)
                    continue;

                int minDelayIn = Transport::getMinDelay(condition._minDelayIn, tripinfo->transport);

                // выбираем время отправления и запоминаем время прибытия
                // перебираем дни
                for (int k = begDay; k <= endDay; ++k)
                {
                    // проверяем, что есть сообщение в этот день
                    if (tripinfo->departures.isGoodDay(k) == false)
                        continue;

                    // проверяем, что время нам подходит
                    // для этого определяем время отправления
                    const int departureTime = tripinfo->getDepartureTime(k);

//пытаемся продолжить каждый из путей
                    for (GrowableList::iterator it = growableList.begin(); it != itEndGL; ++it)
                        if (it->badTI != tripinfo)
                        {
                            bool grew = false;
                            Path path(it->vertex);
                            PFResult::Type res = _growPathWithThread(path, tripinfo, departureTime, sameTown,
                                                                     minDelayIn, condition, grew);
                            if (res != PFResult::ok)
                                return res;                                
                            if (grew)
                                it->badTI = tripinfo;
                        }
                }
            }
        }
        for (GrowableList::iterator it = growableList.begin(); it != itEndGL; ++it)
            it->vertex->decreaseAllReferences();
#ifdef DEBUG_ROUTE
        if (_pathStorage._needDebugLog) {
            std::fstream out(DEBUG_LOG_FILE, std::ios::out | std::ios::app);
            out << " frontInc=" << (_pathStorage._pathFront.size()-oldFront) << std::endl;
        }
#endif

    return PFResult::ok;
    }
    
    //================================================================================================
    /// попытка продолжить путь currentPath
    PFResult::Type TheGraph::_searchMainStep(Path &currentPath, const Condition &condition) const
    {
        if (!currentPath.growable())
            return PFResult::ok;
        bool good = _pathStorage.goodPath(currentPath, condition.getChanges());
        if (!good)
            return PFResult::ok;
        
#ifdef DEBUG_ROUTE
        if (_pathStorage._needDebugLog)
        if (currentPath.getLastVertex()->isDebugingThread(_pathStorage._debugStations, _pathStorage._debugRoutes))
        {
            std::fstream out(DEBUG_LOG_FILE, std::ios::out | std::ios::app);
            out << "MAIN STEP from st=" << currentStation->name << " path=";
            currentPath.toString(out);
            out << std::endl;
        }
#endif

        //последнее ребро ездовое
        //продолжаем вдоль этой нитки и еще готовим станции для пересадок-телепортаций
        if (currentPath.getLastTripInfo()) 
        {
            const TripInfo* nextTI = currentPath.getLastTripInfo()->nextTI;
            if (nextTI)
            {
                // если путь содержит уже такую станцию, то прерываемся
                const StationInfo* neighbour = nextTI->stationInfoTo;
                if (currentPath.containAnyStationExceptThread(neighbour->costationsSet, nextTI->threadStart))
                         return PFResult::ok;

                Path path = currentPath.copy();
                const Vertex* startVertex = path.getStartVertex();
                PFResult::Type res = _prolongEdge(path, condition, startVertex);
                if (res == PFResult::result || res == PFResult::bad_path)
                    res = PFResult::ok;
                else
                {
                    if (res != PFResult::ok)
                        return res;
                    //путь с концом в этой станции добавляем если есть еще куда ехать на этом транспорте
                    const TripInfo* ti = path.getLastTripInfo();
                    if (ti->nextTI)
                        if (!_pathStorage.addPathToFront(path))
                        {
                            // если места нет, возвращаемся
                            return PFResult::mem_path;
                        }
                    //А выходим в любом случае
                    res = _addExit(path, condition, startVertex);
                    if (res != PFResult::ok)
                        return res;
                }
            }
        }
        //пришла станция после пересадки или телепортации. продолжаем короткими ребрами до соседей
        else if (SG.st.isTown( currentPath.getLastVertex()->getEndStationInfo() ))
        {
            _addTeleportationFromTownCenterFull(currentPath, condition);
        }
        else
        {
            _addEdgesFromStationFull(currentPath, condition);
        }
        return PFResult::ok;
    }
    
    std::string TheGraph::resultToXML(const std::string& result) const
    {
        std::ostringstream ost;
        ost << "<routes result=\"" << result
            << "\" bestTime=\"" << getBestTime()
            << "\" bestPrice=\"" << getBestPrice()
            << "\" bestDiscomfort=\"" << getBestDiscomfort()
            << "\" bestChanges=\"" << getBestChanges()
            << "\" bestChangesNumber=\"" << getBestChangesNumber()
            << "\" routesNumber=\"" << getResultNumber()
            << "\" pricedRoutesNumber=\"" << getResultPricedNumber()
            << "\">\n";
        ost << "\t<group>\n";
        for (size_t j = 0; j < _pathStorage._result.size(); ++j)
        {
            _pathStorage._result._paths[j].toXML(ost, "\t\t\t", SG);
        }
        ost << "\t</group>\n";
        ost << "</routes>\n";
        return ost.str();
    }
    std::string TheGraph::resultToDebugString() const
    {
        std::string answer;
        for (size_t j = 0; j < _pathStorage._result.size(); ++j)
        {
            answer += _pathStorage._result._paths[j].toDebugString("\t\t\t", SG);
        }
        answer += "\n";
        return answer;
    }
    void TheGraph::stationResultToDebugString(std::ostream& out, const StationInfo* si) const
    {
        std::string answer;
        out << " vertexesNum=" << _pathStorage._station2Info[si->id1].bestPaths.getVertexNumber();
        out << " est=" << _pathStorage._station2Info[si->id1].est;
        out << " estAndGo=" << _pathStorage._station2Info[si->id1].estAndGo;
        out << " front=" << _pathStorage._station2Info[si->id1].front << std::endl;
        for (size_t j = 0; j < _pathStorage._station2Info[si->id1].bestPaths.getVertexNumber(); ++j)
        {
            out << j << " ";
            _pathStorage._station2Info[si->id1].bestPaths.getVertex(j)->toDebugString(out, "\t\t\t", SG);
        }
    }

}

