#ifndef THEGRAPH_H
#define THEGRAPH_H

#include "condition.h"
#include "result.h"
#include "estimation.h"
#include "stationgraph.h"
#include "pathstorage.h"

namespace Pathfinder
{
    // здесь происходит поиск
    class TheGraph
    {
    private:
        // хранилище взвешанных путей
        mutable PathStorage _pathStorage;

    public:
        // граф сообщений
        StationGraph SG;

        //=---------------- Methods
        // вернуть _allPathNumber
        int getAllPathNumber() const { return _pathStorage._allPathNumber; }
        // вернуть _lastUseful
        int getLastUseful() const { return _pathStorage._lastUseful; }
        // вернуть _maxPathNumber
        int getMaxPathNumber() const { return _pathStorage._maxPathNumber; }
        
        // сохраняем в текстовый файл
        void save(const char * filePath) const;
        // очищаем
        void erase();
        void pathStorageClear() {_pathStorage.clear();}
        void pathStorageStartWork(const std::string& mode, const int timeLimit)
        {
            _pathStorage.startWork(mode, timeLimit);
        }
        void pathStorageSetStationNumber() {_pathStorage.setStationNumber(SG.getStationNumber(), SG._tripInfos.size(), SG.st.stationInfos, SG._tripInfos);}
        
#ifdef DEBUG_ROUTE
        void setDebugRoute(const std::vector<int>& sis, const std::vector<std::string>& routes) const
        {
            if (!sis.empty() || !routes.empty())
                _pathStorage._needDebugLog = true;
            std::fstream out(DEBUG_LOG_FILE, std::ios::out | std::ios::app);
            out << "debugStations: " << sis.size() << std::endl;
            for (std::vector<int>::const_iterator sit = sis.begin(); sit != sis.end(); ++sit)
            {
                const StationInfo* si = SG.st.getStationByOrig(*sit);
                if (si)
                {
                    _pathStorage._debugStations.push_back(si);
                    si->debugPrint(out);
                    out << std::endl;
                }
                else
                {
                    out << "unkonwn station " << (*sit) << std::endl;
                }
            }
            out << "debugRoutes:" << std::endl;
            for (std::vector<std::string>::const_iterator rit = routes.begin(); rit != routes.end(); ++rit)
            {
                std::map<std::string, int>::const_iterator tit = SG._recodeThread.imapback.find(*rit);
                if (tit != SG._recodeThread.imapback.end())
                {
                    const Route* r = SG._threadStart[tit->second].route;
                    if (r)
                    {
                        _pathStorage._debugRoutes.push_back(r);
                        out << r->name;
                        out << " threads=(";
                        for (std::vector<const ThreadStart*>::const_iterator tit = r->threads.begin(); tit != r->threads.end(); ++tit)
                        {
                            out << (*tit)->name;
                            if (!(*tit)->loadedCorrect)
                                out << " incorrectly loaded";
                            out << " | ";
                        }
                        out << ")";
                        out << std::endl;
                    }
                    else
                    {
                        out << "unkonwn route " << (*rit) << std::endl;
                    }
                }
                else
                {
                    out << "unkonwn route " << (*rit) << std::endl;

                }
            }
        }
#endif


        void loadTariffs(const std::string& LOG_FILE, const char* aeroFile, const char* threadFile, const char* currencyRatesFile, std::istream& logFin)
        {
            _pathStorage._tariffStorage.LoadAeroexTariff(LOG_FILE, aeroFile, SG);
            _pathStorage._tariffStorage.LoadThreadTariff(LOG_FILE, threadFile, SG);
//            _pathStorage._tariffStorage.SaveTariffs("log.txt", "tariff1.out", SG);

            DumpSeatPriceLogParser dumpSeatPriceLogParser;
            dumpSeatPriceLogParser.LoadCurrencyRates(currencyRatesFile);
            const size_t tariffLogRes = dumpSeatPriceLogParser.LoadLog(LOG_FILE, logFin, SG);
            message(LOG_FILE, "dump prices: ", tariffLogRes);
            const size_t tariffLogOutRes = dumpSeatPriceLogParser.SaveLog(LOG_FILE, "tariff.out", SG);
            message(LOG_FILE, "dump prices saved: ", tariffLogOutRes);
            _pathStorage._tariffStorage.LogTariff2TariffStorage(dumpSeatPriceLogParser.tariffLogMap, SG);
//            _pathStorage._tariffStorage.SaveTariffs("log.txt", "tariff2.out", SG);

            int numPrices = _pathStorage._tariffStorage.FillBestPrice(SG);
            message(LOG_FILE, "best prices: ", numPrices);
            
            _pathStorage._tariffStorage.MinMaxTariff2TripInfo(SG);
//            _pathStorage._tariffStorage.SaveThreadTariffs("log.txt", "tariff3.out", SG);
        }

        bool tariffsEmpty() const {
            return  _pathStorage._tariffStorage.empty();
        }

        int getBestChanges() const
        {
            return _pathStorage._result.getBestChanges();
        }
        int getBestChangesNumber() const
        {
            return _pathStorage._result.getBestChangesNumber();
        }
        size_t resultNumber() const
        {
            return _pathStorage._result.size();
        }
        int getBestTime() const
        {
            return _pathStorage._result.getBestTime();
        }
        float getBestPrice() const
        {
            return _pathStorage._result.getBestPrice();
        }
        int getBestDiscomfort() const
        {
            return _pathStorage._result.getBestDiscomfort();
        }
        void searchDebugOutput(FILE* out)
        {
            _pathStorage.searchDebugOutput(out);
        }
        
        // поиск
        std::string search(const std::string &LOG_FILE, Condition &condition) const;
        std::string search1(const std::string &LOG_FILE, Condition &condition) const;
        std::string search2(const std::string &LOG_FILE, const Condition &condition) const;
        std::string search3(const std::string &LOG_FILE, const Condition &condition) const;

        // результаты превращаем в XML
        // в recode передается перекодировка в другие id-ники
        std::string resultToXML(const std::string& result) const;
        std::string resultToDebugString() const;
        void stationResultToDebugString(std::ostream& out, const StationInfo* si) const;

        size_t getResultNumber() const
        {
            return _pathStorage._result.size();
        }
        size_t getResultPricedNumber() const
        {
            return _pathStorage._result.getPricedNumber();
        }
        
        const Result& getResult()
        {
            return _pathStorage._result;
        }

    private:
        // поиск, первая итерация
        // возвращаем сообщение об ошибке
        PFResult::Type _searchInit(Condition &condition) const;
        PFResult::Type _firstStep(const Condition &condition) const;
        
        // поиск маршрутов (последующие итерации)
        // функция _searchMain вызывается после _searchInit
        // вход: условия поиск (condition), дополнительные параметры (optimization),
        //       в result хаписываются результаты работы поиска (там уже может что-то быть от _searchInit),
        // выход: результаты хранятся в result, кроме того возвращается строка, описывающая причину завершения поиска:
        PFResult::Type _searchMain(const Condition &condition) const;

        /// попытка продолжить путь currentPath
        PFResult::Type _searchMainStep(Path &currentPath, const Condition &condition) const;

        PFResult::Type _addEdge(Path &currentPath, const TripInfo *tripinfo, int day, const Condition &condition,
                                        const Vertex *startVertex, const bool doCheck = true) const;
        PFResult::Type _addEdge1(Path &currentPath, const TripInfo *tripinfo, int departureAfter,
                                         const Condition &condition, const Vertex *startVertex, const bool doCheck,
                                         const int dt) const;
        PFResult::Type _addTeleportations(Path &currentPath, const Condition &condition) const;
        PFResult::Type _addTeleportation2TownCenter(Path &currentPath, const Condition &, const Vertex *startVertex) const;
        PFResult::Type _addTeleportationFromTownCenter(Path &currentPath, const Condition &) const;
        PFResult::Type _addTeleportationFromTownCenterFull(Path &currentPath, const Condition &condition) const;
        PFResult::Type _prolongEdge(Path &currentPath, const Condition &condition, const Vertex *startVertex,
                                            const bool doCheck = true) const;
        PFResult::Type _addExit(Path &currentPath, const Condition &condition, const Vertex *startVertex) const;
        PFResult::Type _addEdgesFromStationFull(Path &currentPath, const Condition &condition) const;
        PFResult::Type _growPathWithThread(Path &currentPath, const TripInfo *tripinfo, int departureTime,
                                                   const StationInfo *sameTown, int minDelayIn, const Condition &condition,
                                                   bool &flag) const;

    };
}
#endif

