#include "thread.h"
#include "stationgraph.h"

namespace Pathfinder
{
    const StationInfo* Route::town2station(const StationInfo* town) const
    {
        if (!town->isTown())
            return town;
        Town2Stations::const_iterator it = town2stations.find(town);
        if (it == town2stations.end())
            return 0;
        if (it->second.size() == 1)
            return it->second[0];
        return 0;
    }

    const std::string& ThreadStart::getName() const
    {
        return startTI->getName();
    }

    void ThreadStart::updateInfo()
    {
        for (const TripInfo* ti = startTI; ti; ti = ti->nextTI)
        {
            moveTime += ti->duration;
            stopTime += ti->stopTimeBefore;
            len += (ti->stationInfoFrom->location - ti->stationInfoTo->location).Len();
            tiNumber++;
            transports = (Transport::Type)(transports | ti->transport);
        }
    }

    void ThreadStart::interpolateStationsWithOutLocation(const StationLocTimes& res, StationLocLens& stloc)
    {
        double time = 0;
        int curr = 0;
        Geom::Point2d loc = startTI->stationInfoFrom->location;
        if (!loc.IsZero())
        {
            curr++;
        }
        for (const TripInfo* ti = startTI; ti; ti = ti->nextTI)
        {
            time += ti->duration;
            loc = ti->stationInfoTo->location;
            if (!loc.IsZero())
            {
                time = 0;
                curr++;
            }
            else
            {
                size_t n = res.size();
                if (curr > 0 && curr < n)
                {
                    double timeBig = res[curr].time;
                    Geom::Point2d a = res[curr-1].stationInfo->location;
                    Geom::Point2d b = res[curr].stationInfo->location;
                    double alpha = time/timeBig;
                    Geom::Point2d c = (1-alpha)*a + alpha*b;
                    double len = (a-b).Len();
                    if (stloc[ti->stationInfoTo->id1].minlen==0.0 || 1.5*stloc[ti->stationInfoTo->id1].minlen > len)
                    {
                        if (stloc[ti->stationInfoTo->id1].minlen < 1.5*len)
                        {
                            stloc[ti->stationInfoTo->id1].loc += c;
                            stloc[ti->stationInfoTo->id1].n++;
                            if (stloc[ti->stationInfoTo->id1].minlen == 0)
                                stloc[ti->stationInfoTo->id1].minlen = len;
                            else
                                stloc[ti->stationInfoTo->id1].minlen = std::min(stloc[ti->stationInfoTo->id1].minlen, len);
                        }
                        else
                        {
                            stloc[ti->stationInfoTo->id1].loc = c;
                            stloc[ti->stationInfoTo->id1].n = 1;
                            stloc[ti->stationInfoTo->id1].minlen = len;
                            stloc[ti->stationInfoTo->id1].curves.clear();
                        }
                        stloc[ti->stationInfoTo->id1].addCurve(res[curr-1].stationInfo->curve);
                        stloc[ti->stationInfoTo->id1].addCurve(res[curr].stationInfo->curve);
                    }
                }
            }
        }
    }


    size_t ThreadStart::fillShortestPaths(Graph& g)
    {
        size_t res = 0;
        for (const TripInfo* ti1 = startTI; ti1; ti1 = ti1->nextTI)
            if (ti1->isHelper())
            {
                int n = 2;
                double dist = 0;
                Geom::Point2d locFrom = ti1->stationInfoFrom->location;
                Geom::Point2d locTo = ti1->stationInfoTo->location;
                if (dist >= 0.0 && !locFrom.IsZero() && !locTo.IsZero())
                    dist += (locFrom-locTo).Len();
                else
                    continue;
                double predRast = 0;
                for (const TripInfo* ti2 = ti1->nextTI; ti2 && dist > 0.0; ti2 = ti2->nextTI, n++)
                {
                    locFrom = ti2->stationInfoFrom->location;
                    locTo = ti2->stationInfoTo->location;
                    if (locFrom.IsZero() || locTo.IsZero())
                    {
                        dist = -1.0;
                        continue;
                    }
                    double edgeLen = (locFrom-locTo).Len();
                    dist += edgeLen;
                    double rast = (ti1->stationInfoFrom->location - locTo).Len();
                    if (!ti1->isHelper() && rast < predRast)
                    {
                        dist = -1.0;
                        continue;
                    }
                    predRast = rast;

                    Neighbours::iterator itn = g[ti1->stationInfoFrom->id1].sais.begin();
                    Neighbours::iterator itEnd = g[ti1->stationInfoFrom->id1].sais.end();
                    for (; (itn != itEnd) && itn->stationTo != ti2->stationInfoTo; ++itn);

                    //   -
                    if (itn != itEnd)
                    {
                        if (itn->nShortEdgeInside < n )
                        if (ti1->isHelper() || (itn->nShortEdgeInside < n && itn->dist*1.8 > dist && itn->dist > edgeLen))
                        {
                            itn->nShortEdgeInside = n;
                            itn->detailedTI = ti1;
                            res++;
                        }
                    }
                }
            }
        return res;
    }

    void ThreadStart::getStationsWithLocation(StationLocTimes& res)
    {
        len = 0;
        res.clear();
        double time = 0;
        Geom::Point2d loc = startTI->stationInfoFrom->location;
        if (!loc.IsZero())
        {
            res.push_back(StationLocTime());
            res.back().stationInfo = startTI->stationInfoFrom;
        }
        for (const TripInfo* ti = startTI; ti; ti = ti->nextTI)
        {
            time += ti->duration;

            loc = ti->stationInfoTo->location;
            if (!loc.IsZero())
            {
                res.push_back(StationLocTime());
                res.back().stationInfo = ti->stationInfoTo;
                res.back().time = time;
                time = 0;
                size_t n = res.size();
                if (n >= 2)
                    len += (res[n-1].stationInfo->location - res[n-2].stationInfo->location).Len();
            }
        }
    }

    size_t ThreadStart::calcPathPointNumber()
    {
        size_t res = 0;
        for (const TripInfo* ti1 = startTI; ti1; ti1 = ti1->nextTI)
        {
            size_t n = startTI->sai->curve.points.size();
            if (n > 0)
                res += n;
            else
                res++;
        }
        return res;
    }

    void ThreadStart::saveAsPolish_peace(std::ostream& out, const char* color)
    {
        for (const TripInfo* ti1 = startTI; ti1; ti1 = ti1->nextTI)
        {
            if (!ti1->sai->curve.points.empty())
            {
                ti1->sai->curve.savePolish(out, color, getName().c_str());
            }
            else if (ti1->stationInfoFrom->location != Geom::Point2d(0) && ti1->stationInfoTo->location != Geom::Point2d(0))
            {
                out << "[POLYLINE]\nType=" << color << "\nData0=("
                << ti1->stationInfoFrom->location.x << "," << ti1->stationInfoFrom->location.y
                << "),("
                << ti1->stationInfoTo->location.x << "," << ti1->stationInfoTo->location.y
                << ")\n"
                << "Label=" << getName()
                << "\n[END]\n" << std::endl;
            }
        }
    }

    void ThreadStart::saveAsPolish(std::ostream& log, std::ostream& out, const char* color)
    {
        size_t res = 0;
        out << "[POLYLINE]\nType=" << color << "\nData0=";
        for (const TripInfo* ti1 = startTI; ti1; ti1 = ti1->nextTI)
        {
            if (res)
                out << ",";
            const Geom::Curve* curve = &ti1->sai->curve;
            std::map<std::string, Geom::Curve>::const_iterator find = ti1->sai->routeCurves.find(ti1->getRoute());
            if (find != ti1->sai->routeCurves.end())
                curve = &find->second;
            if (!curve->points.empty())
            {
                res += curve->savePolishPointsList(out);
            }
            else if (ti1->stationInfoFrom->location != Geom::Point2d(0) && ti1->stationInfoTo->location != Geom::Point2d(0))
            {
                out << "(" << ti1->stationInfoFrom->location.x << "," << ti1->stationInfoFrom->location.y
                << "),("
                << ti1->stationInfoTo->location.x << "," << ti1->stationInfoTo->location.y << ")";
                if (ti1->stationInfoFrom->curve && ti1->stationInfoTo->curve)
                    log << "!!!";
                log << "jump\t" << ti1->getName() << "\t" << ti1->stationInfoFrom->name << "\t"  << ti1->stationInfoTo->name << "\t"  << (ti1->stationInfoFrom->location - ti1->stationInfoTo->location).Len() << std::endl;
                res += 2;
            }
        }
        out << "\n"
        << "Label=" << getName()
        << "\n[END]\n" << std::endl;
    }

}

