#ifndef TRANSPORT_H1
#define TRANSPORT_H1

#include "pf.h"

namespace Pathfinder
{
    namespace Transport
    {
        enum Type
        {
            NONE                    = 0x00,
            TRAIN                   = 0x01,
            PLANE                   = 0x02,
            BUS                     = 0x04,
            RIVER                   = 0x08,
            SEA                     = 0x10,
            LOCAL_TRAIN             = 0x20,
            WATER                   = 0x400,
            TRAIN_OR_LOCAL_TRAIN    = 0x21, //33
            BUS_OR_LOCAL_TRAIN      = 0x24,
            ALL                     = 0x43F
        };

        inline Type fromInt(const int i)
        {
            if (i==33) return TRAIN_OR_LOCAL_TRAIN;
            return (Type)(0x01 << (i-1));
        }

        inline int toInt(const int tr)
        {
            switch(tr) {
                case TRAIN: return 1;
                case PLANE: return 2;
                case BUS: return 3;
                case RIVER: return 4;
                case SEA: return 5;
                case LOCAL_TRAIN: return 6;
                case WATER: return 11;
                case TRAIN_OR_LOCAL_TRAIN: return 1;
                case NONE: return 0;
                case ALL:;
            }
            return 0;
        }

        inline int getTransportsNumber()
        {
            return 12;
        }
        inline int getWeight(const Transport::Type& tr)
        {
            int res = 0;
            if (tr & PLANE)
                res += 100;
            if (tr & TRAIN)
                res += 10;
            if (tr & BUS)
                res += 1;
            if (tr & LOCAL_TRAIN)
                res += 1;
            return res;
        }
        inline std::string toString(const Transport::Type& tr)
        {
            switch(tr) {
                case TRAIN: return "train";
                case PLANE: return "plane";
                case BUS: return "bus";
                case RIVER: return "river";
                case SEA: return "sea";
                case LOCAL_TRAIN: return "local_train";
                case WATER: return "water";
                case NONE: return "none";
                case ALL: return "all";
                default: return "";
            }
            return "";
        }
        inline bool betterOrEqual(const Transport::Type& tr1, const Transport::Type& tr2)
        {
            int w1 = getWeight(tr1);
            int w2 = getWeight(tr2);
            if (w1 < w2)
                return true;
            if (w1 > w2)
                return false;
            return (tr1 == tr2);
        }

        inline int getMinDelay(const std::vector<int>& optDelay, const Type& tr)
        {
            return optDelay[toInt(tr)];
        }

        /**
         * Возвращает массив времен для пересадки для всех видов транспорта
         * @param t может принимать значения: 0 - время на выход из транспорта, 1 - время на посадку, 2 - время на посадку после телепортации
         * @return время на пересадку в различных случаях
         */
        inline void setDefaultMinDelay(std::vector<int>& times, int t)
        {
            times.clear();
            times.resize(getTransportsNumber(), 20);
            if (t == 0) {
                times[toInt(Transport::PLANE)]       = 30;
                times[toInt(Transport::TRAIN)]       = 10;
                times[toInt(Transport::BUS)]         = 5;
                times[toInt(Transport::LOCAL_TRAIN)] = 6;
            } else if (t == 1) {
                times[toInt(Transport::PLANE)]       = 120;
                times[toInt(Transport::LOCAL_TRAIN)] = 0;
            } else {
                times.clear();
                times.resize(getTransportsNumber(), 70);
                times[toInt(Transport::PLANE)] = 160;
            }
        }

        inline void setDefaultMinDelay_i_love_sport(std::vector<int>& times)
        {
            times.clear();
            times.resize(getTransportsNumber(), 0);
        }

        inline int getDiscomfort(const Transport::Type& tr, int duration, int nightDuration, int isMain
            , std::string& dStr
            )
        {
#ifndef DISCOMFORT_ON
            return 0;
#endif
            if (duration < 0 || nightDuration < 0)
                return 0;
            int res = 0;
            res += std::max(0, duration-U_MOVE);
            if (duration-U_MOVE > 0)
                dStr += " move>48h";
            if (!(tr & Transport::TRAIN)) {
                res += std::max(0, duration-U_MOVE_NO_SLEEP);
                if (duration-U_MOVE_NO_SLEEP > 0)
                    dStr += " seat>6h";
                res += std::max(0, nightDuration-U_MOVE_NO_SLEEP_NIGHT);
                if (nightDuration-U_MOVE_NO_SLEEP_NIGHT > 0)
                    dStr += " nightMove>2h";
            }
            if (!isMain) {
                res += U_MOVE_THRO_CAR;
                dStr += " thro_car";
            }
            if (tr & (Transport::PLANE|Transport::TRAIN)) {
                res += U_MOVE_TICKET_BOOK;
                dStr += " ticket";
            }
            return res;
        }

        inline int getDiscomfort(const Transport::Type& tr, int duration, int nightDuration, bool isMain)
        {
#ifndef DISCOMFORT_ON
            return 0;
#endif
            if (duration < 0)
                return 0;
            int res = 0;
            res += std::max(0, duration-U_MOVE);
            if (!(tr & Transport::TRAIN)) {
                res += std::max(0, duration-U_MOVE_NO_SLEEP);
                res += std::max(0, nightDuration-U_MOVE_NO_SLEEP_NIGHT);
            }
            if (!isMain)
                res += U_MOVE_THRO_CAR;
            if (tr & (Transport::PLANE|Transport::TRAIN))
                res += U_MOVE_TICKET_BOOK;

            return res;
        }

        inline int getStationDiscomfort(const Transport::Type& tr, int maj, int duration, int nightDuration
            )
        {
#ifndef DISCOMFORT_ON
            return 0;
#endif
            if (duration < 0)
                return 0;
            int res = 0;
            if (!(tr & Transport::PLANE)) {
                if (maj > 1)
                    res += std::max(0, duration-U_STATION_WARM);
                res += std::max(0, nightDuration-U_MOVE_NO_SLEEP_NIGHT);
            }
            return res;
        }

        inline int getStationDiscomfort(const Transport::Type& tr, int maj, int duration, int nightDuration
            , std::string& dStr
            )
        {
#ifndef DISCOMFORT_ON
            return 0;
#endif
            if (duration < 0)
                return 0;
            int res = 0;
            if (!(tr & Transport::PLANE)) {
                if (maj > 1) {
                    res += std::max(0, duration-U_STATION_WARM);
                    if (duration-U_STATION_WARM > 0)
                        dStr += " warmSt>20m";
                }
                res += std::max(0, nightDuration-U_MOVE_NO_SLEEP_NIGHT);
                if (nightDuration-U_MOVE_NO_SLEEP_NIGHT > 0)
                    dStr += " nightSt>2h";

//                if (!(tr & Transport::TRAIN))
//                    res += std::max(0, duration-U_STATION_EAT)
            }
            return res;
        }

    }
}

#endif

