#ifndef TRIPINFO_H
#define TRIPINFO_H

#include "stationtown.h"
#include "thread.h"
#include "tariff.h"

namespace Pathfinder
{
    struct StationAndInfo;

    // храним информацию об одном сообщении: отправления, время в пути, нитки, транспорт
    class TripInfo
    {
    public:
        //станции отправления и прибытия
        const StationInfo* stationInfoFrom;
        const StationInfo* stationInfoTo;
        const TripInfo* nextTI;
        const TripInfo* predTI;
        const ThreadStart* threadStart;
        const std::string& getName() const {return threadStart->name;}
        const std::string& getRoute() const {return threadStart->routeName;}
        // идентификатор нитки
        int thread;
        // отправления
        Departures departures;
        // время в пути (длительность)
        int duration;
        int stopTimeBefore;
        bool is_technical_stop;
        // тип транспорта
        Transport::Type transport;
        //просто номер записи в массиве
        size_t id;
        TeleportLinkPtrs townsGoodForIn;
        TeleportLinkPtrs townsGoodForOut;
        const StationAndInfo* sai;
        int durationStart2From;//время от отправления до прибытия в stationInfoFrom
        const Tariff* minTariff;
        const Tariff* maxTariff;
    public:
        // конструктор
        inline TripInfo():
                stationInfoFrom(0), stationInfoTo(0)
                , nextTI(0), predTI(0)
                , threadStart(0)
                , thread(-1)
                , duration(-1)
                , stopTimeBefore(0)
                , is_technical_stop(false)
                , transport(Transport::NONE)
                , id(0)
                , sai(0)
                , durationStart2From(0)
                , minTariff(0), maxTariff(0)
        {}

    public:
        bool isHelper() const
        {
            return threadStart->isHelper();
        }
        int durationStart2To() const
        {
            return durationStart2From + duration;
        }
        // все корректно?
        inline bool isValid() const {return departures.isValid() && (thread >= 0) && transport >= 0;}
        // останавливается ли в stationInfoFrom
        inline bool stopOnFirstStation() const {return true;}//(!is_technical_stop) && (stopTimeBefore>0 || predTI==0);}
        // останавливается ли в stationInfoTo
        inline bool stopOnLastStation() const {return true;}//(nextTI==0)||((!nextTI->is_technical_stop) && nextTI->stopTimeBefore>0);}
        // вернуть время отправления или -1 если в этот день хождения нет
        inline int getDepartureTime(const int day) const { return departures.getTime(day); }
        // вернуть время отправления после времени time (время прибытия на предыдущую станцию) или -1 
        inline int getDepartureTimeAfter(const int time) const
        {
            int day = time / MINUTES_IN_DAY;
            int dt = getDepartureTime(day);
            if (dt == -1 || time > dt)
            {
                day++;
                dt = getDepartureTime(day);
            }
            return dt;
        }
        inline int getDepartureTimeAfter2(const int departureAfter) const
        {
            return threadStart->getDepartureTimeAfter(departureAfter-durationStart2From)+durationStart2From;
        }
        //begTime и endTime должны быть в соседних сутках и не далее 36 часов между ними
        inline int getDepartureTimesBetween(const int begTime, const int endTime, std::vector<int>& res) const
        {
            res.clear();
            if (threadStart->interval_begin_time < 0)//обычный неинтервальный рейс
            {
                int begDay = begTime/MINUTES_IN_DAY;
                int endDay = endTime/MINUTES_IN_DAY;
                for (int k = begDay; k <= endDay; k++)
                {
                    // проверяем, что есть сообщение в этот день
                    if (departures.isGoodDay(k) == false)
                        continue;

                    // проверяем, что время нам подходит
                    // для этого определяем время отправления
                    int departureTime = k * MINUTES_IN_DAY + departures.getTime();
                    if (departureTime >= begTime && departureTime <= endTime)
                        res.push_back(departureTime);
                }
            }
            else
            {
                int startb = begTime - durationStart2From;
                int starte = endTime - durationStart2From;
                int begDay = startb/MINUTES_IN_DAY;
                int endDay = starte/MINUTES_IN_DAY;
                for (int k = begDay; k <= endDay; k++)
                {
                    if (threadStart->departures.isGoodDay(k) == false)
                        continue;
                    int beg = k * MINUTES_IN_DAY + threadStart->interval_begin_time;
                    int end = k * MINUTES_IN_DAY + threadStart->interval_end_time;
                    for (int departureTime = beg; departureTime <= end; departureTime += threadStart->interval_delta)
                        if (departureTime >= startb && departureTime <= starte)
                            res.push_back(departureTime + durationStart2From);
                }
            }
            return res.size();
        }
        inline int getArrivalTimeAfter(const int time) const
        {
            int day = time / MINUTES_IN_DAY;
            int dt = getArrivalTime(day);
            if (time > dt)
            {
                day++;
                dt = getArrivalTime(day);
            }
            return dt;
        }
        // вернуть время (в минутах, начиная с ORIGIN_YEAR) прибытия или -1 если в день day отправления нет
        inline int getArrivalTime(const int day) const
        {
            const int result(getDepartureTime(day));
            return (result < 0 ? result : result + duration);
        }

        bool townGoodForStart(const StationInfo* town, const TeleportLink::Type tlt = TeleportLink::START) const
        {
            TeleportLinkPtrs::const_iterator it = townsGoodForIn.begin();
            TeleportLinkPtrs::const_iterator itEnd = townsGoodForIn.end();
            for (; it != itEnd /* && !(((*it)->type&tlt) */ && (*it)->town != town; ++it);
            if (it != townsGoodForIn.end())
                return true;
            return false;
        }

        bool townGoodForFinish(const StationInfo* town, const TeleportLink::Type tlt = TeleportLink::START) const
        {
            TeleportLinkPtrs::const_iterator it = townsGoodForOut.begin();
            TeleportLinkPtrs::const_iterator itEnd = townsGoodForOut.end();
            for (; it != itEnd /*&& !(((*it)->type&tlt)*/ && (*it)->town != town; ++it);
            if (it != townsGoodForOut.end())
                return true;
            return false;
        }

        // сравнение по времени отправления
        inline bool operator<(const TripInfo & ti) const { return departures.time < ti.departures.time; }
    };

    // информация по всем сообщениях
    typedef std::vector<TripInfo*> Info;
    typedef std::vector<TripInfo> TripInfos;
    typedef std::vector<const TripInfo*> TripInfoPtrs;

}

#endif

