#if !defined(VERTEX_H)
#define VERTEX_H

#include "thread.h"
#include "transport.h"
#include "estimation.h"

namespace Pathfinder
{
    class Path;
    // сегмент пути
    // хранит информацию о станции и о том, как в эту станцию прибыли
    class Vertex
    {
    private:
        const TripInfo* _tripinfo;
        // конец сегмента - станция прибытия
        const StationInfo* _endStationInfo;
        // начальное время - время отправления
        int _beginStationTime;
        // конечное время - время прибытия
        int _endStationTime;
        // транспорт
        Transport::Type _transport;

        // указатель на предыдущий сегмент
        Vertex * _previous;
        // количество ссылок на это сегмент
        // т.е. количество раз, когда он является предыдущим
        // или в каком количестве путей он участвует?
        int _numberOfReferences;
        bool _growable;

        // флажок, которым можно помечать, неперспективные узлы. Узлы, которые не стоит продолжать.
        bool _makeLonger;
        bool _noTeleporations;

        // пройденный путь 
        Estimation _obt;
        // оценка снизу веса до станции назначения 
        Estimation _est;

        const StationInfo* _teleportationTown;
        int _segmentPrice;
    public:
        // конструктор по умолчанию
        Vertex()
           :
            _tripinfo(0),
            _endStationInfo(0),
            _beginStationTime(-1),
            _endStationTime(-1),
            _transport(Transport::NONE),
            _previous(0),
            _numberOfReferences(0),
            _growable(false),
            _makeLonger(true),
            _noTeleporations(false),
            _obt(0, 0, 0, Transport::NONE, 0),
            _teleportationTown(0),
            _segmentPrice(-1)
        {}

    public: 
        // название класса
        static std::string call() { return "Vertex"; }

        // метода доступа:
        inline const TripInfo* getTripInfo() const {return _tripinfo; }
        inline const ThreadStart* getThreadStart() const {return (_tripinfo)?_tripinfo->threadStart:0; }
        inline Station getEndStation() const { return _endStationInfo->id1; }
        inline const StationInfo* getEndStationInfo() const { return _endStationInfo; }
        inline void setEndStationInfo(const StationInfo* si) { _endStationInfo=si; }
        inline int getBeginStationTime() const { return _beginStationTime; }
        inline int getEndStationTime() const { return _endStationTime; }
        inline void setBeginStationTime(const int t) { _beginStationTime = t; }
        inline void setEndStationTime(const int t) { _endStationTime = t; }
        inline int getGrowable() const { return _growable; }
        inline Transport::Type getTransport() const { return _transport; }

        inline bool getNoTeleportations() const { return _noTeleporations; }
        inline void setNoTeleportations(const bool t) { _noTeleporations = t; }

        inline int getSegmentPrice() const { return _segmentPrice; }
        inline void setSegmentPrice(const int p) { _segmentPrice = p; }
        float getSegmentPriceForPrint() const {
            if (_segmentPrice < 0 || _segmentPrice >= MAX_PRICE)
                return -1;
            return _segmentPrice/100.0;
        }

        inline Vertex * getPrevious() const { return _previous; }
        inline int getNumberOfReferences() const { return _numberOfReferences; }

        inline const Estimation& getObt() const { return _obt; }
        inline Estimation& getObtForChange() { return _obt; }
        inline const Estimation& getEst() const { return _est; }
        void setZeroEst() 
        {
            _est.trPoint = 0;
            _est.weight = 0;
        }
        inline const int getNumberOfChanges() const { return _obt.changes; }

        inline bool getMakeLonger() const { return _makeLonger; }
        inline void setMakeLonger(const bool b) 
        {
            _makeLonger = b; 
        }

        inline const StationInfo* getTeleportationTown() const { return _teleportationTown; }
        inline void setTeleportationTown(const StationInfo* town) { _teleportationTown = town; }
        // время прохождения сегмента
        inline int getDuration() const { return _endStationTime - _beginStationTime; }
        // вернуть время в пути (вместе с остановками)
        int getTotalTime() const
        {
            // если путь пустой, возвращаем ноль
            const Vertex* it = this;
            if (it == 0)
                return 0;
            // пробегаем весь путь от конца к началу
            while (it != 0)
            {
                // добежали до начала
                if (it->_previous == 0)
                    return _endStationTime - it->_beginStationTime;
                it = it->_previous;
            }
            return -1;
        }

        inline Estimation getTotalEst() const 
        { 
            Estimation res(_obt);
            res.plusplus(_est);
            res.changes--;
            return res; 
        }

        inline int getTotalChanges() const 
        { 
            return _obt.changes+_est.changes-1;
        }

        // не инициализированный?
        inline bool empty() const
        {
            return _endStationInfo == 0 && _beginStationTime == -1 && _endStationTime == -1;
        }

        // корректность
        inline bool valid() const
        {
            // конечное время больше начального
            if (_endStationTime < _beginStationTime)
                return false;
            return true;
        }

    public:
        // инициализация
        inline void setStation(const StationInfo* si, const Estimation& stEst) 
        { 
            _endStationInfo = si; 
            _est = stEst;
        }
        inline void setTripInfo(const TripInfo* tripinfo, const Estimation& e)
        { 
            _est = e;
            _tripinfo = tripinfo; 
            _est.weight -= getDuration();
            _est.trPoint -= Transport::getWeight(tripinfo->transport);
        }
        inline void setInfo(const int & beginStationTime, const int & endStationTime, const Transport::Type& transport)
        { 
            assert(beginStationTime >= 0 && endStationTime >= 0);
            _beginStationTime = beginStationTime;
            _endStationTime = endStationTime;
            _transport = transport;
        }
        inline void setPriceUnchecked(const int price) 
        { 
            _obt.price = price;
        }
        inline void addPriceChecked(const int price) 
        { 
            _obt.plusPrice(price);
        }

        inline void setDiscomfortChecked(const int d
            , const char* dStr = 0
            ) 
        { 
            _obt.setDiscomfort(d, dStr);
        }
        inline void addDiscomfortChecked(const int d
            , const char* dStr = 0
            ) 
        { 
            _obt.plusDiscomfort(d, dStr);
        }

        inline void addBeginStationTime(const int& add)
        { 
            _beginStationTime += add;
        }
        inline Vertex* addVertex(Vertex* vertex)
        {
            vertex->_previous = this;
            return vertex;
        }
        void calcObt()
        {
            if (getPrevious() == 0)
            {
                _obt.changes = 1;
                return;
            }

            int plusw = _obt.weight;
            int plusp = _obt.price;
            int plusd = _obt.discomfort;

#ifdef DISCOMFORT_STR_ON
            std::string dStr = _obt.discomfortString;
#endif
            _obt = getPrevious()->getObt();
            _obt.plusPrice(plusp);
            _obt.plusDiscomfort(plusd);
#ifdef DISCOMFORT_STR_ON
            _obt.discomfortString += dStr;
#endif

            //первый бегтайм = 0, второй - времени начала движения, и дальше - копируется
            if (!getPrevious()->getPrevious())
                _obt.begTime = getBeginStationTime();

            if (getThreadStart() == 0) //приехали на телепортации
            {
                _obt.weight += plusw; 
                if (getPrevious()->getThreadStart())
                {
                    _obt.changes ++;
                    if (_endStationInfo != getPrevious()->getEndStationInfo())
                    {
                        _obt.trPoint++;//телепортация
                    }
                    if (_endStationInfo->majority == -1) //город
                        _teleportationTown = _endStationInfo;
                }
                else
                    if (getPrevious()->getEndStationInfo()->majority == -1)
                        _teleportationTown = getPrevious()->getEndStationInfo();
            }
            else //приехали на транспорте
            {
                // время в пути
                const int duration = getDuration();

                // время на остановку
                int stopTimeB = getBeginStationTime(), stopTimeE = getBeginStationTime();
                if (getPrevious()->getThreadStart())
                    stopTimeB = getPrevious()->getEndStationTime();
                else
                {
                    _obt.trPoint += Transport::getWeight(_transport);
                    if (getPrevious()->getPrevious())
                    {
                        stopTimeB = getPrevious()->getBeginStationTime();
                    }
                    if (!_tripinfo->threadStart->isMain())
                        _obt.trPoint++;
                    if (getPrevious()->getEndStationInfo()->majority > 1)
                        _obt.trPoint++;
                    _obt.hash += getPrevious()->getEndStationInfo()->id1;    
                    if (_tripinfo->predTI)
                        _obt.timeForBoarding += _tripinfo->stopTimeBefore;
                    else
                        _obt.timeForBoarding += 100;

#ifdef DISCOMFORT_ON
        int nightTime = 0;
        if (getPrevious()->getEndStationInfo()->timeZone)
            nightTime = getPrevious()->getEndStationInfo()->timeZone->getNightTime(stopTimeB, stopTimeE);
#ifdef DISCOMFORT_STR_ON
                    std::string dStr;
                    int d = Transport::getStationDiscomfort(getPrevious()->getEndStationInfo()->transports, getPrevious()->getEndStationInfo()->majority, stopTimeE-stopTimeB, nightTime, dStr);
                    setDiscomfortChecked(getPrevious()->getObt().discomfort + d, dStr.c_str());
#else
                    int d = Transport::getStationDiscomfort(getPrevious()->getEndStationInfo()->transports, getPrevious()->getEndStationInfo()->majority, stopTimeE-stopTimeB, nightTime);
                    setDiscomfortChecked(getPrevious()->getObt().discomfort + d);
#endif
#endif
                }

                // пересчитываем вес: остановка + движение
                _obt.weight += stopTimeE - stopTimeB;
                _obt.weight += duration;
            }
        }
        void incTeleportations()
        {
            _obt.trPoint++;
        }
        bool checkMakeLonger(const int changes) const
        {
            if (getEst().empty())    
                return false;
            if (getTotalChanges() > changes)
                return false;
            return true;
        }

        void increaseAllReferences()
        {
            _growable = true;
            for (Vertex* it = this; it != 0; it = it->_previous)
                it->_numberOfReferences++;
        }
        void decreaseAllReferences()
        {
            if (_growable)
                for (Vertex* it = this; it != 0; it = it->_previous)
                    it->_numberOfReferences--;
             _growable = false;
       }

    public:
        // проходит путь через станцию station?
        bool containStation(const StationInfo* si)
        {
            for (Vertex* it = this; it != 0; it = it->_previous)
                if (it->_endStationInfo == si)
                    return true;
            return false;
        }
        bool containStations(const CostationsSet& stations)
        {
            for (Vertex* it = this; it != 0; it = it->_previous)
                if (it->endStationInSet(stations))
                    return true;
            return false;
        }
        bool containThread(const std::string& threadName) const
        {
            for (const Vertex* it = this; it; it = it->_previous)
                if (it->_tripinfo && it->_tripinfo->threadStart->name==threadName)
                    return true;
            return false;
        }

#ifdef DEBUG_ROUTE
        bool isDebugingThread(const StationInfoPtrs& debugStations, const std::vector<const Route*>& debugRoutes) const
        {
            if (debugStations.empty() && debugRoutes.empty())
                return false;
            int iSt = debugStations.size()-1;
            int iR = debugRoutes.size()-1;
            for (const Vertex* it = this; it; it = it->_previous)
            {
                if (iSt >= 0 && it->getEndStationInfo() == debugStations[iSt])
                    iSt--;
                if (iR >= 0 && it->getThreadStart() && it->getThreadStart()->route == debugRoutes[iR])
                    iR--;
            }
//            std::fstream out("log.txt", std::ios::out | std::ios::app);
//            out << "LOC_TR num " << iSt << " " << iR << std::endl;
            if (iSt == -1 && iR == -1)
                return true;
            return false;
        }
#endif

        bool containAnyStationExceptThread(const CostationsSet& stations, const ThreadStart* thread)
        {
            Vertex* it;
            for (it = this; it != 0 && it->getThreadStart() == thread && !it->endStationInSet(stations); it = it->_previous) {
            }
            if (it == 0)
                return false; //случай невероятный
            if (it->getThreadStart() == thread)
                return false; //станция есть, но в текущей нитке
            if (it->endStationInSet(stations))
                return false; //станция есть, первая в текущей нитке
            for (; it != 0 && !it->endStationInSet(stations); it = it->_previous) {
            }
            if (it == 0)
                return false; //станции в пути не встречаются
            return true; //нашли станцию в пути
        }
        bool endStationInSet(const CostationsSet& stations)
        {
            if (stations.find(_endStationInfo) == stations.end())
                return false;
            return true;
        }

        //собирает все транспорты из пути
        Transport::Type allTransports()
        {
            Transport::Type res(Transport::NONE);    
            for (Vertex * it = this; it; it = it->_previous)
                res = (Transport::Type)(res | it->getTransport());
            return res;
        }

    public:    
        //сравнения
        inline static bool cmpByObt(const Vertex& v0, const Vertex& v1)
        { 
            return v0._obt < v1._obt;
        }
        inline static bool cmpByEst(const Vertex& v0, const Vertex& v1)
        { 
            Estimation e(v0._obt); e.plusplus(v0._est);
            Estimation ewp(v1._obt); ewp.plusplus(v1._est);
            if (e.changes == ewp.changes && ((v0.getThreadStart() != 0) != (v1.getThreadStart() != 0)))
            {    
                return (v0.getThreadStart() != 0) < (v1.getThreadStart() != 0);
            }
            return ewp < e;

        }

        // сейчас сравниваем так
        inline static bool cmp(const Vertex& v0, const Vertex& v1)
        {
            Estimation e(v0._obt); e.plusplus(v0._est);
            Estimation ewp(v1._obt); ewp.plusplus(v1._est);
            if (e.changes == ewp.changes && ((v0.getThreadStart() != 0) != (v1.getThreadStart() != 0)))
            {    
                return (v0.getThreadStart() != 0) < (v1.getThreadStart() != 0);
            }
            return ewp < e;
        }

        // метод сравнения, основанный на количестве пересадок
        inline static bool cmpByChanges(const Vertex& v0, const Vertex& v1)
        { 
            if (v0.getNumberOfChanges() != v1.getNumberOfChanges())
                return v0.getNumberOfChanges() < v1.getNumberOfChanges(); 
            return v0.getObt().weight < v1.getObt().weight; 
        }

        inline bool operator<(const Vertex& v) const
        {
            Estimation e(_obt); e.plusplus(_est);
            Estimation ewp(v._obt); ewp.plusplus(v._est);
            return v._est < _est;
        }

        //задает только частичный порядок
        //отношение "точно, без сомнения лучше"
        //но транзитивность должна быть обязательно!!! 
        inline bool betterByObt(const Vertex& v) const
        {
            if (v._obt < _obt)
                return false;
            return _obt.better(v._obt);
        }
        inline Estimation::BetterItem better4ByObt(const Vertex& v) const
        {
            return _obt.better4result(v._obt);
        }

        inline bool betterByTotalObt(const Vertex& v) const
        {
            Estimation e1(_obt); e1.plusplus(_est); e1.changes--;
            Estimation e2(v._obt); e2.plusplus(v._est); e2.changes--;
            if (e2 < e1)
                return false;
            return e1.better(e2);
        }
        inline Estimation::BetterItem better4ByTotalObt(const Vertex& v) const
        {
            Estimation e1(_obt); e1.plusplus(_est); e1.changes--;
            Estimation e2(v._obt); e2.plusplus(v._est); e2.changes--;
            return e1.better4result(e2);
        }

        // в строку
        std::string toString() const
        {
            std::ostringstream ost;
            ost << '(';
            ost << getThreadStart()->name << "th.";
            ost << _endStationInfo->id << "st." << _beginStationTime << "bst."<< _endStationTime << "est.";
            ost << _transport;
            ost  <<"tr." << _numberOfReferences << "rf)";
            return ost.str();
        }

        void toDebugString(std::ostream& ost, const std::string & indent, const StationGraph& SG) const 
        {
            // обратный порядок меняем на прямой
            const Vertex * it = this;
            std::vector<const Vertex *> reorder;
            while(it != 0)
            {
                reorder.push_back(it);
                it = it->getPrevious();
            }

            const int n = static_cast<int>(reorder.size());
            std::string sub;
            const std::string end = "\" ";

            double price = getObt().getPriceForPrint();
            int sTime = reorder.back()->getBeginStationTime();
            std::string sTimeName = Dates::toString(sTime);
            std::string sDate = sTimeName.substr(0, 10);
            ost << "\t\tprice=\"" << price << "\" time=\"" << getObt().weight << "\" tr=\"" << getObt().discomfort//trPoint 
                << "\" from=\"" << reorder.back()->getEndStationInfo()->name << "\" to=\"" << reorder[0]->getEndStationInfo()->name 
                << "\" date=\"" << sTime << "\"\n";

            // пробегаем по пути, определяем когда происходит пересадка и записываем в ost
            const ThreadStart* predThread = 0;
            std::string predStation = reorder[n-1]->getEndStationInfo()->name;
            for (int i = n-2; i >= 0; i--)
            {
                const Vertex* curr = reorder[i];
                const ThreadStart* thread = curr->getThreadStart();
                if (predThread != thread)
                {
                    std::string departureStation = reorder[i+1]->getEndStationInfo()->name;
                    int departureTime = curr->getBeginStationTime();
                    std::string departureTimeName("void");
                    std::string startDate("1900-01-01");
                    if (departureTime >= 0)
                    {
                        departureTimeName = Dates::toString(departureTime);
                        startDate = departureTimeName.substr(0, 10);
                    }

                    //пропускаем непересадочные станции
                    int j;
                    for (j = i-1; j >= 0 && reorder[j]->getThreadStart() == thread; j--) {
                    }
                    i = j+1;
                    curr = reorder[i];

                    ost << indent;
                    ost << "price=\"" << curr->getObt().getPriceForPrint() << end;
                    ost << "tr=\"" << curr->getObt().discomfort << end;
                    ost << "time=\"" << curr->getObt().weight << end;
                    ost << "thread=\"";
                    if (thread == 0)
                        ost << "NULL";
                    else
                        ost << SG.getThreadOrig(thread->id);
                    ost << end;
                    ost << "departure=\"" << departureStation << " " << departureTimeName << end;
            

                    std::string arrivalStation = curr->getEndStationInfo()->name;
                    predStation = arrivalStation;
                    int arrivalTime = curr->getEndStationTime();
                    std::string arrivalTimeName = (arrivalTime < 0 ? "void" : Dates::toString(arrivalTime));

                    ost << "arrival=\"" << arrivalStation << " " << arrivalTimeName << end;
                    ost << "\n";
                }
                predThread = thread;

            }
            ost << "\n";
        }

    };
    inline std::ostream& operator << (std::ostream& ost, const Vertex& v)
    {  
        ost << "[obt=" << v.getObt() << " lastST=" << v.getEndStationInfo()->name 
            << " timeB=" << Dates::toString(v.getBeginStationTime())
            << " timeE=" << Dates::toString(v.getEndStationTime()) << " ";
        const Vertex* it = &v;
        const Vertex* itPred = 0;
        std::string res;
        while (it != 0)
        {
            if (it->getTripInfo() && (!itPred || !itPred->getTripInfo())) //станция высадки
            {
                ost << it->getThreadStart()->name << "(" << it->getEndStationInfo()->name << "<-";
            }
            if (it->getTripInfo() && (!it->getPrevious() || !it->getPrevious()->getTripInfo())) //станция посадки
            {
                ost << it->getTripInfo()->stationInfoFrom->name << ") ";
            }
            itPred = it;
            it = it->getPrevious();
        }
        ost << "] ";
        ost.flush();

        return ost;
    }

}
#endif

