import logging
import os

from shapely.wkb import loads
from shapely.geometry import mapping

from yql.api.v1.client import YqlClient

logging.basicConfig(filename='railway_map.log', level=logging.DEBUG)


DB_REGION_NAMES = ['cis1', 'cis2', 'eu1', 'eu3', 'eu4', 'aao']
FT_TYPE_ID = 611  # transport-railway
RW_MIF = 'rw_full.mif'
RW_MID = 'rw_full.mid'


class GeometryYtGenerator:
    """
    https://st.yandex-team.ru/RASPFRONT-779
    https://doc.yandex-team.ru/ymaps/ymapsdf/ymapsdf-ref/concepts/ft.html?lang=ru
    """
    def __init__(self, token, file_path):
        self.token = token
        self.file_path = file_path
        self.yql_client = YqlClient(db='hahn', token=self.token)

    def run_yql(self, db_name):
        query = """
        SELECT
            ft.isocode, edge.shape
        FROM
            `home/maps/core/garden/stable/ymapsdf/latest/{db}/ft` as ft
        LEFT JOIN
            `home/maps/core/garden/stable/ymapsdf/latest/{db}/ft_edge` AS ft_edge
        on ft.ft_id=ft_edge.ft_id
        LEFT JOIN
            `home/maps/core/garden/stable/ymapsdf/latest/{db}/edge` AS edge
        on ft_edge.edge_id=edge.edge_id
        where ft_type_id={ft_type_id};
        """.format(db=db_name, ft_type_id=FT_TYPE_ID)

        logging.info('query: {}'.format(query))

        request = self.yql_client.query(
            query,
            syntax_version=1
        )
        request.run()

        records = request.full_dataframe.to_dict('records')

        return records

    def write_geometries(self, geometries):
        with open(os.path.join(self.file_path, RW_MIF), mode='a', encoding='utf8') as mif_file:
            for geometry in geometries:
                coordinates = geometry['coordinates']
                mif_file.write('Pline {}\n'.format(len(coordinates)))
                for (x, y) in coordinates:
                    mif_file.write('{} {}\n'.format(x, y))
                mif_file.write('\n')

    def create_mif_file(self, db_region_names):
        for db_region_name in db_region_names:
            logging.info(db_region_name)
            logging.info('start yql')
            records = self.run_yql(db_region_name)
            logging.info('end yql')

            geometries = []
            for record in records:
                coords = record['edge.shape']
                geometry = loads(coords, 'hex')
                map_geometry = mapping(geometry)
                geometries.append(map_geometry)

            logging.info('start save')
            self.write_geometries(geometries)
            logging.info('end save')

    def create_mid_file(self):
        open(os.path.join(self.file_path, RW_MID), mode='w').close()


def main():
    logging.info('start')

    token = os.environ.get('my_yql_token')
    file_path = os.environ.get('file_path')

    generator = GeometryYtGenerator(token=token, file_path=file_path)
    generator.create_mif_file(DB_REGION_NAMES)
    generator.create_mid_file()

    logging.info('end')


if __name__ == "__main__":
    main()
