#include "../socket/serversocket.h"
#include "../socket/socketexception.h"
#include "../webutil.h"

#include "../../pathfinderlib/dates.h"
#include "../../pathfinderlib/get.h"
#include "../../pathfinderlib/thegraph.h"

#include "../../pathfinderlib/pf.h"

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <cstdlib>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <errno.h>
#include <syslog.h>
#include <string>
#include <signal.h>
#include <memory>

using namespace std;

// храним путь до файла с логами
string LOG_FILE;
vector<string> TRUE_FLAG_VALUES = {"y", "yes", "1"};

std::time_t current_t = std::chrono::system_clock::to_time_t(std::chrono::system_clock::now());
const std::tm* Pathfinder::start_time = std::localtime(&current_t);

string getFromEnv(const string& name, const string& defaultValue) {
    char* result = getenv(name.c_str());
    if (result == nullptr) {
        return defaultValue;
    }
    return string(result);
}


bool getFlagFromEnv(const string& name) {
    char* result = getenv(name.c_str());
    if (result == nullptr) {
        return false;
    }
    for(const auto& v: TRUE_FLAG_VALUES) {
        if (result == v) {
            return true;
        }
    }
    return false;
}


void serve(ServerSocket *server, PF::TheGraph &theGraph)
{
    string request, response;

    // основной цикл
    const int begin(-1);
    string mode = getFromEnv("PATHFINDER_MODE", "");
    string timeLimitStr = getFromEnv("PATHFINDER_TIME_LIMIT", "");
    int timeLimit = stoi(timeLimitStr);

    while (true)
    {
        // accept connection
        ServerSocket newSock;
        try
        {
            server->accept(newSock);
        }
        catch (SocketException & e )
        {
             message(LOG_FILE, "SocketException: ", e.description().c_str());
             exit(EXIT_FAILURE);
        }

        message(LOG_FILE, "1 accept", "");

        // запускаем поиск
        {
            Timer timer;
            timer.start();

            // получаем запрос
            if (newSock.recv(request) == 0)
                continue;
            // логируем
            message(LOG_FILE, "request 1: ", request.c_str());
            // разбора запроса
            PF::Get get;
            get._usePrice = not theGraph.tariffsEmpty();
            get.load(request);
            theGraph.pathStorageStartWork(
                    get._mode.empty() ? mode : get._mode,
                    get._timeLimit == -1 ? timeLimit : get._timeLimit
            );
            message(LOG_FILE, "get: ", get._fromID);

            if (get._ping) {
                message(LOG_FILE, "ping!", "");

                newSock << "HTTP/1.0 200 OK\r\n";
                newSock << "Content-Type: text/plain\r\n";
                newSock << "\r\n";
                newSock << "pong!\r\n";

                continue;
            }

#ifdef DEBUG_ROUTE
        if (!get._debugStationIDs.empty() || !get._debugRoutes.empty())
        {
            ofstream out(DEBUG_LOG_FILE);
            out << "---------------------------------------------------------------" << endl;
            out << "request = " << request << endl;
            theGraph.setDebugRoute(get._debugStationIDs, get._debugRoutes);
        }
#endif
            // формируем условия поиска (из запроса)
            PF::Condition condition(get._startTime, get._boarding, get._minDelay, get._maxDelay
                    , get._transport, get._iLoveSport
                    , get._canChangeInStartTown, get._canChangeInFinishTown, get._canChangeInAnyTown, true, get._usePrice, MAX_PRICE);
            condition._fromStation = theGraph.SG.st.getStationByOrig(get._fromID);
            condition._toStation = theGraph.SG.st.getStationByOrig(get._toID);
            condition._toStationIsTown = theGraph.SG.st.isTown(condition._toStation);
            theGraph.SG.st.allStationsFrom_startOnly(condition._fromStation, condition._from);
            theGraph.SG.st.allStationsFrom_finishOnly(condition._toStation, condition._to);
            theGraph.SG.st.addTownsWith(condition._from, condition._endingTowns);
            theGraph.SG.st.addTownsWith(condition._to, condition._endingTowns);

            if (!condition._from.empty() && !condition._to.empty() && theGraph.SG.st.fromOneTown(condition._from[0], condition._to[0]))
            {
                condition._canChangeInStartTown = true;
                condition._canChangeInFinishTown = true;
                condition._canTeleport = false;
            }

            const string res = theGraph.search(LOG_FILE, condition);
            message(LOG_FILE, "result: ", res.c_str());

#ifdef DEBUG_ROUTE
        if (!get._debugStationIDs.empty() || !get._debugRoutes.empty())
        {
            ifstream debug_fin(DEBUG_LOG_FILE);
            string debug_s;
            while (!debug_fin.fail())
            {
                string s;
                getline(debug_fin, s, (char)EOF);
                debug_s += s;
            }
            debug_s = "<pre>"+debug_s+"</pre>";
            PF::Result::wrap(debug_s);
            newSock << debug_s;
        }
#endif
            // обертываем в XML
            response = theGraph.resultToXML(res);
            PF::Result::wrap(response);
            // возвращаем ответ
            newSock << response;

            // пишем ответ, размер
            message(LOG_FILE, "response: ", response.c_str());
            timer.stop();
            char buf[1000];
            sprintf(buf, "result: %s\tsize: %d\t time: %d\trequest: %s", res.c_str(), (int)theGraph.getResultNumber(), (int)timer.getTime(), request.c_str());
            message(LOG_FILE, "!itog!\t", buf);
            theGraph.pathStorageClear();
        }
    }
}

// сложная процедура остановки дочерних процессов, т.к. в некоторых ситуациях kill(0, SIGTERM);
// бывает недостаточно и процесс переподымания пересадочника ломается, не хватает памяти под два пересадочника
// поэтому процесс останавливается путем пятикратного рассылания SIGTERM,
// если не помогло - пятикратного рассылания SIGKILL
// waitpid(-1, NULL, WNOHANG); - неблокирующая процедура, которая ловит сигналы от детей
// если детей не осталось, возвращает -1
int kill_and_wait(int sig, int attempts) {
    kill(0, sig);
    sleep(1);

    int i = 0;
    for (pid_t pid = waitpid(-1, NULL, WNOHANG); i <= attempts && pid >= 0; i++, pid = waitpid(-1, NULL, WNOHANG))
    {
        message(LOG_FILE, "waitpid: ", pid);
        if (pid == 0)
        {
            kill(0, sig);
            sleep(1);
        }
    }
    if (i <= attempts)
        return 1;
    return 0;
}

void kill_child_processes()
{
    if (kill_and_wait(SIGTERM, 10))
        message(LOG_FILE, "shutdowned by term", "");
    else if (kill_and_wait(SIGKILL, 10))
        message(LOG_FILE, "shutdowned by kill", "");
    else
        message(LOG_FILE, "ERROR! Not shutdowned", "");
}

void term(int i)
{
    message(LOG_FILE, "shutdown ", i);

    kill_child_processes();

    exit(EXIT_SUCCESS);
}

int spawn(ServerSocket *server, PF::TheGraph &theGraph, const char* pidFile) {
    // делаем fork
    pid_t pid = fork();

    if (pid < 0)
    {
         // если неудача, то выходим
        message(LOG_FILE, "error: spawn fork pid - ", pid);
        message(LOG_FILE, "errno: ", errno);
        kill_child_processes();
        remove(pidFile);
        message(LOG_FILE, "child processes killed and pid-file removed", "");
        exit(EXIT_FAILURE);
    } else if (pid > 0) {
        message(LOG_FILE, "process started: ", pid);
        return pid;
    }

    struct sigaction sa;

    sa.sa_handler = SIG_DFL;

    sigaction(SIGTERM, &sa, NULL);

    serve(server, theGraph);

    message(LOG_FILE, "serve should not ever exit", "");
    exit(EXIT_FAILURE);
}

int main(int argc, char **argv)
{
    // если конфигурационный файл не получен, то выходим
    if (argc < 6)
    {
        fprintf(stderr, "error: unexpected input\n");
        exit(EXIT_FAILURE);
    }

    const char* socketChar = argv[1];
    const char* dataDir = argv[2];
    const char* logFile = argv[3];
    const char* pidFile = argv[4];
    const char* procNum = argv[5];
    bool useAsDaemon = true;
    bool printLogsToStderr = false;
    bool pricesOn = getFlagFromEnv("PATHFINDER_ENABLE_PRICES");
    if (argc >= 7) {
        useAsDaemon = atoi(argv[6]);
    }
    if (argc >= 8) {
        printLogsToStderr = atoi(argv[7]);
    }

    if (printLogsToStderr) {
        LOG_FILE = "";
    } else {
        LOG_FILE = logFile;
    }
    message(LOG_FILE, "socket: ", socketChar);
    message(LOG_FILE, "dataDir: ", dataDir);
    message(LOG_FILE, "log: ", logFile);
    message(LOG_FILE, "pid: ", pidFile);
    message(LOG_FILE, "procNum: ", procNum);
    message(LOG_FILE, "useAsDaemon: ", useAsDaemon);
    message(LOG_FILE, "printLogsToStderr: ", printLogsToStderr);

    const int socketInt = atoi(socketChar);
    string prefix = dataDir;
    prefix += '/';
    const int procNumber = atoi(procNum);

    const string thegraphTail("/thegraph");
    const string townTail("/townOrig-name.map");
    const string stationTail("/station-table.map");
    // строим пути
    const string tableFile(prefix + thegraphTail);
    const string townFile(prefix + townTail);
    const string stationFile(prefix + stationTail);

    // загружаем данные, результат пишем в graphRes
    // сюда загрузим данные, там же будет работать поиск
    PF::TheGraph theGraph;
    const string timeZoneFile = prefix + "/time-zone.map";
    const size_t tzRes = theGraph.SG.st.timeZoneStorage.Load(timeZoneFile.c_str());

    const int towns = theGraph.SG.st.loadTowns(townFile.c_str());
    message(LOG_FILE, "towns loaded: ", towns);
    const int stations = theGraph.SG.st.loadStations(stationFile.c_str());
    message(LOG_FILE, "stations loaded: ", stations);

    const string stationGeoFile = prefix + "/stationOrig-geo.map";
    const int geoRes = theGraph.SG.st.loadGeo(stationGeoFile.c_str());

    const int graphRes = theGraph.SG.load(LOG_FILE, tableFile.c_str(), PF::Transport::ALL);

    const string stationTeleportFile = prefix + "/station_teleport.map";
    const size_t teleportRes = theGraph.SG.st.loadTime2Center(stationTeleportFile.c_str());

    const string stationIntervalFile = prefix + "/interval-threads.map";
    const size_t intervalRes = theGraph.SG.loadIntervalThreadsInfo("log.txt", stationIntervalFile.c_str());

    theGraph.SG.createAllStructures(LOG_FILE);

    if (pricesOn)
    {
        const string aeroextariffFile(prefix + "/aeroextariff.map");
        const string threadtariffFile(prefix + "/threadtariff.map");
        ifstream dumpFin((prefix + "/../dump").c_str());
        const string currencyRatesFile(prefix + "/currency-rates.map");
        theGraph.loadTariffs("log.txt", aeroextariffFile.c_str(), threadtariffFile.c_str(), currencyRatesFile.c_str(), dumpFin);
    }

    message(LOG_FILE, "graph loaded: ", graphRes);
    if (towns <= 0 || stations <= 0 || graphRes <= 0)
    {
        message(LOG_FILE, "load error, thegraph : ", graphRes);
        exit(EXIT_FAILURE);
    }
    theGraph.pathStorageSetStationNumber();

    if (procNumber < 0)
        return 0;

    // создаем ServerSocket
    message(LOG_FILE, "create server socket ...", "\n");

    ServerSocket *server;

    try
    {
        server = new ServerSocket(socketInt, LOG_FILE);
    }
    catch (SocketException & e )
    {
        message(LOG_FILE, "SocketException: ", e.description().c_str());
        exit(EXIT_FAILURE);
    }

    message(LOG_FILE, "ok\n#########################\nready", "\n");

    pid_t pid;
    if (useAsDaemon) {
        // пишем pID в pid-файл
        pid = fork();
        if (pid < 0) // неудача
        {
            message(LOG_FILE, "error: fork pid - ", pid);
            remove(pidFile);
            message(LOG_FILE, "pid-file removed", "");
            exit(EXIT_FAILURE);
        }
        else if (pid > 0) // пид записываем, родителя грохаем
        {
            ofstream fpid(pidFile);
            fpid << pid << endl;
            fpid.close();
            exit(EXIT_SUCCESS);
        }

        // создан новый процесс
        // создаем новый sID для дочернего процесса
        const pid_t sid = setsid();
        if (sid < 0)
        {
            // если неудача, то выходим
            message(LOG_FILE, "error: setsid - ", sid);
            message(LOG_FILE, "pid-file removed", "");
            remove(pidFile);
            exit(EXIT_FAILURE);
        }
    }

    // закрываем стандартные файловые дескрипторы
    close(STDIN_FILENO);
    close(STDOUT_FILENO);
    close(STDERR_FILENO);

    struct sigaction sa;

    sa.sa_handler = term;

    sigaction(SIGTERM, &sa, NULL);

    // форкаем процессы-обработчики
    for (int pn = 0; pn < procNumber; pn++)
    {
        spawn(server, theGraph, pidFile);
    }

    int status;

    while ((pid = wait(&status))) {
        if(pid == -1) {
            message(LOG_FILE, "wait failure: ", errno);
            continue;
        }

        spawn(server, theGraph, pidFile);
    }

    // выход
    message(LOG_FILE, "exit failure", "");
    exit(EXIT_FAILURE);
}

