import logging.config

from aiohttp import web
from opentracing_async_instrumentation.client_hooks import aiohttpserver

from travel.library.python.aioapp.middlewares import context_middleware, error_middleware_with_log
from travel.library.python.rasp_vault.api import get_secret
from travel.library.python.tvm_ticket_provider import provider_fabric
from travel.rasp.library.python.ticket_daemon_client.async_client import TicketDaemonClient
from travel.rasp.library.python.train_api_client.async_client import TrainApiClient
from travel.rasp.pathfinder_maps.cache import AioRedisCache
from travel.rasp.pathfinder_maps.clients.geobase_client import LocalGeobaseClient
from travel.rasp.pathfinder_maps.clients.maps_client import MapsClient, MapsClientStressMock
from travel.rasp.pathfinder_maps.clients.morda_backend_client import MordaBackendClient
from travel.rasp.pathfinder_maps.services.data_provider import PickleDataProvider, ProtobufDataProvider
from travel.rasp.pathfinder_maps.services.geobase_service import GeobaseService
from travel.rasp.pathfinder_maps.services.joiners.maps_joiner import MapsJoiner
from travel.rasp.pathfinder_maps.services.joiners.pathfinder_joiner import PathfinderJoiner
from travel.rasp.pathfinder_maps.services.morda_backend_service import MordaBackendService
from travel.rasp.pathfinder_maps.services.nearest_point_finder import NearestPointFinder
from travel.rasp.pathfinder_maps.services.path_joiner import PathJoiner
from travel.rasp.pathfinder_maps.services.polling_services.polling_cache import PollingCache
from travel.rasp.pathfinder_maps.services.polling_services.polling_service import PollingService
from travel.rasp.pathfinder_maps.services.polling_services.ticket_daemon_service import TicketDaemonService
from travel.rasp.pathfinder_maps.services.polling_services.train_api_service import TrainApiService
from travel.rasp.pathfinder_maps.services.protobuf_builder import ProtobufBuilder
from travel.rasp.pathfinder_maps.services.work_db_discoverer import MordaBackendWorkDBDiscoverer
from travel.rasp.pathfinder_maps.views import Handler


log = logging.getLogger(__name__)


def create_app(settings):
    logging.config.dictConfig(settings.LOGGING_CONFIG)

    redis_password = get_secret(settings.REDIS_PASSWORD_PATH)['REDIS_PASSWORD']
    cache = AioRedisCache(settings.REDIS_HOSTS, settings.REDIS_SERVICE_NAME, redis_password)
    polling_cache = PollingCache(cache)

    train_api_client = TrainApiClient(log, settings.TRAIN_API_ENDPOINT)
    train_api_service = TrainApiService(train_api_client, settings.TRAINS_URL_PREFIX)

    ticket_daemon_client = TicketDaemonClient(log, settings.TICKET_DAEMON_ENDPOINT)
    ticket_daemon_service = TicketDaemonService(ticket_daemon_client)

    polling_service = PollingService(train_api_service, ticket_daemon_service, polling_cache)

    protobuf_data_provider = ProtobufDataProvider()
    protobuf_data_provider.load_from_dir(settings.PROTO_DIR)

    morda_backend_client = MordaBackendClient(settings.MORDA_BACKEND_URL)

    pickle_data_provider = PickleDataProvider(MordaBackendWorkDBDiscoverer(morda_backend_client))
    pickle_data_provider.load_from_dir(settings.PROTO_DIR)

    geobase_service = GeobaseService(LocalGeobaseClient(settings.GEOBASE_PATH))

    nearest_point_finder = NearestPointFinder(protobuf_data_provider, geobase_service)
    nearest_point_finder.prepare()

    morda_backend_service = MordaBackendService(morda_backend_client, protobuf_data_provider)

    tvm_provider = provider_fabric.create(
        source_id=settings.TVM_SOURCE_ID,
        destinations=[settings.TVM_MAPS_ID],
        secret=settings.TVM_TOKEN
    )

    if settings.MAPS_MASSTRANSIT_URL is None:
        maps_client = MapsClientStressMock()
    else:
        maps_client = MapsClient(settings.MAPS_MASSTRANSIT_URL, tvm_provider, settings.TVM_MAPS_ID)

    protobuf_builder = ProtobufBuilder(
        protobuf_data_provider, pickle_data_provider, maps_client, settings.RASP_SEARCH_URL
    )

    maps_joiner = MapsJoiner(maps_client, morda_backend_service, nearest_point_finder, protobuf_builder)
    pathfinder_joiner = PathfinderJoiner(maps_client, morda_backend_service, nearest_point_finder, protobuf_builder)

    path_joiner = PathJoiner(
        maps_client, polling_cache, nearest_point_finder, settings.RASP_SEARCH_URL,
        maps_joiner, pathfinder_joiner, settings.TRAVEL_TIME_COEFFICIENT
    )

    handler = Handler(path_joiner, polling_service)

    middlewares = [error_middleware_with_log(log), context_middleware, aiohttpserver.enable_tracing]
    app = web.Application(middlewares=middlewares)
    app.add_routes([
        web.get('/ping', handler=handler.ping),
        web.get('/route', handler=handler.route),
        web.get('/poll', handler=handler.poll)
    ])

    async def on_shutdown(app):
        await cache.shutdown()

    app.on_shutdown.append(on_shutdown)

    log.info('Starting pathfinder_maps on {} with port {}'.format(settings.HOSTS, settings.PORT))
    return app
