import logging
from datetime import timedelta

import aioredis

logger = logging.getLogger(__name__)


class AioRedisCache:
    SENTINEL_PORT = 26379
    EXPIRE_TIME = int(timedelta(minutes=5).total_seconds())

    def __init__(self, redis_hosts, redis_service_name, password):
        self._sentinel_hosts = [
            (h, self.SENTINEL_PORT)
            for h in redis_hosts
        ]
        self._password = password
        self._sentinel = None
        self._service = redis_service_name

    async def get_sentinel(self):
        if self._sentinel is None:
            self._sentinel = await aioredis.create_sentinel(
                self._sentinel_hosts, password=self._password,
                timeout=3, minsize=len(self._sentinel_hosts) // 2
            )
        return self._sentinel

    async def get_slave(self):
        sentinel = await self.get_sentinel()
        return sentinel.slave_for(self._service)

    async def get_master(self):
        sentinel = await self.get_sentinel()
        return sentinel.master_for(self._service)

    async def get_from_cache(self, polling_key: str):
        slave = await self.get_slave()
        return await slave.get(f'/pathfinder-maps/{polling_key}')

    async def set_cache(self, polling_key: str, data: str):
        master = await self.get_master()
        return await master.setex(f'/pathfinder-maps/{polling_key}', self.EXPIRE_TIME, data)

    async def shutdown(self):
        if self._sentinel is not None:
            self._sentinel.close()
            await self._sentinel.wait_closed()
