import json
import logging
import requests

from aiohttp import client

from travel.rasp.pathfinder_maps.models.variant import json_to_variant


log = logging.getLogger(__name__)


def point_to_arg(point):
    _type, _id = point
    if _type == 'settlement':
        return 'c{}'.format(_id)
    if _type == 'station':
        return 's{}'.format(_id)


class MordaBackendClient:
    _DATE_FORMAT = '%Y-%m-%d'

    def __init__(self, endpoint):
        self._endpoint = endpoint

    def get_maintenance_conf(self):
        url = '{}/maintenance/'.format(self._endpoint)

        try:
            response = requests.get(url)
            log.debug('Request: {}, response code: {}'.format(response.request.url, response.status_code))
            response.raise_for_status()
        except requests.RequestException as ex:
            log.exception('morda_bakend maintenance conf request failed: {}'.format(repr(ex)))
            return None

        return json.loads(response.content)

    async def get_pm_variants(self, from_type, from_id, to_type, to_id, departure_dt, ttypes):
        url = '{}/ru/search/pathfinder-maps-variants/'.format(self._endpoint)
        data = await self._get_variants(url, from_type, from_id, to_type, to_id, departure_dt, ttypes)
        if data is None:
            return []
        return list(map(json_to_variant, data['variants']))

    async def _get_variants(self, url, from_type, from_id, to_type, to_id, departure_dt, ttypes):
        params = [
            ('pointFrom', point_to_arg((from_type, from_id))),
            ('pointTo', point_to_arg((to_type, to_id))),
            ('when', departure_dt.strftime(self._DATE_FORMAT))
        ]
        for ttype in ttypes:
            params.append(('transportType', ttype))

        async with client.ClientSession() as session:
            async with session.get(url, params=params) as response:
                try:
                    log.debug('morda-backend request: %s', response.real_url)
                    response.raise_for_status()
                    result = await response.json()
                except Exception:
                    log.exception('Error for url: %s', response.real_url)
                    return None
                return result
