import contextlib
import logging
from datetime import datetime
from enum import Enum

from travel.library.python.aioapp.utils import request_var
from travel.rasp.pathfinder_maps.const import TTYPE

base_logger = logging.getLogger(__name__)
thread_warnings_logger = logging.getLogger('yt.rasp-pathfinder-maps-thread-warnings')
request_logger = logging.getLogger('yt.rasp-pathfinder-maps-request-logger')


class LogWarningType(Enum):
    MISSING_GEOMETRY = 2
    MISSING_STATION = 3
    MISSING_SETTLEMENT = 4


def get_request_id():
    try:
        context = request_var.get()
        return context.get('request_id')
    except Exception:
        return None


def _log_warning(thread_id, data, log_type):
    try:
        base_logger.warning({
            'thread_id': thread_id,
            'warning_type': log_type.value,
            'data': data
        })
        thread_warnings_logger.info({
            'thread_id': thread_id,
            'warning_type': log_type.value,
            'data': data,
            'unixtime': int(datetime.now().timestamp()),
            'request_id': get_request_id(),
        })
    except Exception as ex:
        base_logger.exception('Logbroker writing error: {}'.format(repr(ex)))


def log_missing_station(thread_id, data):
    _log_warning(thread_id, data, LogWarningType.MISSING_STATION)


def log_missing_settlement(thread_id, data):
    _log_warning(thread_id, data, LogWarningType.MISSING_SETTLEMENT)


def log_missing_geometry(thread_id, data):
    _log_warning(thread_id, data, LogWarningType.MISSING_GEOMETRY)


class RequestContext:
    def __init__(self, rll, dtm):
        self.request_id = get_request_id() or ''
        self.point_from = rll[0].as_string
        self.point_to = rll[-1].as_string
        self.dtm = dtm.isoformat()
        self.is_maps_answer = False
        self.algo_variant = None
        self.pathfinder_query = {}
        self.variants = []
        self.fail_reasons = []
        self.is_failed = False

    def set_maps_answer(self):
        base_logger.warning('Small distance. Using maps answer.')
        self.is_maps_answer = True

    def maps_first(self):
        self.algo_variant = 'maps'
        base_logger.debug('Maps first')

    def pathfinder_first(self):
        self.algo_variant = 'pathfinder'
        base_logger.debug('Pathfinder first')

    def set_pathfinder_settlements(self, settlements):
        self.pathfinder_query['settlement_from'] = settlements[0][1]
        self.pathfinder_query['settlement_to'] = settlements[1][1]

    def set_pathfinder_stations(self, stations_from, stations_to):
        for ttype in [TTYPE.aero, TTYPE.train, TTYPE.bus]:
            self.pathfinder_query[ttype.name] = {
                '{}_from'.format(stations_from[ttype][0][0]): stations_from[ttype][0][1],
                '{}_to'.format(stations_to[ttype][0][0]): stations_to[ttype][0][1]
            }

    def add_variant(self, variant):
        variant_types = []
        for route in variant.routes[::2]:
            variant_types.append(TTYPE(route.thread_info[2]).name)
        self.variants.append(variant_types)

    def set_fail_reason(self, reason):
        self.fail_reasons.append(reason)
        base_logger.warning(reason)

    def set_failed(self):
        self.is_failed = True
        base_logger.warning('No builded variants')

    def write_log(self):
        request_logger.info({
            'request_id': self.request_id,
            'point_from': self.point_from,
            'point_to': self.point_to,
            'dtm': self.dtm,
            'is_maps_answer': self.is_maps_answer,
            'algo': self.algo_variant,
            'pathfinder_query': self.pathfinder_query,
            'variants': self.variants,
            'fail_reasons': self.fail_reasons if self.is_failed else [],
            'unixtime': int(datetime.now().timestamp())
        })


@contextlib.contextmanager
def get_request_context(request):
    request_context = RequestContext(request['rll'], request['dtm'])
    yield request_context
    request_context.write_log()
