from datetime import datetime

from dataclasses import dataclass

from travel.rasp.pathfinder_maps.const import TTYPE, UTC_TZ

DATE_FORMAT = '%Y-%m-%dT%H:%M:%S'


@dataclass
class Route:
    thread_id: str
    departure_datetime: datetime
    departure_station_id: int
    arrival_datetime: datetime
    arrival_station_id: int
    departure_station = {}
    arrival_station = {}
    thread_info = None
    _thread_stations = None

    def __hash__(self):
        return hash(self.thread_id) ^ (hash(self.departure_datetime) << 10) ^ (hash(self.arrival_datetime) << 20)

    def __str__(self):
        return self.thread_id

    @property
    def polling_key(self):
        ddt = int(self.departure_datetime.timestamp())
        adt = int(self.arrival_datetime.timestamp())
        return f'{self.thread_id}-{ddt}-{adt}-{self.departure_station_id}-{self.arrival_station_id}'.replace(' ', '_')

    @property
    def is_maps_teleportation(self):
        return self.thread_id == 'NULL' and self.departure_station_id != self.arrival_station_id

    @property
    def thread_uid(self):
        return self.thread_info[0]

    @property
    def thread_name(self):
        return self.thread_info[1]

    @property
    def thread_type(self):
        return TTYPE(self.thread_info[2])

    @property
    def thread_stations(self):
        if self._thread_stations is not None:
            return self._thread_stations
        self._thread_stations = self.thread_info[3]
        if not self._thread_stations:
            self._thread_stations = [self.departure_station['id'], self.arrival_station['id']]
        return self._thread_stations

    @property
    def thread_title(self):
        return self.thread_info[4]

    def to_dict(self):
        return {
            'thread_id': self.thread_id,
            'departure_datetime': self.departure_datetime.strftime(DATE_FORMAT),
            'departure_station_id': self.departure_station_id,
            'arrival_datetime': self.arrival_datetime.strftime(DATE_FORMAT),
            'arrival_station_id': self.arrival_station_id,
            'thread_info': self.thread_info[:]
        }

    @staticmethod
    def from_dict(data):
        route = Route(
            data['thread_id'],
            _parse_route_dt(data['departure_datetime']),
            data['departure_station_id'],
            _parse_route_dt(data['arrival_datetime']),
            data['arrival_station_id']
        )
        route.thread_info = data['thread_info']
        return route


def _parse_route_dt(dt: datetime, date_format: str = DATE_FORMAT) -> datetime:
    return UTC_TZ.localize(datetime.strptime(dt, date_format))


def json_to_route(data) -> Route:
    route = Route(
        data['thread']['uid'] or data['thread']['number'],
        _parse_route_dt(data['departure'][:-6]),
        int(data['departure_station']['id']),
        _parse_route_dt(data['arrival'][:-6]),
        int(data['arrival_station']['id'])
    )
    route.thread_info = [
        data['thread']['uid'],
        data['thread']['number'],
        data['transport']['id'],
        data['thread'].get('full_path', []),
        data['thread']['title']
    ]
    return route
