import travel.rasp.pathfinder_maps.maps_protos.common2.geo_object_pb2 as geo_object_pb2
import travel.rasp.pathfinder_maps.maps_protos.common2.response_pb2 as response_pb2
import travel.rasp.pathfinder_maps.maps_protos.masstransit.route_pb2 as route_pb2
import travel.rasp.pathfinder_maps.maps_protos.masstransit.section_pb2 as section_pb2

from travel.rasp.pathfinder_maps.const import UTC_TZ
from travel.rasp.pathfinder_maps.models.route import Route
from travel.rasp.pathfinder_maps.protos.utils import build_rasp_section, form_polyline, join_bounds
from travel.rasp.pathfinder_maps.utils import seconds_to_duration_text


def build_bounding_box_protobuf(points, bounding_box):
    lons, lats = zip(*[(point.lon, point.lat) for point in points])
    bounding_box.lower_corner.lon = min(lons)
    bounding_box.lower_corner.lat = min(lats)
    bounding_box.upper_corner.lon = max(lons)
    bounding_box.upper_corner.lat = max(lats)


def build_estimation_time(meta, datetime, tz):
    utc_dt = datetime.astimezone(UTC_TZ)
    tz_dt = datetime.astimezone(tz)
    meta.value = int(utc_dt.timestamp())
    meta.tz_offset = int(tz_dt.utcoffset().total_seconds())
    meta.text = tz_dt.strftime('%H:%M')


def build_maps_stress_result(points, dtm, atm):
    response = response_pb2.Response()

    railway = build_railway_polyline(points)

    sections = [
        build_stop_protobuf(points[0], 1, '1'),
        railway,
        build_stop_protobuf(points[1], 2, '2')
    ]

    build_bounding_box_protobuf(points, response.reply.bounded_by)

    route = build_route_protobuf([
        build_pathfinder_section('test', sections, atm, dtm, UTC_TZ, UTC_TZ, 1, '1', 'train', 2)],
        points
    )
    route_meta = route.metadata[0].Extensions[route_pb2.ROUTE_METADATA]
    route_meta.estimation.departure_time.value = int(dtm.timestamp())
    route_meta.estimation.departure_time.tz_offset = 0
    route_meta.estimation.departure_time.text = ''
    route_meta.estimation.arrival_time.value = int(atm.timestamp())
    route_meta.estimation.arrival_time.tz_offset = 0
    route_meta.estimation.arrival_time.text = ''
    response.reply.geo_object.extend([route])

    return response


def build_pathfinder_protobuf(points, sections):
    response = response_pb2.Response()
    build_bounding_box_protobuf(points, response.reply.bounded_by)
    response.reply.geo_object.extend([build_route_protobuf(sections, points)])
    return response


def build_pathfinder_section(
        rasp_link: str, route: Route, geo_objects, thread_id, thread_name, thread_type_name, thread_stations_len
):
    arrival_dt, departure_dt = route.arrival_datetime, route.departure_datetime
    arrival_tz, departure_tz = route.arrival_station['timezone'], route.departure_station['timezone']
    delta = (arrival_dt - departure_dt).total_seconds()

    section = geo_object_pb2.GeoObject()
    section.geo_object.extend(geo_objects)
    join_bounds([x.bounded_by for x in geo_objects[1::2]], section.bounded_by)

    section_meta = section.metadata.add().Extensions[section_pb2.SECTION_METADATA]
    section_meta.weight.time.value = delta
    section_meta.weight.time.text = seconds_to_duration_text(delta)
    section_meta.weight.walking_distance.value = 0
    section_meta.weight.walking_distance.text = '0 м'
    section_meta.weight.transfers_count = thread_stations_len - 1

    build_estimation_time(section_meta.estimation.departure_time, departure_dt, departure_tz)
    build_estimation_time(section_meta.estimation.arrival_time, arrival_dt, arrival_tz)

    transport = section_meta.transport.add()
    transport.line.id = str(thread_id)
    transport.line.name = thread_name
    transport.line.vehicle_type.append(thread_type_name)

    transport_thread = transport.thread.add()
    transport_thread.thread.id = str(thread_id)

    build_rasp_section(section, route, rasp_link)

    return section


def build_railway_polyline(points):
    section = geo_object_pb2.GeoObject()
    build_bounding_box_protobuf(points, section.bounded_by)

    lons_first, lons_deltas, lats_first, lats_deltas = form_polyline(points)
    polyline = section.geometry.add().polyline
    polyline.lons.first = lons_first
    polyline.lons.deltas[:] = lons_deltas
    polyline.lats.first = lats_first
    polyline.lats.deltas[:] = lats_deltas

    return section


def build_railway_stop(station):
    stop = geo_object_pb2.GeoObject()

    geometry = stop.geometry.add()
    geometry.point.lon = station.Longitude
    geometry.point.lat = station.Latitude

    metadata = stop.metadata.add().Extensions[route_pb2.STOP_METADATA]
    metadata.stop.id = 'station__{}'.format(station.Id)
    metadata.stop.name = station.TitleDefault

    return stop


def build_route_protobuf(sections, points):
    route = geo_object_pb2.GeoObject()

    weights = (
        section.metadata[0].Extensions[section_pb2.SECTION_METADATA].weight
        for section in sections if section.metadata
    )
    time_delta, walking_delta = map(sum, zip(*[
        (weight.time.value, weight.walking_distance.value) for weight in weights
    ]))

    route_metadata = route.metadata.add().Extensions[route_pb2.ROUTE_METADATA]
    route_metadata.weight.time.value = time_delta
    route_metadata.weight.time.text = seconds_to_duration_text(time_delta)
    route_metadata.weight.walking_distance.value = walking_delta
    route_metadata.weight.walking_distance.text = seconds_to_duration_text(walking_delta)
    route_metadata.weight.transfers_count = len(points) - 1

    build_bounding_box_protobuf(points, route.bounded_by)

    route.geo_object.extend(sections)

    return route


def build_stop_protobuf(point, stop_id, name):
    stop = geo_object_pb2.GeoObject()

    geometry = stop.geometry.add()
    geometry.point.lon = point.lon
    geometry.point.lat = point.lat

    metadata = stop.metadata.add().Extensions[route_pb2.STOP_METADATA]
    metadata.stop.id = str(stop_id)
    metadata.stop.name = name

    return stop


def build_wait_section(delta):
    section = geo_object_pb2.GeoObject()

    section_meta = section.metadata.add().Extensions[section_pb2.SECTION_METADATA]
    section_meta.weight.time.value = delta
    section_meta.weight.time.text = seconds_to_duration_text(delta)
    section_meta.weight.walking_distance.value = 0
    section_meta.weight.walking_distance.text = '0 м'
    section_meta.weight.transfers_count = 0
    section_meta.wait.CopyFrom(section_pb2.Wait())

    return section
