import logging
from datetime import timedelta

import pytz

from travel.rasp.pathfinder_maps.const import MSK_TZ, POST_WAIT_TIMES, TTYPE, UTC_TZ
from travel.rasp.pathfinder_maps.logs import log_missing_station
from travel.rasp.pathfinder_maps.models.variant import build_teleportation_route
from travel.rasp.pathfinder_maps.serialization.proto_to_json import settlement_proto_to_json, station_proto_to_json

log = logging.getLogger(__name__)


class MordaBackendService:
    def __init__(self, morda_backend_client, protobuf_data_provider):
        self._morda_backend_client = morda_backend_client
        self._protobuf_data_provider = protobuf_data_provider

    async def search(self, query, ttypes=tuple(TTYPE)):
        ttypes = [ttype.rasp_name for ttype in ttypes]

        dtm = query['date']
        if dtm.tzname() is None:
            dtm = UTC_TZ.localize(query['date'])
        dtm = dtm.astimezone(MSK_TZ)

        variants = await self._morda_backend_client.get_pm_variants(
            query['from_type'], query['from_id'],
            query['to_type'], query['to_id'],
            dtm, ttypes
        )
        result = self._prepare_variants(variants)

        variants = await self._morda_backend_client.get_pm_variants(
            query['from_type'], query['from_id'],
            query['to_type'], query['to_id'],
            dtm.date() + timedelta(days=1), ttypes
        )
        result += self._prepare_variants(variants)
        return result

    def _prepare_variants(self, variants):
        result = []
        for variant in set(variants):
            variant = self._prepare_variant(variant)
            if variant is not None:
                result.append(variant)
        return result

    def _prepare_variant(self, variant):
        routes = self._add_teleportations(variant.routes)
        routes = self._build_routes_for_variant(routes)
        if routes is None:
            return None
        variant.routes = routes
        return variant

    def _add_teleportations(self, routes):
        result = []
        for i, route in enumerate(routes):
            result.append(route)
            if i != len(routes)-1:
                route = build_teleportation_route(routes[i], routes[i+1])
                result.append(route)
        return result

    def _build_routes_for_variant(self, routes):
        result = []
        for i, route in enumerate(routes):
            departure_station = self._get_station_json_by_id(route.departure_station_id, route.thread_id)
            arrival_station = self._get_station_json_by_id(route.arrival_station_id, route.thread_id)
            if departure_station is None or arrival_station is None:
                return None

            route.departure_station = departure_station
            route.arrival_station = arrival_station

            # временное решение, пока карты не обрабатывают время прилёта как сегмент
            if route.thread_id != 'NULL':
                route.arrival_datetime += POST_WAIT_TIMES[TTYPE(route.thread_info[2])]

            result.append(route)

        return result if result else None

    def _get_station_json_by_id(self, station_id, thread_id):
        station_proto = self._protobuf_data_provider.station_repo.get(station_id)
        if station_proto is None:
            log_missing_station(thread_id, {'station_id': station_id})
            return None
        station = station_proto_to_json(station_proto)

        timezone_proto = self._protobuf_data_provider.timezone_repo.get(station['timezone_id'])
        station['timezone'] = pytz.timezone(timezone_proto.Code)

        settlement_proto = self._protobuf_data_provider.settlement_repo.get(station['settlement_id'])
        if settlement_proto and not settlement_proto.IsHidden:
            station['settlement'] = settlement_proto_to_json(settlement_proto)

        del station['settlement_id']
        del station['timezone_id']
        return station
