from scipy.spatial.distance import cdist

from travel.proto.dicts.rasp.settlement_pb2 import TSettlement
from travel.rasp.pathfinder_maps.const import TTYPE
from travel.rasp.pathfinder_maps.utils import RoutePoint


TRAIN_IN_CITY_ID = 111


def is_settlement_city(settlement: TSettlement) -> bool:
    return settlement.Majority != 0 and settlement.Majority <= 4


class NearestPointFinder:
    def __init__(self, protobuf_data_provider, geobase_service):
        self._settlement_repo = protobuf_data_provider.settlement_repo
        self._station_repo = protobuf_data_provider.station_repo
        self._nearest_settlement_repo = protobuf_data_provider.nearest_settlement_repo
        self._geobase_service = geobase_service

        self._settlement_coords, self._settlement_points = [], []
        self._db = {}
        self._geoid_to_settlement = {}

    def prepare(self):
        for x in self._settlement_repo.itervalues():
            if x.GeoId:
                self._geoid_to_settlement[x.GeoId] = x.Id

            if x.Longitude == x.Latitude == 0:
                continue
            self._settlement_coords.append((x.Longitude, x.Latitude))
            self._settlement_points.append(('settlement', x.Id))

        ttype_dicts = {t: [[], []] for t in [TTYPE.train.value, TTYPE.aero.value, TTYPE.bus.value, TRAIN_IN_CITY_ID]}
        for x in self._station_repo.itervalues():
            if x.Longitude == x.Latitude == 0:
                continue

            if x.IsHidden:
                continue

            if x.TransportType == TTYPE.train.value and x.SettlementId is not None:
                settlement = self._settlement_repo.get(x.SettlementId)
                if settlement is not None and is_settlement_city(settlement) and 'train' in x.TypeChoices:
                    d = ttype_dicts[TRAIN_IN_CITY_ID]
                    d[0].append((x.Longitude, x.Latitude))
                    d[1].append(('station', x.Id))

            d = ttype_dicts.get(x.TransportType)
            if d is not None:
                d[0].append((x.Longitude, x.Latitude))
                d[1].append(('station', x.Id))

        self._db = ttype_dicts

    def _get_rasp_settlement(self, point):
        settlement_id = None
        for settlement_geoid in self._geobase_service.iterate_settlements_by_location(point):
            settlement_id = self._geoid_to_settlement.get(settlement_geoid, None)
            if not self._settlement_repo.get(settlement_id):
                continue
        return settlement_id

    async def get_settlement_from_geobase(self, point):
        settlement_id = self._get_rasp_settlement(point)
        if settlement_id is None:
            return None

        nearest_settlement_id = self._nearest_settlement_repo.get(settlement_id)
        return 'settlement', settlement_id if nearest_settlement_id is None else nearest_settlement_id.SettlementTo

    async def get_settlement(self, point):
        settlement = await self.get_settlement_from_geobase(point)
        if settlement:
            return settlement

        return self._settlement_points[cdist(
            [(point.lon, point.lat)],
            self._settlement_coords,
            self._geobase_service.calculate_distance
        ).argmin()]

    async def get_stations_by_ttype(self, point):
        lon, lat = point.lon, point.lat

        maps_coords, pathfinder_stations = {}, {}

        for _type in [TTYPE.train, TTYPE.bus, TTYPE.aero]:
            coords, stations = self._db[_type.value]

            coords_min_index = cdist(
                [(lon, lat)],
                coords,
                self._geobase_service.calculate_distance
            ).argmin()

            coordinates = coords[coords_min_index]
            station = stations[coords_min_index]

            maps_coords[_type] = [RoutePoint(*coordinates)]
            pathfinder_stations[_type] = [station]

        closest_train_station = pathfinder_stations[TTYPE.train][0]

        train_in_city_coords, train_in_city_stations = self._db[TRAIN_IN_CITY_ID]
        coords_min_index = cdist(
            [(lon, lat)],
            train_in_city_coords,
            self._geobase_service.calculate_distance
        ).argmin()
        closest_train_in_city_coordinates = train_in_city_coords[coords_min_index]
        closest_train_in_city_station = train_in_city_stations[coords_min_index]

        if closest_train_in_city_station != closest_train_station:
            maps_coords[TTYPE.train].append(RoutePoint(*closest_train_in_city_coordinates))
            pathfinder_stations[TTYPE.train].append(closest_train_in_city_station)

        return maps_coords, pathfinder_stations
