import asyncio

import travel.rasp.pathfinder_maps.maps_protos.pm_protos.polling_pb2 as polling_pb2

from travel.rasp.pathfinder_maps.const import TTYPE
from travel.rasp.pathfinder_maps.services.polling_services.polling_cache import PollingCache
from travel.rasp.pathfinder_maps.services.polling_services.ticket_daemon_service import TicketDaemonService
from travel.rasp.pathfinder_maps.services.polling_services.train_api_service import TrainApiService


class PollingService:
    def __init__(
        self,
        train_api_service: TrainApiService,
        ticket_daemon_service: TicketDaemonService,
        cache: PollingCache
    ):
        self._train_api_service = train_api_service
        self._ticket_daemon_service = ticket_daemon_service
        self._cache = cache
        self.services = {
            TTYPE.train: self._train_api_service,
            TTYPE.aero: self._ticket_daemon_service
        }

    async def poll(self, polling_keys):
        response = polling_pb2.RaspPollingResponse()
        routes = await asyncio.gather(*[self._collect(key) for key in polling_keys])
        response.rasp_section.extend([x for x in routes if x is not None])
        return response

    async def _collect(self, key):
        result = await self._cache.get_from_cache(key)
        if not result:
            return None
        route = result['route']

        if result['is_done']:
            return self._build_polling_section(route, True, result['price'], result['travel_link'])

        route_ttype = TTYPE(route.thread_info[2])
        service = self.services.get(route_ttype)
        if service is None:
            await self._dump_state(route, False, True)
            return self._build_polling_section(route, True)

        polling_answer = await service.get_price(result['route'], result['is_polling'])
        if polling_answer is None:
            await self._dump_state(route, False, True)
            return self._build_polling_section(route, True)

        is_done = polling_answer.price is not None or not polling_answer.is_polling
        await self._dump_state(
            route, polling_answer.is_polling, is_done, polling_answer.price, polling_answer.order_url
        )
        return self._build_polling_section(route, is_done, polling_answer.price, polling_answer.order_url)

    async def _dump_state(self, route, is_polling, is_done, price=None, order_url=None):
        data = {
            'route': route.to_dict(),
            'price': price,
            'travel_link': order_url,
            'is_polling': is_polling,
            'is_done': is_done
        }
        await self._cache.set_cache(route.polling_key, data)

    def _build_polling_section(self, route, is_done, price=None, order_url=None):
        polling_section = polling_pb2.RaspPollingSection()
        polling_section.polling_key = route.polling_key
        polling_section.is_polling_done = is_done
        if price is not None:
            polling_section.price = str(price)
        if order_url is not None:
            polling_section.travel_link = order_url
        return polling_section
