import logging
import urllib.parse as urlparse
from collections import defaultdict
from typing import Optional

from travel.library.python.aioapp.utils import localize_dt
from travel.rasp.library.python.ticket_daemon_client.async_client import TicketDaemonClient, TicketDaemonError
from travel.rasp.pathfinder_maps.const import UTC_TZ
from travel.rasp.pathfinder_maps.models.route import Route
from travel.rasp.pathfinder_maps.services.polling_services.polling_answer import PollingAnswer

log = logging.getLogger(__name__)


class TicketDaemonService:
    def __init__(self, client: TicketDaemonClient):
        self._client = client

    async def get_price(self, route: Route, poll: bool) -> Optional[PollingAnswer]:
        route_key = route.departure_station_id, route.arrival_station_id, route.departure_datetime
        try:
            query = await self._client.get_prices(
                f's{route.departure_station_id}', f's{route.arrival_station_id}', route.departure_datetime,
                poll=poll, asker='pathfinder'
            )
        except TicketDaemonError as ex:
            log.exception(repr(ex))
            return PollingAnswer(None, None, False)

        if not poll:
            return PollingAnswer(None, None, True)

        tariff = self._parse_answer(query, route_key)
        if not tariff:
            return None

        price, order_url = tariff
        querying = self._calculate_polling_status(query['data']['status'])

        return PollingAnswer(price, order_url, querying)

    def _parse_answer(self, query, route_key):
        variants = defaultdict(list)
        for variant in query['data']['variants']:
            key = variant['forward']
            avia_url_parts = urlparse.urlparse(variant['order_link'])

            all_query_params = dict(urlparse.parse_qsl(avia_url_parts.query))
            query_params = {
                p: v for (p, v) in all_query_params.items()
                if not p.startswith('utm')
            }

            avia_url_parts_list = list(avia_url_parts)
            avia_url_parts_list[4] = urlparse.urlencode(query_params)
            order_url = urlparse.urlunparse(avia_url_parts_list)

            variants[key].append((variant['tariff']['value'], order_url))

        variants = {k: min(v) for k, v in variants.items()}
        flight = self._get_flight(query['data']['reference']['flights'], route_key)
        if not flight:
            return None

        itinerary = self._get_itinerary(query['data']['reference']['itineraries'], flight['key'])
        return variants[itinerary]

    def _get_flight(self, flights, route_key):
        for flight in flights:
            station_from, station_to = flight['station_from'], flight['station_to']
            departure_dt = localize_dt(
                flight['departure']['local'],
                flight['departure']['tzname'],
                '%Y-%m-%d %H:%M:%S'
            ).astimezone(UTC_TZ)
            if route_key == (station_from, station_to, departure_dt):
                return flight

    def _get_itinerary(self, itineraries, flight_key):
        for key, variants in itineraries.items():
            if flight_key in variants:
                return key

    def _calculate_polling_status(self, polling_status):
        done_count = len([x for x in polling_status.values() if x == 'done'])
        all_count = len(polling_status.values())
        return .75 > done_count/all_count
