import logging
from typing import Optional

from travel.library.python.aioapp.utils import localize_dt
from travel.rasp.library.python.train_api_client.async_client import TrainApiClient, TrainApiError
from travel.rasp.pathfinder_maps.const import UTC_TZ
from travel.rasp.pathfinder_maps.models.route import Route
from travel.rasp.pathfinder_maps.services.polling_services.polling_answer import PollingAnswer

log = logging.getLogger(__name__)


class TrainApiService:
    def __init__(self, client: TrainApiClient, travel_prefix: str):
        self._client = client
        self._travel_prefix = travel_prefix

    async def get_price(self, route: Route, poll: bool) -> Optional[PollingAnswer]:
        route_key = route.departure_station_id, route.arrival_station_id, route.departure_datetime
        try:
            query = await self._client.get_prices(
                f's{route.departure_station_id}', f's{route.arrival_station_id}', route.departure_datetime,
                poll=poll, asker='pathfinder'
            )
        except TrainApiError as ex:
            log.exception(repr(ex))
            return PollingAnswer(None, None, False)

        segment, querying = self._parse_answer(query, route_key)
        if not segment:
            return PollingAnswer(None, None, querying)

        tariff = self._parse_tariff(segment)
        if tariff is None:
            return PollingAnswer(None, None, querying)

        return PollingAnswer(*tariff, querying)

    def _parse_answer(self, data, route_key):
        segments, querying = data['segments'], data['querying']
        for segment in segments:
            segment_departure = localize_dt(segment['departure'][:-9], UTC_TZ)
            key = segment['stationFrom']['id'], segment['stationTo']['id'], segment_departure
            if key == route_key:
                return segment, querying
        return None, querying

    def _parse_tariff(self, segment):
        tariff = segment.get('tariffs')
        if not tariff:
            return None

        prices = []
        for data in tariff.get('classes', {}).values():
            price = data['price']['value']
            order_url = '{}{}'.format(self._travel_prefix, data['trainOrderUrl'])
            prices.append((price, order_url))

        if not prices:
            return None
        return min(prices)
