import time
from aiohttp import client

import ticket_parser2
import ticket_parser2.api.v1 as tp2


class AbstractTvmClient:
    async def get_tickets(self, dest_ids):
        raise NotImplementedError()


class ApiTvmClient(AbstractTvmClient):
    API_URL = 'https://tvm-api.yandex.net/2'

    def __init__(self, source_id, secret):
        self._source_id = source_id
        self._secret = secret
        self._service_context = None

    async def get_tickets(self, dest_ids):
        ts = int(time.time())
        dst = ','.join(map(str, dest_ids))

        async with client.ClientSession() as session:
            async with session.post(
                    '{}/ticket/'.format(self.API_URL),
                    data={
                        'grant_type': 'client_credentials',
                        'src': self._source_id,
                        'dst': dst,
                        'ts': ts,
                        'sign': self._service_context.sign(ts, dst),
                    }
            ) as response:
                response.raise_for_status()
                tickets = await response.json(content_type=None)
        return tickets

    async def _get_keys(self):
        async with client.ClientSession() as session:
            async with session.get(
                    '{}/keys'.format(self.API_URL),
                    params={'lib_version': ticket_parser2.__version__}
            ) as response:
                return await response.text()

    async def refresh_keys(self):
        tvm_keys = await self._get_keys()
        if self._service_context is None:
            self._service_context = tp2.ServiceContext(self._source_id, self._secret, tvm_keys)
        else:
            self._service_context.reset_keys(tvm_keys)


class QloudTvmClient(AbstractTvmClient):
    API_URL = 'http://localhost:1/tvm'

    def __init__(self, source_id, token):
        self._source_id = source_id
        self._token = token

    async def _get(self, method, params=None, headers=None):
        if headers is None:
            headers = {}
        headers['Authorization'] = self._token

        async with client.ClientSession() as session:
            async with session.get(
                    '{}{}'.format(self.API_URL, method),
                    params=params,
                    headers=headers
            ) as response:
                response.raise_for_status()
                return await response.json()

    async def get_tickets(self, dest_ids):
        data = await self._get('/tickets', {
            'src': self._source_id,
            'dsts': ','.join(map(str, dest_ids)),
        })

        tickets = {}
        for name, item in data.items():
            tickets[name] = item['ticket']
            tickets[item['tvm_id']] = item['ticket']

        return tickets
