import logging
import logging.config

from aiohttp import web
from opentracing_async_instrumentation.client_hooks import aiohttpserver

from travel.library.python.aioapp.middlewares import error_middleware_with_log, context_middleware
from travel.rasp.library.python.morda_backend_client.async_client import MordaBackendClient
from travel.rasp.library.python.ticket_daemon_client.async_client import TicketDaemonClient
from travel.rasp.library.python.train_api_client.async_client import TrainApiClient
from travel.library.python.rasp_vault.api import get_secret
from travel.rasp.pathfinder_proxy.cache import AioRedisCache
from travel.rasp.pathfinder_proxy.services.train_fee_service import TrainFeeService
from travel.rasp.pathfinder_proxy.tvm import TvmFactory
from travel.rasp.pathfinder_proxy.views import Handler


log = logging.getLogger(__name__)


async def ping(request):
    return web.Response(body='Ok', status=200)


def create_app(settings):
    logging.config.dictConfig(settings.LOGGING_CONFIG)
    redis_password = get_secret(settings.REDIS_PASSWORD_PATH)['REDIS_PASSWORD']
    cache = AioRedisCache(settings, redis_password)
    tvm = TvmFactory(settings)
    train_fee_service = TrainFeeService(tvm, settings)
    handler = Handler(
        MordaBackendClient(log, settings.MORDA_BACKEND_ENDPOINT),
        TrainApiClient(log, settings.TRAIN_API_ENDPOINT),
        TicketDaemonClient(log, settings.TICKET_DAEMON_ENDPOINT),
        train_fee_service,
        cache,
        settings
    )
    middlewares = [context_middleware, error_middleware_with_log(log), aiohttpserver.enable_tracing]
    app = web.Application(middlewares=middlewares)
    app.add_routes([
        web.get('/service/transfers-with-prices/', handler=handler.transfers_with_prices),
        web.get('/service/transfers-with-prices/poll/', handler=handler.transfers_with_prices_poll),
        web.get('/ping', handler=ping),
    ])

    async def on_shutdown(app):
        await cache.shutdown()

    app.on_shutdown.append(on_shutdown)

    return app
