import asyncio
import logging
from abc import ABC, abstractmethod
from concurrent import futures

import aioredis
from redis.exceptions import ConnectionError, TimeoutError
from redis.sentinel import Sentinel

from travel.rasp.pathfinder_proxy.const import CacheType, TTransport

logger = logging.getLogger(__name__)


class AbstractCache(ABC):
    @abstractmethod
    async def get_from_cache(self, cache_type, point_from_key, point_to_key, departure_dt, tld, language):
        pass

    @abstractmethod
    async def set_cache(self, cache_type, point_from_key, point_to_key, departure_dt, tld, language, result):
        pass

    @staticmethod
    def _build_cache_key(cache_type, point_from_key, point_to_key, departure_dt, tld, language, transport_types=None):
        dt = departure_dt.strftime('%Y-%m-%dT%H:%M')
        key = f'/pathfinder-proxy/{cache_type.value}/{tld}/{language}/{point_from_key}.{point_to_key}.{dt}'
        if transport_types:
            transport_types = [TTransport.get_name(transport_type) for transport_type in sorted(transport_types)]
            key = '{}.{}'.format(key, '.'.join(transport_types))
        return key

    async def shutdown(self):
        pass


class ThreadBasedRedisCache(AbstractCache):
    SENTINEL_PORT = 26379

    def __init__(self, settings, password):
        self._password = password
        sentinel_hosts = [
            (h, self.SENTINEL_PORT)
            for h in settings.REDIS_HOSTS
        ]
        self._sentinel = Sentinel(
            sentinel_hosts, socket_timeout=0.1, min_other_sentinels=len(sentinel_hosts) // 2
        )
        self._service = settings.REDIS_SERVICE_NAME
        self._master = None
        self._slave = None
        self._executer = futures.ThreadPoolExecutor(max_workers=30)
        self._loop = asyncio.get_event_loop()

    @property
    def master(self):
        if self._master is None:
            self._master = self._sentinel.master_for(self._service, password=self._password, socket_timeout=0.3)
        return self._master

    @property
    def slave(self):
        if self._slave is None:
            self._slave = self._sentinel.slave_for(self._service, password=self._password)
        return self._slave

    @staticmethod
    def _expire_for(cache_type):
        return CacheType.get_expire_by_type(cache_type)

    async def get_from_cache(self, cache_type, point_from_key, point_to_key, departure_dt, tld, language):
        try:
            return await self._async_call(self.master, 'get', [
                self._build_cache_key(cache_type, point_from_key, point_to_key, departure_dt, tld, language),
            ])
        except (ConnectionError, TimeoutError):
            self._slave = None
            raise

    async def set_cache(self, cache_type, point_from_key, point_to_key, departure_dt, tld, language, result):
        try:
            return await self._async_call(self.master, 'setex', [
                self._build_cache_key(cache_type, point_from_key, point_to_key, departure_dt, tld, language),
                self._expire_for(cache_type),
                result
            ])
        except (ConnectionError, TimeoutError):
            self._master = None
            raise

    async def _async_call(self, instance, method, args):
        def call():
            try:
                result = getattr(instance, method)(*args)
                logger.debug('instance: %s, successful call %s with args %s', instance, method, args)
                return result
            except Exception:
                logger.exception('instance: %s, failed cache call %s with args %s', instance, method, args)
                raise

        future = self._loop.run_in_executor(self._executer, call)
        return await asyncio.wait_for(future, timeout=1)


class AioRedisCache(AbstractCache):
    SENTINEL_PORT = 26379

    def __init__(self, settings, password):
        self._sentinel_hosts = [
            (h, self.SENTINEL_PORT)
            for h in settings.REDIS_HOSTS
        ]
        self._password = password
        self._sentinel = None
        self._service = settings.REDIS_SERVICE_NAME

    async def get_sentinel(self):
        if self._sentinel is None:
            self._sentinel = await aioredis.create_sentinel(
                self._sentinel_hosts, password=self._password,
                timeout=3, minsize=len(self._sentinel_hosts) // 2
            )
        return self._sentinel

    async def get_slave(self):
        sentinel = await self.get_sentinel()
        return sentinel.slave_for(self._service)

    async def get_master(self):
        sentinel = await self.get_sentinel()
        return sentinel.master_for(self._service)

    @staticmethod
    def _expire_for(cache_type):
        return CacheType.get_expire_by_type(cache_type)

    async def get_from_cache(self, cache_type, point_from, point_to, when, tld, language, transport_types=None):
        key = self._build_cache_key(cache_type, point_from, point_to, when, tld, language, transport_types)
        slave = await self.get_slave()
        return await slave.get(key)

    async def set_cache(self, cache_type, point_from, point_to, when, tld, language, result, transport_types=None):
        key = self._build_cache_key(cache_type, point_from, point_to, when, tld, language, transport_types)
        master = await self.get_master()
        return await master.setex(key, self._expire_for(cache_type), result)

    async def shutdown(self):
        self._sentinel.close()
        await self._sentinel.wait_closed()
