import logging
from datetime import datetime

from travel.library.python.aioapp.utils import request_var
from travel.rasp.pathfinder_proxy.const import CacheType, TTransport

log = logging.getLogger(__name__)

pathfinder_proxy_search_logger = logging.getLogger('yt.pathfinder-proxy-search')
pathfinder_proxy_tariffs_logger = logging.getLogger('yt.pathfinder-proxy-tariffs')
pathfinder_proxy_result_logger = logging.getLogger('yt.pathfinder-proxy-result')


def get_request_id():
    try:
        context = request_var.get()
        return context.get('request_id')
    except Exception:
        return None


def get_timestamp():
    return int(datetime.now().timestamp())


def get_transport_types_codes(transport_types):
    return [TTransport.get_name(t_type) for t_type in transport_types]


def prepare_search_transfer_variants(transfer_variants):
    def get_station_dict(station):
        station_dict = {
            'id': station['id'],
            'title': station.get('title')
        }

        settlement = station.get('settlement')
        if settlement:
            settlement_dict = {
                'id': settlement['id'],
                'title': settlement.get('title')
            }
        else:
            settlement_dict = None

        country_dict = {
            'id': station.get('countryId')
        }

        station_dict['settlement'] = settlement_dict
        station_dict['country'] = country_dict

        return station_dict

    result_variants = []

    for variant in transfer_variants:
        segments = []
        for segment in variant.get('segments', []):
            segments.append({
                'arrival': segment['arrival'],
                'departure': segment['departure'],
                'thread': {
                    'uid': segment['thread']['uid'],
                    'title': segment['thread']['title'],
                    'number': segment['thread']['number']
                },
                'transport_type': segment['transport']['code'],
                'station_from': get_station_dict(segment['stationFrom']),
                'station_to': get_station_dict(segment['stationTo']),
                'tariffs': segment.get('tariffs'),
                'has_dynamic_pricing': segment.get('hasDynamicPricing')
            })

        result_variants.append({
            'segments': segments,
            'segments_count': len(segments)
        })

    return result_variants


def prepare_train_tariffs(tariffs):
    def get_thread_dict(thread):
        if thread:
            return {
                'number': thread.get('number')
            }

    result_tariffs = {
        'querying': tariffs['querying'],
        'segments': []
    }

    for segment in tariffs['segments']:
        result_tariffs['segments'].append({
            'arrival': segment['arrival'],
            'departure': segment['departure'],
            'thread': get_thread_dict(segment['thread']),
            'title': segment['title'],
            'station_from': {
                'id': segment['stationFrom']['id']
            },
            'station_to': {
                'id': segment['stationTo']['id']
            },
            'tariffs': segment['tariffs'],
            'has_dynamic_pricing': segment['hasDynamicPricing'],
            'key': segment['key']
        })

    return result_tariffs


def prepare_plane_tariffs(tariffs):
    result_tariffs = {
        'status': tariffs['status']
    }
    data = tariffs.get('data')

    if data:
        result_tariffs.update({
            'variants': data.get('variants'),
            'flights': data.get('reference', {}).get('flights'),
            'itineraries': data.get('reference', {}).get('itineraries')
        })

    return result_tariffs


def log_proxy_search(point_from, point_to, when, tld, language, transport_types, status, transfer_variants):
    try:
        data = {
            'point_from': point_from,
            'point_to': point_to,
            'when': when,
            'tld': tld,
            'language': language,
            'transport_types': get_transport_types_codes(transport_types),
            'status': status,
            'unixtime': get_timestamp(),
            'request_id': get_request_id(),
        }

        result_variants = prepare_search_transfer_variants(transfer_variants)
        data['transfer_variants'] = result_variants

        pathfinder_proxy_search_logger.info(data)
    except Exception as ex:
        log.exception('Не смогли записать log_proxy_search {}'.format(repr(ex)))


def log_proxy_tariffs(cache_type, point_from, point_to, when, tld, language, transport_type, status, tariffs, poll):
    try:
        data = {
            'cache_type': cache_type.value,
            'point_from': point_from,
            'point_to': point_to,
            'when': when,
            'tld': tld,
            'language': language,
            'transport_type': transport_type,
            'status': status,
            'unixtime': get_timestamp(),
            'request_id': get_request_id(),
            'poll': poll
        }

        result_tariffs = {}
        if tariffs:
            if cache_type == CacheType.TRAIN_API:
                result_tariffs = prepare_train_tariffs(tariffs)
            else:
                result_tariffs = prepare_plane_tariffs(tariffs)

        data['transfer_variants_tariffs'] = result_tariffs
        pathfinder_proxy_tariffs_logger.info(data)
    except Exception as ex:
        log.exception('Не смогли записать log_proxy_tariffs {}'.format(repr(ex)))


def log_proxy_result(cache_type, point_from, point_to, when, tld, language, transport_types, status, transfer_variants):
    try:
        data = {
            'cache_type': cache_type.value,
            'point_from': point_from,
            'point_to': point_to,
            'when': when,
            'tld': tld,
            'language': language,
            'transport_types': get_transport_types_codes(transport_types),
            'status': status,
            'unixtime': get_timestamp(),
            'request_id': get_request_id(),
        }

        result_variants = prepare_search_transfer_variants(transfer_variants)
        data['transfer_variants'] = result_variants

        pathfinder_proxy_result_logger.info(data)
    except Exception as ex:
        log.exception('Не смогли записать log_proxy_result {}'.format(repr(ex)))
