import json
import logging

import pytz

from travel.library.python.aioapp.utils import localize_dt
from travel.rasp.library.python.morda_backend_client.async_client import MordaBackendError
from travel.rasp.pathfinder_proxy.const import CacheType, TTransport, Status
from travel.rasp.pathfinder_proxy.logs import log_proxy_search

logger = logging.getLogger(__name__)


def _add_ids(transfer_variants):
    for variant in transfer_variants:
        segment_ids = []
        for segment in variant['segments']:
            id = '-'.join((
                str(segment['stationFrom']['id']),
                str(segment['stationTo']['id']),
                localize_dt(segment['departure'][:-9], pytz.UTC).strftime('%Y-%m-%dT%H:%M'),
                segment['transport']['code']
            ))
            segment['id'] = id
            segment_ids.append(id)
        variant['id'] = '_'.join(segment_ids)


class MordaBackendService:
    def __init__(self, morda_backend_client, cache):
        self._morda_backend_client = morda_backend_client
        self._cache = cache

    async def get_transfer_variants(self, point_from, point_to, when, tld, language, transport_types=None):
        if transport_types is None:
            transport_types = [TTransport.TRAIN]

        transfer_variants = await self._cache.get_from_cache(
            CacheType.MORDA_BACKEND,
            point_from,
            point_to,
            when,
            tld,
            language,
            transport_types
        )
        if transfer_variants:
            transfer_variants = json.loads(transfer_variants)
        else:
            try:
                transfer_variants = await self._morda_backend_client.get_transfers(
                    point_from=point_from,
                    point_to=point_to,
                    when=when,
                    transport_types=[TTransport.get_name(transport_type) for transport_type in transport_types],
                    language=language,
                )
                status = Status.DONE.value
            except MordaBackendError as ex:
                transfer_variants = []
                status = repr(ex)

            log_proxy_search(
                point_from=point_from,
                point_to=point_to,
                when=when,
                tld=tld,
                language=language,
                transport_types=transport_types,
                status=status,
                transfer_variants=transfer_variants
            )

            if transfer_variants:
                _add_ids(transfer_variants)
                await self._cache.set_cache(
                    CacheType.MORDA_BACKEND,
                    point_from,
                    point_to,
                    when,
                    tld,
                    language,
                    json.dumps(transfer_variants, ensure_ascii=False),
                    transport_types
                )
        return transfer_variants
