import json
import logging

from aiohttp import web

from travel.rasp.pathfinder_proxy.const import CacheType, Status
from travel.rasp.pathfinder_proxy.logs import log_proxy_result
from travel.rasp.pathfinder_proxy.serialization import (
    SERPTransfersQuery, TransferVariantsWithPricesQuerySchema, bandit_type_from_headers, icookie_from_headers,
    req_id_from_headers, device_from_headers, yandex_uid_from_headers
)
from travel.rasp.pathfinder_proxy.services.morda_backend_service import MordaBackendService
from travel.rasp.pathfinder_proxy.services.price_collector import PriceCollector

logger = logging.getLogger(__name__)


class Handler:
    serp_transfers_schema = SERPTransfersQuery(strict=True)

    def __init__(self, morda_backend_client, train_api_client, ticket_daemon_client, train_fee_service,
                 cache, settings):
        self._cache = cache
        self._morda_backend_service = MordaBackendService(morda_backend_client, cache)
        self._price_collector = PriceCollector(cache, train_api_client, ticket_daemon_client, settings)
        self._train_fee_service = train_fee_service

    async def transfers_with_prices(self, request):
        query = TransferVariantsWithPricesQuerySchema(strict=True).load(request.query).data
        is_bot = query.pop('is_bot')
        include_price_fee = query.pop('include_price_fee')
        icookie = icookie_from_headers(request.headers)
        bandit_type = bandit_type_from_headers(request.headers)
        req_id = req_id_from_headers(request.headers)
        device = device_from_headers(request.headers)
        yandex_uid = yandex_uid_from_headers(request.headers)

        result = await self._cache.get_from_cache(CacheType.TRANSFERS_WITH_PRICES, **query)

        if result:
            result = json.loads(result)
            if result['status'] != Status.ERROR.value:
                if include_price_fee:
                    result['transfer_variants'] = await self._train_fee_service.apply_fee(
                        result['transfer_variants'], icookie, bandit_type, yandex_uid, device, req_id
                    )
                return web.json_response(text=json.dumps(result, ensure_ascii=False))

        transfer_variants = await self._morda_backend_service.get_transfer_variants(**query)

        if transfer_variants and is_bot:
            result = json.dumps({
                'transfer_variants': transfer_variants,
                'status': Status.DONE.value
            }, ensure_ascii=False)
            return web.json_response(text=result)

        if transfer_variants:
            result = json.dumps({
                'transfer_variants': transfer_variants,
                'status': Status.QUERYING.value
            }, ensure_ascii=False)

            log_proxy_result(
                cache_type=CacheType.TRANSFERS_WITH_PRICES,
                point_from=query['point_from'],
                point_to=query['point_to'],
                when=query['when'],
                tld=query['tld'],
                language=query['language'],
                transport_types=query['transport_types'],
                status=Status.QUERYING.value,
                transfer_variants=transfer_variants
            )

            await self._cache.set_cache(CacheType.TRANSFERS_WITH_PRICES, result=result, **query)
            self._price_collector.collect(transfer_variants, query)
            if include_price_fee:
                transfer_variants = await self._train_fee_service.apply_fee(transfer_variants, icookie, bandit_type,
                                                                            yandex_uid, device, req_id)
            return web.json_response(text=json.dumps({
                'transfer_variants': transfer_variants,
                'status': Status.QUERYING.value
            }, ensure_ascii=False))
        return web.json_response(status=204)

    async def transfers_with_prices_poll(self, request):
        query = TransferVariantsWithPricesQuerySchema(strict=True).load(request.query).data
        query.pop('is_bot')
        include_price_fee = query.pop('include_price_fee')
        icookie = icookie_from_headers(request.headers)
        bandit_type = bandit_type_from_headers(request.headers)
        req_id = req_id_from_headers(request.headers)
        device = device_from_headers(request.headers)
        yandex_uid = yandex_uid_from_headers(request.headers)

        result = await self._cache.get_from_cache(CacheType.TRANSFERS_WITH_PRICES, **query)

        if result:
            result = json.loads(result)
            if include_price_fee:
                result['transfer_variants'] = await self._train_fee_service.apply_fee(
                    result['transfer_variants'], icookie, bandit_type, yandex_uid, device, req_id
                )
            return web.json_response(text=json.dumps(result, ensure_ascii=False))
        return web.json_response(status=204)
